/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring;

import java.lang.reflect.Modifier;
import java.text.MessageFormat;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.spi.SimpleRefactoringElementImpl;
import org.openide.filesystems.FileObject;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;

/**
 * Changes the modifier of element.
 * For example, it is used by EncapsulateField refactoring - 
 * field is changed to private as default, but user can
 * use another value. You can use it also for other features.
 *
 * @author  Pavel Flaska
 */
public class ChangeModElement extends SimpleRefactoringElementImpl {
    
    private int modifier;
    private Feature element;
    
    private PositionBounds bounds;
    private final String text;
    private final String displayText;
    
    /** 
     * Creates a new instance of Change field access modifier element.
     *
     * @param element   access modifier will be changed on this parameter
     * @param modifier  new modifier for the element
     */
    public ChangeModElement(Feature element, int modifier) {
        this.element = element;
        this.modifier = modifier;
        bounds = null;
        String elementType;
        if (element instanceof Field) {
            elementType = "field"; // NOI18N
        } else if (element instanceof Constructor) {
            elementType = "constructor"; // NOI18N
        } else if (element instanceof Method) {
            elementType = "method"; // NOI18N
        } else {
            elementType = "class"; //NOI18N
        }
        Object[] o = new Object[] { 
                elementType, 
                element.getName(), 
                Modifier.toString(modifier) 
        };
        text = new MessageFormat(
            NbBundle.getMessage(ChangeModElement.class, "DSC_ChangeModifier")).format(o);
        displayText = new MessageFormat(
            NbBundle.getMessage(ChangeModElement.class, "DSC_ChangeModifier_html")).format(o);
        
    }
    
    /**
     * Returns refactored element.
     *
     * @return  refactored element
     */        
    public Element getJavaElement() { return element; }
    
    /**
     * Returns bounds of the field declaration,
     * e.g. <code>int a = 5</code>
     *
     * @return bounds of the field declaration
     */        
    public PositionBounds getPosition() {
        if (bounds == null) {
            bounds = JavaMetamodel.getManager().getElementPosition(element);
        }
        return bounds;
    }
    
    /**
     * Returns the text describing the change provided by element.
     *
     * @return  the text describing element functionality
     */        
    public String getText() { return text; }

    /**
     * Returns text containing the description of the element
     * (i.e. 'Change declaration' and the current declaration of the method.
     *
     * @return description text with method declaration change text
     */
    public String getDisplayText() { return displayText; }
    
    /** 
     * Performs the change represented by this refactoring element.
     */
    public void performChange() {
        if (isEnabled()) {
            element.setModifiers(modifier);
        }
    }
    
    public FileObject getParentFile() {
        return null;
    }
}
