/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.experimental.plugins;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.Iterator;
import java.util.StringTokenizer;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.Import;
import org.netbeans.jmi.javamodel.ImportClass;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.api.AbstractRefactoring;
import org.netbeans.modules.refactoring.api.MoveClassRefactoring;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.api.RenameRefactoring;
import org.netbeans.modules.refactoring.experimental.CopyClassRefactoring;
import org.netbeans.modules.refactoring.plugins.JavaRefactoringPlugin;
import org.netbeans.modules.refactoring.plugins.MoveClassRefactoringPlugin;
import org.netbeans.modules.refactoring.spi.SimpleRefactoringElementImpl;
import org.netbeans.modules.refactoring.spi.RefactoringElementImplementation;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;


/** Plugin that implements the core functionality of Copy Class Refactoring.
 *
 * @author Jan Becicka
 */
public class CopyClassRefactoringPlugin extends JavaRefactoringPlugin {
    /** Reference to the parent refactoring instance */
    private final CopyClassRefactoring refactoring;
    
    /** Creates a new instance of PullUpRefactoringPlugin
     * @param refactoring Parent refactoring instance.
     */
    CopyClassRefactoringPlugin(CopyClassRefactoring refactoring) {
        this.refactoring = refactoring;
    }
    
    /** Checks pre-conditions of the refactoring.
     * @return Problems found or <code>null</code>.
     */
    public Problem preCheck() {
        // fire operation start on the registered progress listeners (3 steps)
        fireProgressListenerStart(AbstractRefactoring.PRE_CHECK, 4);
        try {
            Resource resource = refactoring.getResource();
            
            // check whether the element is valid
            Problem result = isElementAvail(resource);
            if (result != null) {
                // fatal error -> don't continue with further checks
                return result;
            }
            
            if (!CheckUtils.isElementInOpenProject(resource)) {
                return new Problem(true, NbBundle.getMessage(JavaRefactoringPlugin.class, "ERR_ProjectNotOpened"));
            }
            
            // increase progress (step 1)
            fireProgressListenerStep();
            
            
            // increase progress (step 2)
            fireProgressListenerStep();
            
            // increase progress (step 3)
            fireProgressListenerStep();
            
            // all checks passed -> return null
            return null;
        } finally {
            // fire operation end on the registered progress listeners
            fireProgressListenerStop();
        }
    }
    
    public Problem fastCheckParameters() {
        if (!Utilities.isJavaIdentifier(refactoring.getNewName())) {
            String msg = new MessageFormat(NbBundle.getMessage(RenameRefactoring.class, "ERR_InvalidIdentifier")).format(
                new Object[] {refactoring.getNewName()}
            );
            return createProblem(null, true, msg);
        }
        if (!isValidPackageName(refactoring.getTargetPackageName())) {
            String msg = new MessageFormat(NbBundle.getMessage(RenameRefactoring.class, "ERR_InvalidPackage")).format(
                new Object[] {refactoring.getTargetPackageName()}
            );
            return createProblem(null, true, msg);
        }
        String name = refactoring.getTargetPackageName().replace('.','/') + '/' + refactoring.getNewName() + ".java"; // NOI18N
        if (refactoring.getTargetClassPathRoot().getFileObject(name) != null)
            return createProblem(null, true, new MessageFormat(NbBundle.getMessage(MoveClassRefactoring.class, "ERR_ClassToMoveClashes")).format(new Object[]{refactoring.getNewName()}));
        return null;
    }
    
    private static boolean isValidPackageName(String name) {
        StringTokenizer tokenizer = new StringTokenizer(name, "."); // NOI18N
        while (tokenizer.hasMoreTokens()) {
            if (!Utilities.isJavaIdentifier(tokenizer.nextToken())) {
                return false;
            }
        }
        return true;
    }

    public Problem checkParameters() {
        return null;
    }

    public Problem prepare(RefactoringElementsBag refactoringElements) {
        refactoringElements.add(refactoring, 
                new CopyClass(
                    refactoring.getResource(),
                    refactoring.getTargetClassPathRoot(),
                    refactoring.getTargetPackageName(),
                    refactoring.getNewName()
                ));
        //refactoringElements.add(refactoring, new InsertImport(refactoring.getResource(), refactoring.getResource().getPackageName(), false));
        return null;
    }
    
    private static class CopyClass extends SimpleRefactoringElementImpl implements RefactoringElementImplementation{
        
        private DataObject source;
        private FileObject targetRoot;
        private String targetPackageName;
        private String newName;
        private Resource resource;
        
        public CopyClass (Resource resource, FileObject targetRoot, String packageName, String newName) {
            this.source = ((JMManager) JMManager.getManager()).getDataObject(resource);
            this.resource = resource;
            this.targetRoot = targetRoot;
            this.targetPackageName = packageName;
            this.newName = newName;
        }
        
        public String getText() {
            return getDisplayText ();
        }
    
        public String getDisplayText() {
            return new MessageFormat (NbBundle.getMessage(CopyClassRefactoringPlugin.class, "TXT_CopyClassToPackage")).format ( // NOI18N
                new Object[] {newName, targetPackageName, resource.getName()}
            );
        }

        public Element getJavaElement() {
            return null;
        }

        public PositionBounds getPosition() {
            return null;
        }

        public void performChange() {
            //JavaMetamodel.getManager().registerExtChange(this);
            String nameAfterCopy = null;
            try {
                FileObject fo = FileUtil.createFolder(targetRoot, targetPackageName.replace('.','/'));
                DataFolder folder = DataFolder.findFolder(fo);
                objectToDelete = source.copy(folder);
                nameAfterCopy = objectToDelete.getName();
                objectToDelete.rename(newName);
            } catch (IOException ioe) {
                ErrorManager.getDefault().notify(ioe);
            }
            Resource r = JavaModel.getResource(objectToDelete.getPrimaryFile());
            String name = resource.getPackageName();
            ImportClass proxy = ((JavaModelPackage) r.refOutermostPackage ()).getImport();
            Import addedImport = proxy.createImport (name, null, false, true);
            r.addImport(addedImport);
            for (Iterator i = r.getClassifiers().iterator(); i.hasNext(); ) {
                JavaClass c = (JavaClass) i.next();
                if (c.getSimpleName().equals(nameAfterCopy)) {
                    c.setSimpleName(newName);
                }
            }
        }

        private DataObject objectToDelete = null;

        public FileObject getParentFile() {
            return source.getPrimaryFile();
        }
    } 
    
    static class InsertImport extends SimpleRefactoringElementImpl {
        
        protected ImportClass proxy;
        protected Resource source;
        protected String newName;
        protected boolean isNamed;
        private PositionBounds bounds = null;
        
        protected Import addedImport = null;
        
        public InsertImport (Resource source, String newName, boolean isNamed) {
            this.source = source;
            this.newName = newName;
            this.isNamed = isNamed;
            proxy = ((JavaModelPackage) source.refOutermostPackage ()).getImport ();
        }
        
        public String getText() {
            return getDisplayText ();
        }
    
        public String getDisplayText() {
            return new MessageFormat (NbBundle.getMessage(MoveClassRefactoringPlugin.class, "LBL_InsertImport")).format ( // NOI18N
                new Object[] {newName}
            );
        }

        public void performChange() {
            addedImport = proxy.createImport (newName, null, false, !isNamed);
            source.addImport(addedImport);
        }

        public Element getJavaElement() {
            return source;
        }

        public PositionBounds getPosition() {
            if (bounds == null) {
                bounds = JavaMetamodel.getManager().getElementPosition(source);
            }
            return bounds;
        }

        public FileObject getParentFile() {
            return null;
        }
        
    } 
}
