/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.util.actions;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Iterator;
import javax.swing.ActionMap;
import org.netbeans.junit.*;
import org.openide.actions.CopyAction;
import org.openide.windows.TopComponent;
import org.openide.util.HelpCtx;

/** Test CallbackSystemAction: changing performer, focus tracking.
 * @author Jesse Glick
 */
public class CallbackSystemActionTest extends NbTestCase {
    
    static {
        // Get Lookup right to begin with.
        ActionsInfraHid.class.getName();
    }
    
    public CallbackSystemActionTest(String name) {
        super(name);
    }
    

    protected void setUp () throws Exception {
        super.setUp();
    }
    

    protected void tearDown() throws Exception {
        super.tearDown();
        SimpleCallbackAction.waitInstancesZero();
    }
    

    protected boolean runInEQ () {
        return true;
    }

    public void testPropertyChangeListenersDetachedAtFinalizeIssue58100 () throws Exception {
        
        class MyAction extends javax.swing.AbstractAction 
        implements org.openide.util.actions.ActionPerformer {            
            public void actionPerformed (java.awt.event.ActionEvent ev) {
            }
            public void performAction (SystemAction a) {
            }
        }
        MyAction action = new MyAction ();
        ActionMap map = new ActionMap ();
        CallbackSystemAction systemaction = (CallbackSystemAction)SystemAction.get(SimpleCallbackAction.class);
        map.put (systemaction.getActionMapKey(), action);
        org.openide.util.Lookup context = org.openide.util.lookup.Lookups.singleton(map);
        javax.swing.Action delegateaction = systemaction.createContextAwareInstance(context);
        
        assertTrue ("Action is expected to have a PropertyChangeListener attached", action.getPropertyChangeListeners().length > 0);

        Reference actionref = new WeakReference (systemaction);
        systemaction = null;
        delegateaction = null;
        assertGC ("CallbackSystemAction is supposed to be GCed", actionref);
        
        assertEquals ("Action is expected to have no PropertyChangeListener attached", 0, action.getPropertyChangeListeners().length);
    }
    
    public void testSurviveFocusChangeInTheNewWay () throws Exception {
        doSurviveFocusChangeInTheNewWay (false);
    }
    
    public void testSurviveFocusChangeInTheNewWayEvenActionIsGCed () throws Exception {
        doSurviveFocusChangeInTheNewWay (true);
    }
    
    private void doSurviveFocusChangeInTheNewWay (boolean doGC) throws Exception {
        class MyAction extends javax.swing.AbstractAction {
            public int cntEnabled;
            public int cntPerformed;
            
            public boolean isEnabled () {
                cntEnabled++;
                return true;
            }
            
            public void actionPerformed (java.awt.event.ActionEvent ev) {
                cntPerformed++;
            }
        }
        MyAction myAction = new MyAction ();
        
        TopComponent other = new TopComponent ();
        TopComponent tc = new TopComponent ();
        SurviveFocusChgCallbackAction a = (SurviveFocusChgCallbackAction)SurviveFocusChgCallbackAction.get (SurviveFocusChgCallbackAction.class);
        tc.getActionMap().put (a.getActionMapKey (), myAction);
        
        ActionsInfraHid.UT.setActivated(other);
        try {
            assertFalse ("Disabled on other component", a.isEnabled ());
            ActionsInfraHid.UT.setActivated (tc);
            assertTrue ("MyAction is enabled", a.isEnabled ());
            assertEquals ("isEnabled called once", 1, myAction.cntEnabled);
            
            if (doGC) {
                java.lang.ref.WeakReference ref = new java.lang.ref.WeakReference (a);
                a = null;
                assertGC ("Action can disappear", ref);
                a = (SurviveFocusChgCallbackAction)SurviveFocusChgCallbackAction.get (SurviveFocusChgCallbackAction.class);
            }
            
            ActionsInfraHid.UT.setActivated (other);
            assertTrue ("Still enabled", a.isEnabled ());
            assertEquals ("isEnabled called still only once (now it is called twice)", 2, myAction.cntEnabled);
        } finally {
            ActionsInfraHid.UT.setActivated (null);
        }
        
        WeakReference ref = new WeakReference (a);
        WeakReference ref2 = new WeakReference (myAction);
        WeakReference ref3 = new WeakReference (tc);
        a = null;
        myAction = null;
        tc = null;
        assertGC ("We are able to clear global action", ref);
        assertGC ("Even our action", ref2);
        assertGC ("Even our component", ref3);
    }
    
    /** Make sure that the performer system works and controls enablement.
     */
    public void testPerformer() throws Exception {
        CallbackSystemAction a = (CallbackSystemAction)SystemAction.get(SimpleCallbackAction.class);
        assertFalse(a.isEnabled());
        Performer p = new Performer();
        assertEquals(0, p.count);
        a.setActionPerformer(p);
        assertTrue(a.isEnabled());
        a.actionPerformed(null);
        assertEquals(1, p.count);
        a.setActionPerformer(null);
        assertFalse(a.isEnabled());
    }
    
    /** Make sure that focus changes turn on or off actions as appropriate.
     */
    public void disabledtestFocusChanges() throws Exception {
        helperTestFocusChanges();
        // CallbackSystemAction keeps a listener separately from the action,
        // so make sure the actions still work after collected and recreated.
        // Note that similar code fails to work in NodeActionTest because the
        // GC will not complete, so if the GC assert here starts to fail, it is
        // OK to comment out the GC, its assert, and the second call to
        // helperTestFocusChanges().
        SimpleCallbackAction.waitInstancesZero();
        helperTestFocusChanges();
    }
    private void helperTestFocusChanges() throws Exception {
        TopComponent t1 = new TopComponent();
        t1.setName("t1");
        TopComponent t2 = new TopComponent();
        t2.setName("t2");
        ActionsInfraHid.UT.setActivated(t1);
        try {
            CallbackSystemAction a1 = (CallbackSystemAction)SystemAction.get(SurviveFocusChgCallbackAction.class);
            assertTrue(a1.getSurviveFocusChange());
            CallbackSystemAction a2 = (CallbackSystemAction)SystemAction.get(SimpleCallbackAction.class);
            assertFalse(a2.getSurviveFocusChange());
            CallbackSystemAction a3 = (CallbackSystemAction)SystemAction.get(DoesNotSurviveFocusChgCallbackAction.class);
            assertFalse(a3.getSurviveFocusChange());
            Performer p = new Performer();
            a1.setActionPerformer(p);
            a2.setActionPerformer(p);
            a3.setActionPerformer(p);
            assertTrue(a1.isEnabled());
            assertTrue(a2.isEnabled());
            assertTrue(a3.isEnabled());
            ActionsInfraHid.UT.setActivated(t2);
            assertTrue(a1.isEnabled());
            assertEquals(p, a1.getActionPerformer());
            assertFalse(a2.isEnabled());
            assertEquals(null, a2.getActionPerformer());
            assertFalse(a3.isEnabled());
            assertEquals(null, a3.getActionPerformer());
        } finally {
            ActionsInfraHid.UT.setActivated(null);
        }
    }

    public void testGlobalChanges () throws Exception {
        class MyAction extends javax.swing.AbstractAction {
            public int cntEnabled;
            public int cntPerformed;
            
            public boolean isEnabled () {
                cntEnabled++;
                return true;
            }
            
            public void actionPerformed (java.awt.event.ActionEvent ev) {
                cntPerformed++;
            }
        }
        MyAction myAction = new MyAction ();
        
        TopComponent tc = new TopComponent ();
        tc.getActionMap().put (javax.swing.text.DefaultEditorKit.copyAction, myAction);
        CopyAction a = (CopyAction)CopyAction.get (CopyAction.class);
        
        ActionsInfraHid.UT.setActivated(tc);
        try {
            assertTrue ("MyAction is enabled", a.isEnabled ());
            assertEquals ("isEnabled called once", 1, myAction.cntEnabled);
            a.setActionPerformer(null);
            assertEquals ("An enabled is currentlly called again", 2, myAction.cntEnabled);
        } finally {
            ActionsInfraHid.UT.setActivated (null);
        }
    }
    
    /** Action performer that counts invocations. */
    public static final class Performer implements ActionPerformer {
        public int count = 0;
        public void performAction(SystemAction action) {
            count++;
        }
    }
    
    /** Simple callback action. */
    public static final class SimpleCallbackAction extends CallbackSystemAction {
        public String getName() {
            return "SimpleCallbackAction";
        }
        public HelpCtx getHelpCtx() {
            return null;
        }
        
        private static ArrayList INSTANCES_WHO = new ArrayList();
        private static Object INSTANCES_LOCK = new Object();
        public static int INSTANCES = 0;
        
        public static void waitInstancesZero() throws InterruptedException {
            synchronized (INSTANCES_LOCK) {
                for (int i = 0; i < 10; i++) {
                    if (INSTANCES == 0) return;

                    ActionsInfraHid.doGC();
                    
                    INSTANCES_LOCK.wait(1000);
                }
                failInstances("Instances are not zero");
            }
        }

        private static void failInstances(String msg) {
            StringWriter w = new StringWriter();
            PrintWriter pw = new PrintWriter(w);
            pw.println(msg + ": " + INSTANCES);
            for (Iterator it = INSTANCES_WHO.iterator(); it.hasNext();) {
                Exception elem = (Exception) it.next();
                elem.printStackTrace(pw);
            }
            pw.close();
            fail(w.toString());
        }
        
        public SimpleCallbackAction() {
            synchronized (INSTANCES_LOCK) {
                INSTANCES++;
                INSTANCES_LOCK.notifyAll();
                INSTANCES_WHO.add(new Exception("Incremented to " + INSTANCES));
                
                if (INSTANCES == 2) {
                    failInstances("Incremented to two. That is bad");
                }
            }
        }
        protected boolean clearSharedData() {
            synchronized (INSTANCES_LOCK) {
                INSTANCES--;
                INSTANCES_LOCK.notifyAll();
                INSTANCES_WHO.add(new Exception("Decremented to " + INSTANCES));
            }
            return super.clearSharedData();
        }
        protected boolean asynchronous() {
            return false;
        }
    }
    
    /** Similar but survives focus changes. */
    public static final class SurviveFocusChgCallbackAction extends CallbackSystemAction {
        protected void initialize() {
            super.initialize();
            setSurviveFocusChange(true);
        }
        public String getName() {
            return "SurviveFocusChgCallbackAction";
        }
        public HelpCtx getHelpCtx() {
            return null;
        }
        protected boolean asynchronous() {
            return false;
        }
    }
    
    /** Similar but does not; should behave like SimpleCallbackAction (it just sets the flag explicitly). */
    public static final class DoesNotSurviveFocusChgCallbackAction extends CallbackSystemAction {
        protected void initialize() {
            super.initialize();
            setSurviveFocusChange(false);
        }
        public String getName() {
            return "SurviveFocusChgCallbackAction";
        }
        public HelpCtx getHelpCtx() {
            return null;
        }
        protected boolean asynchronous() {
            return false;
        }
    }
    
    
    
    
    //
    // Set of tests for ActionMap and context
    //
    
    public void testLookupOfStateInActionMap () throws Exception {
        class MyAction extends javax.swing.AbstractAction 
        implements org.openide.util.actions.ActionPerformer {
            int actionPerformed;
            int performAction;
            
            public void actionPerformed (java.awt.event.ActionEvent ev) {
                actionPerformed++;
            }
            
            public void performAction (SystemAction a) {
		performAction++;
            }
        }
        MyAction action = new MyAction ();
        
        ActionMap map = new ActionMap ();
        CallbackSystemAction system = (CallbackSystemAction)SystemAction.get(SurviveFocusChgCallbackAction.class);
        system.setActionPerformer (null);
        map.put (system.getActionMapKey(), action);

        
        
        javax.swing.Action clone;
        
        
        //
        // Without action map
        //
        
        clone = system.createContextAwareInstance(org.openide.util.Lookup.EMPTY);
        
        assertTrue ("Action should not be enabled if no callback provided", !clone.isEnabled());
        
        system.setActionPerformer (action);
        assertTrue ("Is enabled, because it has a performer", clone.isEnabled());
        system.setActionPerformer (null);
        assertTrue ("Is disabled, because the performer has been unregistered", !clone.isEnabled ());
        
        //
        // test with actionmap
        //
        action.setEnabled (false);
        
        org.openide.util.Lookup context = org.openide.util.lookup.Lookups.singleton(map);
        clone = system.createContextAwareInstance(context);
        
        CntListener listener = new CntListener ();
        clone.addPropertyChangeListener (listener);
        
        assertTrue ("Not enabled now", !clone.isEnabled ());
        action.setEnabled (true);
        assertTrue ("Clone is enabled because the action in ActionMap is", clone.isEnabled ());
        listener.assertCnt ("One change expected", 1);
        
        system.setActionPerformer (action);
        clone.actionPerformed(new java.awt.event.ActionEvent (this, 0, ""));
        assertEquals ("MyAction.actionPerformed invoked", 1, action.actionPerformed);
        assertEquals ("MyAction.performAction is not invoked", 0, action.performAction);
        
        
        action.setEnabled (false);
        assertTrue ("Clone is disabled because the action in ActionMap is", !clone.isEnabled ());
        listener.assertCnt ("Another change expected", 1);
        
        clone.actionPerformed(new java.awt.event.ActionEvent (this, 0, ""));
        assertEquals ("MyAction.actionPerformed invoked again", 2, action.actionPerformed);
        assertEquals ("MyAction.performAction is not invoked, remains 0", 0, action.performAction);
        
    }
    
    private static final class CntListener extends Object
    implements java.beans.PropertyChangeListener {
        private int cnt;
        
        public void propertyChange(java.beans.PropertyChangeEvent evt) {
            cnt++;
        }
        
        public void assertCnt (String msg, int count) {
            assertEquals (msg, count, this.cnt);
            this.cnt = 0;
        }
    } // end of CntListener
    
}
