/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.util.actions;

import java.awt.Image;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.image.BufferedImage;
import java.awt.image.ImageObserver;
import java.awt.image.PixelGrabber;
import javax.swing.Icon;
import javax.swing.JButton;
import org.netbeans.junit.*;
import junit.textui.TestRunner;
import org.openide.util.HelpCtx;
import org.openide.util.lookup.AbstractLookup;

/** Test general aspects of system actions.
 * Currently, just the icon.
 * @author Jesse Glick
 */
public class AsynchronousTest extends NbTestCase {
    
    public AsynchronousTest (String name) {
        super(name);
    }
    
    public static void main(String[] args) {
        TestRunner.run(new NbTestSuite(AsynchronousTest.class));
    }
    
    protected void setUp () {
        System.setProperty("org.openide.util.Lookup", "org.openide.util.actions.AsynchronousTest$Lkp");
        assertNotNull ("ErrManager has to be in lookup", org.openide.util.Lookup.getDefault ().lookup (ErrManager.class));
        ErrManager.messages.delete (0, ErrManager.messages.length ());
    }
    
    protected boolean runInEQ () {
        return true;
    }
    
    public void testExecutionOfActionsThatDoesNotOverrideAsynchronousIsAsynchronousButWarningIsPrinted () throws Exception {
        DoesNotOverride action = (DoesNotOverride)DoesNotOverride.get (DoesNotOverride.class);
        
        synchronized (action) {
            action.actionPerformed (new ActionEvent(this, 0, ""));
            Thread.sleep (500);
            assertFalse ("Not yet finished", action.finished);
            action.wait ();
            assertTrue ("The asynchronous action is finished", action.finished);
        }
        
        if (ErrManager.messages.toString ().indexOf (DoesNotOverride.class.getName () + " should override") < 0) {
            fail ("There should be warning about not overriding asynchronous: " + ErrManager.messages);
        }
    }
    
    public void testExecutionCanBeAsynchronous () throws Exception {
        DoesOverrideAndReturnsTrue action = (DoesOverrideAndReturnsTrue)DoesOverrideAndReturnsTrue.get (DoesOverrideAndReturnsTrue.class);
        
        synchronized (action) {
            action.actionPerformed (new ActionEvent(this, 0, ""));
            Thread.sleep (500);
            assertFalse ("Not yet finished", action.finished);
            action.wait ();
            assertTrue ("The asynchronous action is finished", action.finished);
        }
        
        if (ErrManager.messages.toString ().indexOf (DoesOverrideAndReturnsTrue.class.getName ()) >= 0) {
            fail ("No warning about the class: " + ErrManager.messages);
        }
    }
    
    public void testExecutionCanBeSynchronous () throws Exception {
        DoesOverrideAndReturnsFalse action = (DoesOverrideAndReturnsFalse)DoesOverrideAndReturnsFalse.get (DoesOverrideAndReturnsFalse.class);
        
        synchronized (action) {
            action.actionPerformed (new ActionEvent(this, 0, ""));
            assertTrue ("The synchronous action is finished immediatelly", action.finished);
        }
        
        if (ErrManager.messages.toString ().indexOf (DoesOverrideAndReturnsTrue.class.getName ()) >= 0) {
            fail ("No warning about the class: " + ErrManager.messages);
        }
    }

    public void testExecutionCanBeForcedToBeSynchronous () throws Exception {
        DoesOverrideAndReturnsTrue action = (DoesOverrideAndReturnsTrue)DoesOverrideAndReturnsTrue.get (DoesOverrideAndReturnsTrue.class);
        
        synchronized (action) {
            action.actionPerformed (new ActionEvent(this, 0, "waitFinished"));
            assertTrue ("When asked for synchronous the action is finished immediatelly", action.finished);
        }
        
        if (ErrManager.messages.toString ().indexOf (DoesOverrideAndReturnsTrue.class.getName ()) >= 0) {
            fail ("No warning about the class: " + ErrManager.messages);
        }
    }

    public static class DoesNotOverride extends CallableSystemAction {
        boolean finished;
        
        public HelpCtx getHelpCtx () {
            return HelpCtx.DEFAULT_HELP;
        }
        
        public String getName () {
            return "Should warn action";
        }
        
        public synchronized void performAction () {
            notifyAll ();
            finished = true;
        }
        
    }
    
    public static class DoesOverrideAndReturnsTrue extends DoesNotOverride {
        public boolean asynchronous () {
            return true;
        }
    }
    
    public static final class DoesOverrideAndReturnsFalse extends DoesOverrideAndReturnsTrue {
        public boolean asynchronous () {
            return false;
        }
    }
    
    
    public static final class Lkp extends AbstractLookup {
        public Lkp () {
            this (new org.openide.util.lookup.InstanceContent ());
        }
        
        private Lkp (org.openide.util.lookup.InstanceContent ic) {
            super (ic);
            ic.add (new ErrManager ());
        }
    }
    
    private static final class ErrManager extends org.openide.ErrorManager {
        public static final StringBuffer messages = new StringBuffer ();
        
        public Throwable annotate (Throwable t, int severity, String message, String localizedMessage, Throwable stackTrace, java.util.Date date) {
            return t;
        }
        
        public Throwable attachAnnotations (Throwable t, org.openide.ErrorManager.Annotation[] arr) {
            return t;
        }
        
        public org.openide.ErrorManager.Annotation[] findAnnotations (Throwable t) {
            return null;
        }
        
        public org.openide.ErrorManager getInstance (String name) {
            return this;
        }
        
        public void log (int severity, String s) {
            messages.append (s);
            messages.append ('\n');
        }
        
        public void notify (int severity, Throwable t) {
        }
        
    } 
}
