/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.masterfs;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.ref.WeakReference;
import java.util.*;
import org.openide.filesystems.*;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.jar.JarOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import org.openide.util.RequestProcessor;

import org.openide.util.io.NbMarshalledObject;
import org.openide.util.Utilities;

import javax.swing.filechooser.FileSystemView;
import org.netbeans.modules.masterfs.filebasedfs.fileobjects.WriteLockUtils;
import org.netbeans.modules.masterfs.providers.ProvidedExtensionsTest;

public class MasterFileObjectTestHid extends TestBaseHid{
    private FileObject root;

    public MasterFileObjectTestHid(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        root = testedFS.findResource(getResourcePrefix());
    }

    protected String[] getResources(String testName) {
        return new String[] {"testdir/ignoredir/nextdir/", 
                             "testdir/mountdir/nextdir/",
                             "testdir/mountdir2/nextdir/",
                             "testdir/mountdir2/fname/notestfile",
                             "testdir/mountdir2/fname2/testfile",
                             "testdir/mountdir4/file.ext",
                             "testdir/mountdir5/file.ext",
                             "testdir/mountdir6/file.ext",
                             "testdir/mountdir6/file2.ext",
                             "testdir/mountdir7/file2.ext",
                             "testdir/mountdir8/",
                             "testdir/mountdir9/",                
        };
    }

    public void testIssue70456() throws Exception {
        assertNotNull(root);
        FileSystem fs = root.getFileSystem();
        assertNotNull(fs);        
        FileSystem.Status s = fs.getStatus();
        assertNotNull(s);
        assertTrue(s instanceof FileSystem.HtmlStatus);
        FileSystem.HtmlStatus hs = (FileSystem.HtmlStatus)s;
        String aname = hs.annotateNameHtml("<default package>",new HashSet(Arrays.asList(new FileObject[] {root})));
        assertTrue(hs.getClass().getName(), aname.indexOf('<') == -1);
        assertTrue(hs.getClass().getName(), aname.indexOf('>') == -1);
    }
    
    public void testFileUtilToFileObjectIsValid() throws Exception {
        char SEP = File.separatorChar;
        final File fileF = new File(FileUtil.toFile(root).getAbsolutePath() + SEP + "dir" + SEP + "file2");
        File dirF = fileF.getParentFile();
        
        for (int cntr = 0; cntr < 10; cntr++) {
            dirF.mkdir();
            new FileOutputStream(fileF).close();
            root.getFileSystem().refresh(false);
            final List valid = new ArrayList();
            FileUtil.toFileObject(fileF).addFileChangeListener(new FileChangeListener() {
                public void fileAttributeChanged(FileAttributeEvent fe) {
                    update();
                }
                public void fileChanged(FileEvent fe) {
                    update();
                }
                public void fileDataCreated(FileEvent fe) {
                    update();
                }
                public void fileDeleted(FileEvent fe) {
                    update();
                }
                public void fileFolderCreated(FileEvent fe) {
                    update();
                }
                public void fileRenamed(FileRenameEvent fe) {
                    update();
                }
                
                private void update() {
                    FileObject fo;
                    File f = fileF;
                    
                    while ((fo = FileUtil.toFileObject(f)) == null) {
                        f = f.getParentFile();
                    }
                    
                    valid.add(Boolean.valueOf(fo.isValid()));
                }
            });
            fileF.delete();
            dirF.delete();
            root.getFileSystem().refresh(false);
            
            assertTrue("at least one event", valid.size() > 0);
            
            for (Iterator i = valid.iterator(); i.hasNext(); ) {
                assertTrue("valid=" + valid + ", count=" + cntr, ((Boolean) i.next()).booleanValue());
            }
        }
    }
    
    public void testRefresh69744() throws Exception {
        File thisTest = new File(getWorkDir(),"thisFolder/thisFolder");
        thisTest.mkdirs();
        thisTest = new File(thisTest,"thisTest");
        thisTest.createNewFile();
        FileObject testf = FileUtil.toFileObject(thisTest);
        assertNotNull(testf);
        assertGC("",new WeakReference(testf.getParent()));        
        modifyFileObject(testf, "abc");
        FileSystem fs = testf.getFileSystem();
        final List l = new ArrayList();
        FileChangeListener fcl = new FileChangeAdapter() {
            public void fileChanged(FileEvent fe) {
                l.add(fe);
            }
        };
        Thread.sleep(1500);
        fs.addFileChangeListener(fcl);
        try {
            modifyFileObject(testf, "def");
            assertFalse(l.isEmpty());
        } finally {
            fs.removeFileChangeListener(fcl);
        }
    }
    
    private void modifyFileObject(final FileObject testf, String content) throws IOException {
        FileLock lock = null;
        OutputStream os = null;
        try {
            lock = testf.lock();
            os = testf.getOutputStream(lock);
            os.write(content.getBytes());
        } finally {
            if (os != null) os.close();
            if (lock != null) lock.releaseLock();            
        }
    }
    
    public void testCaseInsensitivity() throws Exception {
        if (!Utilities.isWindows()) return;
        
        File testa = new File(getWorkDir(), "a");
        File testA = new File(getWorkDir(), "A");
        
        if (testA.exists()) {
            assertTrue(testA.delete());
        }
        if (!testa.exists()) {
            assertTrue(testa.createNewFile());
        }

        //FileBasedFileSystem's case sensitivity depends on platform. This is different behaviour
        // than originally provided by AbstractFileSystem.
        FileObject A = root.getFileObject("A");
        assertNotNull(A);
        assertNotNull(root.getFileObject("a"));
        assertSame(root.getFileObject("A"), root.getFileObject("a"));
        assertSame(URLMapper.findFileObject(testa.toURI().toURL()), 
                URLMapper.findFileObject(testA.toURI().toURL()));
        
        //but 
        root.getChildren();
        assertEquals("A",root.getFileObject("A").getName());        
        assertEquals("A",root.getFileObject("a").getName());        
    }

    private class TestListener extends FileChangeAdapter {
        private List fileObjects;
        TestListener(List fileObjects) {
            this.fileObjects = fileObjects;
        }
        public void fileFolderCreated(FileEvent fe) {
            assertTrue(fileObjects.remove(fe.getFile())); 
        }

        public void fileDeleted(FileEvent fe) {
            assertTrue(fileObjects.remove(fe.getFile())); 
        }

        public void fileDataCreated(FileEvent fe) {
            assertTrue(fileObjects.remove(fe.getFile())); 
        }        
    }
    
    public void testSimulatesRefactoringRename() throws Exception {
        if (!Utilities.isWindows()) return;
        assertNotNull(root);
        FileSystem fs = root.getFileSystem();
        assertNotNull(fs);        
        FileObject main = root.createData("Main.java");
        FileUtil.createData(root,"subpackage/newclass.java");
        final List fileObjects = new ArrayList();
        final Set allSubPackages = new HashSet();
        final TestListener tl = new TestListener(fileObjects);
        fs.addFileChangeListener(tl);
        try {
            fs.runAtomicAction(new FileSystem.AtomicAction(){
                public void run() throws IOException {
                    FileObject subpackage = root.getFileObject("subpackage");
                    allSubPackages.add(subpackage);
                    FileObject newclass = subpackage.getFileObject("newclass.java");
                    FileObject subpackage1 = root.createFolder("subpackage1");
                    fileObjects.add(subpackage1);
                    allSubPackages.add(subpackage1);                    
                    FileObject newclass1 = subpackage1.createData("newclass.java");
                    fileObjects.add(newclass1);
                    subpackage.delete();
                    fileObjects.add(subpackage);
                    fileObjects.add(newclass);
                }
            });
        } finally {
            fs.removeFileChangeListener(tl);
        }
        assertTrue(fileObjects.isEmpty());
        assertNotNull(root.getFileObject("Main.java"));
        assertNotNull(root.getFileObject("subpackage1"));
        assertNotNull(root.getFileObject("subpackage1/newclass.java"));
        assertNull(root.getFileObject("subpackage"));
        fs.addFileChangeListener(tl);
        try {
             fs.runAtomicAction(new FileSystem.AtomicAction(){
                public void run() throws IOException {
                    FileObject subpackage1 = root.getFileObject("subpackage1");
                    FileObject newclass = root.getFileObject("subpackage1/newclass.java");
                    FileObject Subpackage = root.createFolder("Subpackage");
                    allSubPackages.add(Subpackage);
                    assertEquals(3,allSubPackages.size());
                    
                    fileObjects.add(Subpackage);
                    FileObject newclass1 = Subpackage.createData("newclass.java");
                    fileObjects.add(newclass1);
                    subpackage1.delete();
                    fileObjects.add(subpackage1);
                    fileObjects.add(newclass);
                }
            });
        } finally {
            fs.removeFileChangeListener(tl);
        }
        assertTrue(fileObjects.isEmpty());
        assertNotNull(root.getFileObject("Main.java"));
        assertNotNull(root.getFileObject("Subpackage/newclass.java"));
        assertNull(root.getFileObject("subpackage1"));
        assertEquals(3,allSubPackages.size());
    }
    
    public void testRefresh60479 () throws Exception {
        final List l = new ArrayList();
        File rootFile = FileUtil.toFile(root);
        assertTrue(rootFile.exists());
        
        File testFile = new File (rootFile, "testRefresh60479.txt");
        assertTrue(testFile.createNewFile());
        assertTrue(testFile.exists());
        
        FileObject testFo = FileUtil.toFileObject(testFile);
        assertNotNull(testFo);
        FileLock lock = testFo.lock();        
        OutputStream os = null;
        
        try {
            os = testFo.getOutputStream(lock);
            os.write("abcdefgh".getBytes());
            lock.releaseLock();
            os.close();
            Thread.sleep(3000);
            os = new FileOutputStream(testFile);
            os.write("ijkl".getBytes());            
            os.close();            
        } finally {            
            if (lock != null && lock.isValid()) {
                lock.releaseLock();
            }
            if (os != null) {
                os.close();
            }
        }
        
        testFo.addFileChangeListener(new FileChangeAdapter(){
            public void fileChanged(FileEvent fe) {
                l.add(fe);
            }
            
        });
        
        testFo.refresh(true);
        assertEquals(1,l.size());
    }
            
    public void testNormalization51910 () throws Exception {
        if (!Utilities.isWindows()) return;
        
        File rootFile = FileUtil.toFile(root);
        assertTrue(rootFile.exists());
        
        File testFile = new File (rootFile, "abc.txt");
        assertTrue(testFile.createNewFile());
        assertTrue(testFile.exists());
        
        File testFile2 = new File (rootFile, "ABC.TXT");
        assertTrue(testFile2.exists());
        
        
        assertEquals(FileUtil.normalizeFile(testFile).toURI().toURL(), FileUtil.normalizeFile(testFile2).toURI().toURL());
    }   
    
    public void testEventsAfterCreatedFiles55550() throws Exception {
        FileObject parent = root.getFileObject("testdir/mountdir8");  
        assertNotNull(parent);
        assertTrue(parent.isFolder());
        parent.getChildren();
        
        File parentFile = FileUtil.toFile(parent);
        assertNotNull(parentFile);
        assertTrue(parentFile.getAbsolutePath(),parentFile.exists());
        File newFile = new File(parentFile, "sun-web.xml");
        assertFalse(newFile.getAbsolutePath(),newFile.exists());
                        
        class FCLImpl extends FileChangeAdapter {
            boolean created;
            public void fileDataCreated(FileEvent e) {
                created = true;
            }
        }        
        FCLImpl fl = new FCLImpl();        
        parent.addFileChangeListener(fl);
        
        assertTrue(newFile.getAbsolutePath(), newFile.createNewFile());
        assertTrue(newFile.exists());
        
        // !!!! This is the source of the problem !!!
        // ask for the new file
        // remove this line ans the test passes
        FileUtil.toFileObject(newFile);
        
        
        parent.refresh();
        parent.removeFileChangeListener(fl);
        assertTrue("Didn't receive a FileEvent on the parent.", fl.created);
    }
    
    public void testIssue49037 () throws Exception {
        assertNotNull(root);
        FileObject fo = root.getFileObject("testdir/");
        assertNotNull(fo);
        
        File f = FileUtil.toFile (fo);
        assertNotNull(f);
        
        File newFile = new File (f, "absolutelyNewFile");
        assertFalse(newFile.exists());
        
        new FileOutputStream (newFile).close();
        assertTrue(newFile.exists());
        assertNotNull(FileUtil.toFileObject(newFile));        
    }
    
    public void testDeadlock55547 () throws Exception {
        assertNotNull(root);
        File rootFile = new File (FileUtil.toFile(root), "testdir/mountdir9/");
        assertNotNull(rootFile);
        assertTrue(rootFile.exists());
        
        IgnoreDirFileSystem fs1 = new IgnoreDirFileSystem();
        
        fs1.setRootDirectory(rootFile);
        MountTable.getDefault().mount(rootFile.getAbsolutePath(), fs1);
        
        FileObject fo = root.getFileObject("testdir/mountdir9/");
        assertNotNull(fo);

        File f = FileUtil.toFile (fo);
        assertNotNull(f);
       
        File nFile = new File (f, "newchild.txt");
        assertFalse(nFile.exists());
        assertTrue(nFile.createNewFile());
        
        final Object wait = new Object ();
        
        FileChangeListener fcl = null;
        final List list = new ArrayList ();
        fo.addFileChangeListener(fcl = new FileChangeAdapter () {
            public void fileDataCreated(final FileEvent fe) {
                RequestProcessor.Task task = RequestProcessor.getDefault().create(new Runnable() {
                    public void run() {
                        synchronized (wait) {
                            assertNotNull(root.getFileObject("testdir/"));
                            list.add(fe);
                        }
                    }                    
                });
                
                task.schedule(0);
                task.waitFinished();                
            }            
        });                        
        
        if (!ProvidedExtensionsTest.ProvidedExtensionsImpl.isImplsDeleteRetVal()) {
            assertNotNull(fo.getFileObject("newchild.txt"));
            assertTrue(list.size() == 1);
        }
        fo.removeFileChangeListener(fcl);
    }
        
    public void testToFileObject () throws Exception {
        assertNotNull(root);

        File rootFile = FileUtil.toFile (root);
        File mnt1 = new File (rootFile, "testdir/mountdir2/fname");
        
        IgnoreDirFileSystem fs1 = new IgnoreDirFileSystem();
        
        fs1.setRootDirectory(mnt1);
        MountTable.getDefault().mount(mnt1.getAbsolutePath(), fs1);
        
        File test = new File (rootFile, "testdir/mountdir2/fname2/testfile");
        assertNotNull (FileUtil.toFileObject(test));                        
    }
    
    public void testFileUtilFromFile () throws Exception {        
        assertNotNull(root);
        
        File f = FileUtil.normalizeFile(getWorkDir());
        IgnoreDirFileSystem ifs = new IgnoreDirFileSystem();
        ifs.setRootDirectory(f);
        
        Repository.getDefault().addFileSystem(ifs);
        Repository.getDefault().addFileSystem(testedFS);
        
        FileObject[] fos = FileUtil.fromFile(f);
        assertTrue(fos.length > 0);
        assertEquals(fos[0].getFileSystem(), testedFS );
        
    }

    public void testFileSystemStatus () throws Exception {
        assertNotNull(root);

        File rootFile = FileUtil.toFile (root);
        File mnt1 = new File (rootFile, "testdir/mountdir6");
        
        IgnoreDirFileSystem fs1 = new IgnoreDirFileSystem();
        
        fs1.setRootDirectory(mnt1);
        MountTable.getDefault().mount(mnt1.getAbsolutePath(), fs1);
        
        assertNotNull (FileUtil.toFileObject(mnt1));                        
        
        java.util.Set files = new java.util.HashSet ();
        java.util.Enumeration en = FileUtil.toFileObject(mnt1).getChildren(true);
        while (en.hasMoreElements()) {
            files.add(en.nextElement());
        }
        FileSystem.Status status = root.getFileSystem().getStatus();
        String name = status.annotateName("name", files);

        Iterator it = files.iterator();
        while (it.hasNext()) {
            FileObject fo = (FileObject)it.next();            
            assertTrue (name.indexOf(fo.getNameExt()) != -1);
        }        
        
    }
    
    public void testIssue45485 () {
        assertNotNull(root);        
        final FileObject testdir = root.getFileObject("testdir.");        
        assertNull(testdir);        
    }
    
    public void testMount1 () throws Exception {
        final List events = new ArrayList ();
        assertNotNull(root);
        final FileObject ignore = root.getFileObject("testdir/ignoredir");
        assertNotNull(ignore);
        assertTrue(ignore.isFolder());
        MountTable mt = MountTable.getDefault();
        FileObject fo = root.getFileObject("testdir").getParent();
        File f = FileUtil.normalizeFile(FileUtil.toFile(fo));
        IgnoreDirFileSystem ifs = new IgnoreDirFileSystem();
        ifs.setRootDirectory(f);
        
        mt.mount(f.getAbsolutePath(), ifs);            
        
        assertNotNull(root.getFileObject("testdir/ignoredir"));        
        mt.unmount(ifs);
        FileObject file = root.getFileObject("testdir/ignoredir");
        assertNotNull(file);
        assertTrue(file.isFolder());
        assertNotNull(file.getParent());
        assertTrue(FileUtil.isParentOf(root, file));        
    }

    public void testDeleteMountPoint () throws Exception {
        assertNotNull(root);
        FileObject mountDir = root.getFileObject("testdir/mountdir");
        assertNotNull(mountDir);
        File mountFile = FileUtil.normalizeFile(FileUtil.toFile(mountDir));
        IgnoreDirFileSystem ifs = new IgnoreDirFileSystem();
        ifs.setRootDirectory(mountFile);
        
        MountTable mt = MountTable.getDefault();
        mt.mount(mountFile.getAbsolutePath(), ifs);

        if (!ProvidedExtensionsTest.ProvidedExtensionsImpl.isImplsDeleteRetVal()) {
            mountDir.delete();
            assertTrue(!mountDir.isValid());
            assertNull(root.getFileObject("testdir/mountdir"));
        }
    }

    public void testDeleteCaptureExteralChanges () throws Exception {
        String externalName = "newfile.external";        
        assertNotNull(root);
        FileObject fileObject = root.getFileObject("testdir/mountdir5/file.ext");        
        assertNotNull(fileObject);
        File f = FileUtil.toFile(fileObject);
                        
        assertNotNull(f);
        f = FileUtil.normalizeFile(f);
        assertTrue(f.exists());
        assertTrue (f.delete());
        
        if (!ProvidedExtensionsTest.ProvidedExtensionsImpl.isImplsDeleteRetVal()) {        
            assertFalse(fileObject.isValid());       
        }
    }
        
    public void testFindResourceCaptureExteralChanges () throws Exception {
        String externalName = "newfile.external";        
        assertNotNull(root);
        FileObject fileObject = root.getFileObject("testdir");        
        assertNotNull(fileObject);
        File f = FileUtil.toFile(fileObject);
        
        assertNull(testedFS.findResource(new File (f, externalName).getAbsolutePath().replace('\\',File.separatorChar)));
        assertNull(fileObject.getFileObject(externalName));
        
        assertNotNull(f);
        f = FileUtil.normalizeFile(f);
        assertTrue(f.exists());
        f = new File (f, externalName);
        assertTrue(!f.exists());       
        assertTrue(f.getAbsolutePath(),f.createNewFile());
        assertNotNull(testedFS.findResource(f.getAbsolutePath().replace('\\',File.separatorChar)));
    }

    public void testGetFileObjectCaptureExteralChanges () throws Exception {
        String externalName = "newfile.external2";        
        assertNotNull(root);
        FileObject fileObject = root.getFileObject("testdir");        
        assertNotNull(fileObject);
        File f = FileUtil.toFile(fileObject);
        
        assertNull(testedFS.findResource(new File (f, externalName).getAbsolutePath().replace('\\',File.separatorChar)));
        assertNull(fileObject.getFileObject(externalName));
        
        assertNotNull(f);
        f = FileUtil.normalizeFile(f);
        assertTrue(f.exists());
        f = new File (f, externalName);
        assertTrue(!f.exists());        
        assertTrue(f.getAbsolutePath(),f.createNewFile());
        assertNotNull(fileObject.getFileObject(externalName));        
    }

    public void testGetFileObject47885 () throws Exception {
        assertNotNull(root);
        
        FileObject fileObject = root.getFileObject("testdir/mountdir4/file.ext");        
        assertNotNull(fileObject);
        
        fileObject = root.getFileObject("testdir/mountdir4/file", "ext");        
        assertNull(fileObject);
        
        fileObject = root.getFileObject("testdir\\mountdir4\\file.ext");        
        assertNull(fileObject);
    }
    
    
    public void testValidRoots () throws Exception {
        FileSystemView fsv = FileSystemView.getFileSystemView();
        
        File[] roots = File.listRoots();
        for (int i = 0; i < roots.length; i++) {
            FileObject root = FileUtil.toFileObject(roots[i]);
            if (fsv.isFloppyDrive(roots[i]) || !roots[i].exists()) {
               assertNull(root);
               continue; 
            }
            
            assertNotNull(roots[i].getAbsolutePath (),root);
            assertEquals(testedFS, root.getFileSystem());
            assertTrue(root.isValid());
        }
        assertNotNull(testedFS.getRoot());    
        assertTrue(testedFS.getRoot().isValid());            
    }
    
    public void testDeserializationOfMasterFSLeadsToTheSameFileSystem () throws Exception {
        NbMarshalledObject stream = new NbMarshalledObject (testedFS);
        Object obj = stream.get ();
        assertSame ("After deserialization it is still the same", testedFS, obj);
    }
    

    public void testNormalizeDrivesOnWindows48681 () {
        if ((Utilities.isWindows () || (Utilities.getOperatingSystem () == Utilities.OS_OS2))) {
            File[] roots = File.listRoots();
            for (int i = 0; i < roots.length; i++) {
                File file = roots[i];
                if (FileSystemView.getFileSystemView().isFloppyDrive(file) || !file.exists()) {
                    continue;
                }
                File normalizedFile = FileUtil.normalizeFile(file);
                File normalizedFile2 = FileUtil.normalizeFile(new File (file, "."));
                
                assertEquals (normalizedFile.getPath(), normalizedFile2.getPath());
            }
            
        }
    }
    
    public void testJarFileSystemDelete () throws Exception {
        assertNotNull(root);
        FileObject folderFo = root.getFileObject("testdir/mountdir7");
        File folder = FileUtil.toFile(folderFo);
        assertNotNull(folder);
        
        File f = new File (folder,"jfstest.jar");        
        if (!f.exists()) {
            f.getParentFile().mkdirs();
            f.createNewFile();
        }
        JarOutputStream jos = new JarOutputStream (new FileOutputStream (f));        
        jos.putNextEntry(new ZipEntry ("a/b/c/c.txt"));
        jos.putNextEntry(new ZipEntry ("a/b/d/d.txt"));
                        
       jos.close();        

        FileObject parent = FileUtil.toFileObject(f.getParentFile());
        parent.getChildren();
        JarFileSystem jfs = new JarFileSystem  ();
        try {
            jfs.setJarFile(f);
        } catch (Exception ex) {
            fail ();
        }
        

        ArrayList all = new ArrayList ();
        FileObject jfsRoot = jfs.getRoot();
        Enumeration en = jfsRoot.getChildren(true);
        while (en.hasMoreElements()) {
            all.add ((FileObject) en.nextElement());                        
        }

        assertTrue (all.size() > 0); 
        
        final ArrayList deleted = new ArrayList ();
        jfs.addFileChangeListener(new FileChangeAdapter() {
            public void fileDeleted(FileEvent fe) {
                super.fileDeleted(fe);
                deleted.add (fe.getFile());
            }
        });
        
        Thread.sleep(1000);
        assertTrue (f.getAbsolutePath(), f.delete());
        parent.refresh();
        assertEquals (deleted.size(), all.size());

        for (int i = 0; i < all.size(); i++) {
            FileObject fileObject = (FileObject) all.get(i);
            assertFalse (fileObject.isValid());
        }
        
        
        assertFalse (jfsRoot.isValid());        
    }
    
    
    public void testInitialization () throws Exception {
        assertTrue (MasterFileSystemTest.TestFSProvider.DEFAULT.isInitailized());    
    }
    

    public void testLockFileAfterCrash() throws Exception {
        FileObject testFo = FileUtil.createData(root,"/testAfterCrash/testfile.data");
        File testFile = FileUtil.toFile(testFo);
  
        
        File lockFile = WriteLockUtils.getAssociatedLockFile(testFile);
        if (!lockFile.exists()) {
            assertTrue(lockFile.createNewFile());
        }
                
        assertTrue(lockFile.exists());

        FileObject lockFo = FileUtil.toFileObject(lockFile);        
        assertNull(lockFo);        
        testFo.delete();        
        
        
        lockFo = FileUtil.toFileObject(lockFile);        
        String msg = (lockFo != null) ? lockFo.toString() : "";
        assertNull(msg,lockFo);
    }
    
    private class IgnoreDirFileSystem extends LocalFileSystem {
        org.openide.filesystems.FileSystem.Status status = new org.openide.filesystems.FileSystem.HtmlStatus() {
            public String annotateName (String name, java.util.Set files) {
                java.lang.StringBuffer sb = new StringBuffer (name);                
                Iterator it = files.iterator ();
                while (it.hasNext()) {                    
                    FileObject fo = (FileObject)it.next();
                    try {
                        if (fo.getFileSystem() instanceof IgnoreDirFileSystem) {
                            sb.append ("," +fo.getNameExt());//NOI18N
                        }
                    } catch (Exception ex) {
                        fail ();
                    }
                }
                                
                return sb.toString () ;
            }

            public java.awt.Image annotateIcon (java.awt.Image icon, int iconType, java.util.Set files) {
                return icon;
            }

            public String annotateNameHtml(String name, Set files) {
                return annotateName (name, files);
            }            
            
        };        
        
        public org.openide.filesystems.FileSystem.Status getStatus() {
            return status;
        }
        
        protected String[] children(String name) {
            String[] strings = super.children(name);
            return strings;
        }
    }
}
