/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.loaders;

import java.io.PrintStream;
import javax.swing.Action;
import junit.textui.TestRunner;
import org.netbeans.junit.*;
import org.openide.DialogDescriptor;
import org.openide.cookies.EditCookie;
import org.openide.cookies.EditorCookie;
import org.openide.cookies.OpenCookie;
import org.openide.filesystems.*;
import org.openide.util.actions.SystemAction;

/** DefaultDataObject is supposed to have open operation that shows the text
 * editor or invokes a dialog with questions.
 *
 * @author  Jaroslav Tulach
 */
public final class DefautDataObjectHasOpenTest extends LoggingTestCaseHid {
    
    private FileSystem lfs;
    private DataObject obj;
    
    /** Creates a new instance of DefaultSettingsContextTest */
    public DefautDataObjectHasOpenTest(String name) {
        super(name);
    }
    
    protected void setUp() throws java.lang.Exception {
        super.setUp();
        
        registerIntoLookup(new DD());
        
        DD x = (DD)org.openide.util.Lookup.getDefault().lookup(DD.class);
        assertNotNull("DD is there", x);
        
        String fsstruct [] = new String [] {
            "AA/a.test"
        };
        
        TestUtilHid.destroyLocalFileSystem(getName());
        lfs = TestUtilHid.createLocalFileSystem(getWorkDir(), fsstruct);
        org.openide.filesystems.Repository.getDefault ().addFileSystem (lfs);
        
        FileObject fo = lfs.findResource("AA/a.test");
        assertNotNull("file not found", fo);
        obj = DataObject.find(fo);
        
        assertEquals ("The right class", obj.getClass (), DefaultDataObject.class);

        assertFalse ("Designed to run outside of AWT", javax.swing.SwingUtilities.isEventDispatchThread ());
        DD.options = null;
        DD.disableTest = false;
    }
    
    private void waitForAWT () throws Exception {
        // just wait till all the stuff from AWT is finished
        org.openide.util.Mutex.EVENT.readAccess (new org.openide.util.Mutex.Action () {
            public Object run () {
                return null;
            }
        });
    }
    
    protected void tearDown() throws java.lang.Exception {
        waitForAWT ();
        DD.disableTest = true;
        
        super.tearDown();
        if (obj != null) {
            org.openide.cookies.CloseCookie cc;
            cc = (org.openide.cookies.CloseCookie)obj.getCookie (org.openide.cookies.CloseCookie.class);
            if (cc != null) {
                DD.toReturn = org.openide.NotifyDescriptor.NO_OPTION;
                cc.close ();
            }
        }
        org.openide.filesystems.Repository.getDefault ().removeFileSystem (lfs);
        
        waitForAWT ();
    }
    
    public void testHasOpenCookie () {
        assertNotNull (obj.getCookie (OpenCookie.class));
    }

    public void testHasEditCookie () {
        assertNotNull (obj.getCookie (EditCookie.class));
    }
    
    public void testHasEditorCookieForResonableContentOfFiles () throws Exception {
        EditorCookie c = tryToOpen (
            "Ahoj Jardo," +
            "how are you" +
            "\t\n\rBye.\u00a9\u00e1\u00e9\u00ed\u00f3\u00fa"
        );
        assertNotNull (c);
        
        doRegularCheck (c);
    }
    
    private void doRegularCheck (EditorCookie c) throws Exception {
        assertEquals (
            "Next questions results in the same cookie", 
            c, 
            obj.getCookie(EditorCookie.class)
        );
        assertEquals (
            "Print cookie is provided",
            c,
            obj.getCookie(org.openide.cookies.PrintCookie.class)
        );
        assertEquals (
            "CloseCookie as well",
            c,
            obj.getCookie(org.openide.cookies.CloseCookie.class)
        );
        
        OpenCookie open = (OpenCookie)obj.getCookie (OpenCookie.class);
        open.open ();
        
        javax.swing.text.Document d = c.getDocument();
        assertNotNull (d);
        
        d.insertString(0, "Kuk", null);
        
        assertNotNull (
            "Now there is a save cookie", 
            obj.getCookie (org.openide.cookies.SaveCookie.class)
        );
    }
    
    public void testNoEditorCookieForBinaryFiles () throws Exception {
        EditorCookie c = tryToOpen ("\u0003\u0001\u0000");
        assertNull (c);
    }
    
    public void testOpenAsksAQuestionForBinaryFiles () throws Exception {
        EditorCookie c = tryToOpen ("\u0003\u0001\u0000");
        assertNull (c);
        
        DD.toReturn = DialogDescriptor.CANCEL_OPTION;
        OpenCookie open = (OpenCookie)obj.getCookie (OpenCookie.class);
        open.open ();
        
        assertNotNull ("There was a query", DD.options);
        assertEquals ("Yes no options", 2, DD.options.length);
        assertEquals (DialogDescriptor.OK_OPTION, DD.options[0]);
        assertEquals (DialogDescriptor.CANCEL_OPTION, DD.options[1]);
    }
    
    public void testBinaryFilesQuestionContainsOpenActions () throws Exception {
        EditorCookie c = tryToOpen ("\u0003\u0001\u0000");
        assertNull (c);
        
        DD.toReturn = DialogDescriptor.CLOSED_OPTION;
        OpenCookie open = (OpenCookie)obj.getCookie (OpenCookie.class);
        open.open ();

        assertNotNull ("There was a query", DD.options);
        assertEquals ("Two options", 2, DD.options.length);
        assertEquals (DialogDescriptor.OK_OPTION, DD.options[0]);
        assertEquals (DialogDescriptor.CANCEL_OPTION, DD.options[1]);
        DD.options = null;

        DD.toReturn = DialogDescriptor.OK_OPTION;
        open.open ();

        assertNotNull ("There was a query", DD.options);
        assertEquals ("Still 2 options", 2, DD.options.length);
        assertEquals (DialogDescriptor.OK_OPTION, DD.options[0]);
        assertEquals (DialogDescriptor.CANCEL_OPTION, DD.options[1]);

        c = (EditorCookie)obj.getCookie (EditorCookie.class);
        assertNotNull ("Editor cookie created", c);
        
        doRegularCheck (c);
    }
    
    private EditorCookie tryToOpen (String content) throws Exception {
        FileObject fo = obj.getPrimaryFile();
        FileLock lock = fo.lock();
        PrintStream os = new PrintStream (fo.getOutputStream(lock));
        os.print (content);
        os.close ();
        lock.releaseLock();
        
        return (EditorCookie)obj.getCookie (EditorCookie.class);
    }
    
    public void testThatTheNameOfDataObjectsNodeContainsTheExtensionIssue18780 () throws Exception {
        String name = obj.getPrimaryFile ().getNameExt ();
        assertEquals ("Name contains extension", name, obj.getNodeDelegate ().getName ());
    }
    
    public void testComponentCanBeSerialized () throws Exception {
        doComponentCanBeSerialized ("Sample\nTest\nTo\nSerialized");
    }

    public void testComponentCanBeSerializedEvenItIsBinary () throws Exception {
        // make sure it has the cookie
        OpenCookie open = (OpenCookie)obj.getCookie (OpenCookie.class);
        DD.toReturn = DialogDescriptor.OK_OPTION;
        open.open ();
        open = null;
        
        doComponentCanBeSerialized ("\u0003\u0001\u0000");
    }
    
    public void testComponentCanBeSerializedEvenItIsBig () throws Exception {
        // make sure it has the cookie
        OpenCookie open = (OpenCookie)obj.getCookie (OpenCookie.class);
        DD.toReturn = DialogDescriptor.OK_OPTION;
        open.open ();
        open = null;
        
        FileObject fo = obj.getPrimaryFile();
        FileLock lock = fo.lock();
        PrintStream os = new PrintStream (fo.getOutputStream(lock));
        for (int i = 0; i < 1024 * 1024 / 10; i++) {
            os.print ("Ahoj " + i + "\n");
        }
        os.close ();
        lock.releaseLock();
        assertTrue ("Larger than 1Mb", 1024 * 1024 + 1000 < fo.getSize ());
        
        doComponentCanBeSerialized (null);
    }
    
    private void doComponentCanBeSerialized (String txt) throws Exception {
        EditorCookie c;
        if (txt != null) {
            c = tryToOpen (txt);
        } else {
            c = (EditorCookie)obj.getCookie (EditorCookie.class);
        }
        assertNotNull (c);
        c.open ();
        
        // wait for AWT thread
        javax.swing.SwingUtilities.invokeAndWait (new Runnable () { public void run () {} });

        javax.swing.JEditorPane[] arr = c.getOpenedPanes ();
        assertNotNull ("Something opened", arr);
        assertEquals ("One opened", 1, arr.length);
        
        Object o = javax.swing.SwingUtilities.getAncestorOfClass (org.openide.windows.TopComponent.class, arr[0]);
        assertNotNull ("Top component found", o);
        org.openide.util.io.NbMarshalledObject mar = new org.openide.util.io.NbMarshalledObject (o);

        ((org.openide.windows.TopComponent)o).close ();
        
        obj.setValid (false);
        
        org.openide.windows.TopComponent tc = (org.openide.windows.TopComponent)mar.get ();
        
        assertNotNull ("Successfully deserialized", tc);
     
        if (obj == DataObject.find (obj.getPrimaryFile ())) {
            fail ("Strange, obj should be garbage collected" + obj);
        }
    }

    /** Our own dialog displayer.
     */
    private static final class DD extends org.openide.DialogDisplayer {
        public static Object[] options;
        public static Object toReturn;
        public static boolean disableTest;
        
        public java.awt.Dialog createDialog(org.openide.DialogDescriptor descriptor) {
            throw new IllegalStateException ("Not implemented");
        }
        
        public Object notify(org.openide.NotifyDescriptor descriptor) {
            if (disableTest) {
                return toReturn;
            } else {
                assertNull (options);
                assertNotNull (toReturn);
                options = descriptor.getOptions();
                Object r = toReturn;
                toReturn = null;
                return r;
            }
        }
        
    } // end of DD


}
