/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.mdr.storagemodel.transientimpl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Set;
import java.util.Collection;
import java.util.Iterator;
import org.netbeans.mdr.persistence.SinglevaluedIndex;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.persistence.StorageBadRequestException;
import org.netbeans.mdr.persistence.Storage;
import org.netbeans.mdr.storagemodel.MdrStorage;
import org.netbeans.mdr.util.MapEntryImpl;
import org.netbeans.mdr.util.DebugException;
import org.netbeans.mdr.util.Logger;

/**
 *
 * @author Tomas Zezula
 */
public class TransientSinglevaluedIndex extends TransientIndex implements SinglevaluedIndex {
    
    protected class SinglevaluedEntryKeySet extends EntryKeySet {
        
        protected ArrayList collectKeys() {
            ArrayList result = new ArrayList();
            for (Iterator it = map.values().iterator(); it.hasNext();) {
                Entry entry = (Entry) it.next();
                if (entry.isValid())
                    result.add(entry.getKey());
            }
            return result;
        }
        
        public java.util.Iterator iterator() {
            return new SinglevaluedIterator(true);
        }
        
    }
    
    protected class ValuesCollection implements Collection {
        
        public boolean add(Object obj) {
            throw new UnsupportedOperationException();
        }
        
        public boolean addAll(java.util.Collection collection) {
            throw new UnsupportedOperationException();
        }
        
        public void clear() {
            for (Iterator it = map.values().iterator(); it.hasNext();) {
                Entry entry = (Entry) it.next();
                it.remove();
                try {
                    handleRemove(entry.getKey(), entry);
                } catch (StorageException se) {
                    Logger.getDefault().notify(Logger.INFORMATIONAL, se);
                }
            }
        }
        
        public boolean contains(Object obj) {
            for (Iterator it = map.values().iterator(); it.hasNext();) {
                Entry entry = (Entry) it.next();
                Object value = entry.getValue();
                if (value == null) {
                    it.remove();
                    entry.dispose();
                }
                else if (value.equals(obj)) {
                    return true;
                }
            }
            return false;
        }
        
        public boolean containsAll(java.util.Collection collection) {
            boolean result = true;
            for (Iterator it = collection.iterator(); it.hasNext();) {
                result &= this.contains(it.next());
            }
            return result;
        }
        
        public boolean isEmpty() {
            expungeStaleEntries();
            return map.size() == 0;
        }
        
        public java.util.Iterator iterator() {
            return new SinglevaluedIterator(false);
        }
        
        public boolean remove(Object obj) {
            try {
                return TransientSinglevaluedIndex.this.remove(obj);
            }catch (StorageException se) {
                throw new DebugException(se.toString());
            }
        }
        
        public boolean removeAll(java.util.Collection collection) {
            boolean result = false;
            for (Iterator it = collection.iterator(); it.hasNext();) {
                result |= this.remove(it.next());
            }
            return result;
        }
        
        public boolean retainAll(java.util.Collection collection) {
            boolean result = false;
            for (Iterator it = map.values().iterator(); it.hasNext();) {
                Entry entry = (Entry) it.next();
                if (!entry.isValid()) {
                    it.remove();
                    entry.dispose();
                }
                else if (!collection.contains(entry.getValue())) {
                    it.remove();
                    try {
                        handleRemove(entry.getKey(), entry);
                        result = true;
                    } catch (StorageException se) {
                        Logger.getDefault().notify(Logger.INFORMATIONAL, se);
                    }
                }
            }
            return result;
        }
        
        public int size() {
            expungeStaleEntries();
            return map.size();
        }
        
        public Object[] toArray() {
            ArrayList result = collectValues();
            return result.toArray();
        }
        
        public Object[] toArray(Object[] obj) {
            ArrayList result = collectValues();
            return result.toArray(obj);
        }
        
        public ArrayList collectValues() {
            ArrayList result = new ArrayList();
            for (Iterator it = map.values().iterator(); it.hasNext();) {
                Entry entry = (Entry) it.next();
                if (!entry.isValid()) {
                    it.remove();
                    entry.dispose();
                }
                else
                    result.add(entry.getValue());
            }
            return result;
        }
        
    }
    
    protected class SinglevaluedIterator implements Iterator {
        
        private Iterator innerIt;
        private boolean isKey;
        private Entry top;
        private Entry last;
        
        public SinglevaluedIterator(boolean isKey) {
            this.innerIt = map.values().iterator();
            this.isKey = isKey;
        }
        
        public boolean hasNext() {
            while (top == null) {
                if (!this.innerIt.hasNext())
                    return false;
                top = (Entry) this.innerIt.next();
                if (!top.isValid()) {
                    top.dispose();
                    top = null;
                    innerIt.remove();
                }
            }
            return true;
        }
        
        public Object next() {
            while (top == null) {
                top = (Entry) this.innerIt.next();
                if (!top.isValid()) {
                    top.dispose();
                    top = null;
                    innerIt.remove();
                }
            }
            last = top;
            top = null;
            if (this.isKey)
                return last.getKey();
            else
                return last.getValue();
        }
        
        public void remove() {
            if (last != null) {
                innerIt.remove();
                try {
                    handleRemove(last.getKey(), last);
                } catch (StorageException se) {
                    Logger.getDefault().notify(Logger.INFORMATIONAL, se);
                }
                last = null;
            }
            else {
                throw new IllegalStateException();
            }
        }
    }
    
    /** Creates a new instance of TransientSinglevaluedIndex */
    public TransientSinglevaluedIndex(MdrStorage storage, String name, Storage.EntryType keyType, Storage.EntryType valueType) {
        super(storage, name, keyType, valueType);
    }
    
    public void add(Object key, Object value) throws StorageException {
        this.addNoTx(key, value);
        this.handleAdd(key, value);
    }
    
    public Object get(Object key) throws StorageException, StorageBadRequestException {
        Object result = this.getIfExists(key);
        if (result == null)
            throw new StorageBadRequestException();
        return result;
    }
    
    public Object getIfExists(Object key) throws StorageException {
        if (this.map == null)
            return null;
        else
            expungeStaleEntries();
        Entry e = (Entry) this.map.get(key);
        if (e == null)
            return null;
        if (!e.isValid())
            return null;
        else
            return e.getValue();
    }
    
    public Object getObject(Object key, SinglevaluedIndex repos) throws StorageException {
        Object result = this.getObjectIfExists(key, repos);
        if (result == null)
            throw new StorageBadRequestException();
        return result;
    }
    
    public Object getObjectIfExists(Object key, SinglevaluedIndex repos) throws StorageException {
        Object result = this.get(key);
        if (result == null)
            return result;
        else
            return repos.get(repos);
    }
    
    protected void handleRemove(Object key, Object value) throws StorageException {
        Object val =((Entry)value).getValue();
        this.txlog.push(new CompensatingTransaction.RemoveCTx(key, val));
        ((Entry)value).dispose();
    }
    
    public java.util.Set keySet() throws StorageException {
        if (this.map == null)
            this.map = new HashMap();
        return new SinglevaluedEntryKeySet();
    }
    
    /** Associates the specified value with the specified key in this index.
     * @return true if there was an item in this index that was associated with the key
     * prior to this call
     * @param key
     * @param value
     * @throws StorageException
     */
    public boolean put(Object key, Object value) throws StorageException {
        if (this.map == null)
            this.map = new HashMap();
        else
            expungeStaleEntries();
        Entry e = new Entry(key, value);
        Object oldValue = map.put(key, e);
        if (oldValue != null)
            this.handleRemove(key, oldValue);
        this.handleAdd(key, value);
        return oldValue != null;
    }
    
    /** Replaces the original value associated with the specified key in this index
     * with new value. If no value was associated with this key prior to this call
     * StorageBadRequestException is thrown.
     * @param key
     * @param value
     * @throws StorageException
     * @throws StorageBadRequestException
     */
    public void replace(Object key, Object value) throws StorageException, StorageBadRequestException {
        if (map == null)
            throw new StorageBadRequestException();
        else {
            expungeStaleEntries();
            if (this.map.get(key) == null)
                throw new StorageBadRequestException();
        }
        Entry e = new Entry(key, value);
        Object oldValue = this.map.put(key, e);
        this.handleRemove(key, oldValue);
        this.handleAdd(key, value);
    }
    
    /** Returns a collection view of the values contained in this index.
     * Returned collection is read only and may not be modified.
     * If this index has no items, empty Collection is returned.
     * @return
     * @throws StorageException
     */
    public java.util.Collection values() throws StorageException {
        return new ValuesCollection();
    }
    
    /**
     * Returns key-value pairs, where the key contains the queried prefix.
     */
    public Collection queryByKeyPrefix (Object prefix, SinglevaluedIndex primaryIndex) throws StorageException {
        if (getKeyType () != Storage.EntryType.STRING) {
            throw new UnsupportedOperationException ("Key type must be EntryType.STRING");
        }
        if (!(prefix instanceof String)) {
            throw new StorageBadRequestException ("String object parameter expected.");
        }
        
        java.util.List result = new java.util.LinkedList ();
        Iterator iter = keySet().iterator ();
        while (iter.hasNext ()) {            
            String key = (String) iter.next ();
            if (key.startsWith ((String) prefix)) {
                result.add (new MapEntryImpl (key, getObject (key, primaryIndex)));
            }
        }
        return result;
    }
    
    protected void expungeStaleEntries() {
        KeyedReference ref = null;
        while ((ref = (KeyedReference) this.refQueue.poll()) != null) {
            Object key = ref.getLookupKey();
            Entry e  = (Entry) this.map.remove(key);
            if (e !=  null) // Was not removed by second end
                e.dispose();
        }
    }
    
    protected final void addNoTx(Object key, Object value) throws StorageException {
        if (this.map == null) {
            this.map = new HashMap();
        }
        else {
            expungeStaleEntries();
            if (this.map.get(key) != null)
                throw new StorageBadRequestException();
        }
        Entry e = new Entry(key, value);
        this.map.put(key, e);
    }
    
    protected final Object removeNoTx (Object key, Object value) throws StorageException {
         expungeStaleEntries();
        Entry e = (Entry) this.map.remove(key);
        Object result = e.getValue ();
        e.dispose ();
        return result;
    }
    
}
