/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.mdr.storagemodel.transientimpl;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.util.ListIterator;
import java.util.Iterator;
import org.netbeans.mdr.persistence.MultivaluedOrderedIndex;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.persistence.StorageBadRequestException;
import org.netbeans.mdr.persistence.Storage;
import org.netbeans.mdr.persistence.SinglevaluedIndex;
import org.netbeans.mdr.storagemodel.MdrStorage;
import org.netbeans.mdr.util.DebugException;



/**
 *
 * @author Tomas Zezula
 */
public class TransientMultivaluedOrderedIndex extends TransientMultivaluedIndex implements MultivaluedOrderedIndex {
    
    public class OrderedSlotCollection extends SlotCollection implements List {
        
        
        public OrderedSlotCollection(Object key, Collection st) {
            this(key, st, null);
        }
        
        public OrderedSlotCollection(Object key, Collection st, int index, int toIndex) {
            this(key, st, null, index, toIndex);
        }
        
        public OrderedSlotCollection(Object key, Collection st, SinglevaluedIndex repos) {
            super(key, st, repos);
        }
        
        public OrderedSlotCollection(Object key, Collection st, SinglevaluedIndex repos, int fromIndex, int toIndex) {
            super(key, ((List)st).subList(fromIndex,toIndex), repos);
        }
        
        public void add(int index, Object element) {
            if (this.repos != null)
                throw new UnsupportedOperationException();
            try {
                addToList((List)this.st, index, this.key, element);
                txlog.push( new CompensatingTransaction.AddOrderedCTx(this.key, element,  index));
            } catch (StorageException se) {
                throw new DebugException(se.toString());
            }
        }
        
        public boolean addAll(int index, Collection c) {
            boolean result = false;
            for (Iterator it = c.iterator(); it.hasNext(); index++) {
                this.add(index, it.next());
                result = true;
            }
            return result;
        }
        
        public Object get(int index) {
            Entry e = (Entry) ((List)this.st).get(index);
            if (e == null)
                return null;
            if (!e.isValid()) {
                ((List)this.st).remove(index);
                e.dispose();
                return null;
            }
            try {
                return (this.repos == null) ? e.getValue() : map(this.key, e.getValue(), this.repos);
            } catch (StorageException se) {
                throw new DebugException(se.toString());
            }
        }
        
        public int indexOf(Object o) {
            if (this.repos != null)
                throw new UnsupportedOperationException();
            List l = (List) this.st;
            for (int i= 0; i < l.size(); i++) {
                Entry e = (Entry) l.get(i);
                if (!e.isValid()) {
                    l.remove(i);
                    e.dispose();
                    i--;
                }
                else if (o.equals(e.getValue())) {
                    return i;
                }
            }
            return -1;
        }
        
        public int lastIndexOf(Object o) {
            if (this.repos != null)
                throw new UnsupportedOperationException();
            List l = (List) this.st;
            for (int i = l.size()-1; i>=0; i--) {
                Entry e = (Entry) l.get(i);
                if (!e.isValid()) {
                    l.remove(i);
                    e.dispose();
                }
                else if (o.equals(e.getValue())) {
                    return i;
                }
            }
            return -1;
        }
        
        public ListIterator listIterator() {
            return new OrderedSlotIterator(this.key, (List)this.st, this.repos);
        }
        
        
        public ListIterator listIterator(int index) {
            return new OrderedSlotIterator(this.key, (List)this.st, this.repos, index);
        }
        
        public Object remove(int index) {
            if (this.repos != null)
                throw new UnsupportedOperationException();
            if (index < 0 || index >= this.st.size())
                throw new IndexOutOfBoundsException();
            try {
                Object result = removeFromList((List)this.st, index);
                txlog.push( new CompensatingTransaction.RemoveOrderedCTx(this.key, result, index));
                return result;
            } catch (StorageException se) {
                throw new DebugException(se.toString());
            }
        }
        
        
        public Object set(int index, Object element) {
            if (this.repos != null)
                throw new UnsupportedOperationException();
            try {
                Object result = setInList((List)this.st, index, this.key, element);
                if (result != null) {
                    txlog.push(new CompensatingTransaction.RemoveOrderedCTx(key, result, index));
                }
                txlog.push(new CompensatingTransaction.AddOrderedCTx(key, result, index));
                return result;
            } catch (StorageException se) {
                throw new DebugException(se.toString());
            }
        }
        
        public List subList(int fromIndex, int toIndex) {
            return new OrderedSlotCollection(this.key, this.st, this.repos, fromIndex, toIndex);
        }
        
    }
    
    
    protected class OrderedSlotIterator extends SlotIterator implements ListIterator {
        
        private Entry prev;
        private int position;
        
        public OrderedSlotIterator(List l) {
            this(null, l, null, 0);
        }
        
        public OrderedSlotIterator(List l, int index) {
            this(null, l, null, index);
        }
        
        public OrderedSlotIterator(Object key, List l, SinglevaluedIndex repos) {
            this(null, l, repos, 0);
        }
        
        public OrderedSlotIterator(Object key, List l, SinglevaluedIndex repos, int index) {
            super(key, l, repos, (index == 0) ? l.listIterator() : l.listIterator(index));
        }
        
        public Object next() {
            while (this.top == null) {
                this.top = (Entry) this.innerIt.next();
                if (! this.top.isValid()) {
                    this.innerIt.remove();
                    this.top.dispose();
                    this.top = null;
                }
            }
            this.last = this.top;
            this.top = prev = null;
            try {
                Object result = map(this.key, this.last.getValue(), this.repos);
                this.position++;
                return result;
            }catch (StorageException se) {
                throw new DebugException(se.toString());
            }
        }
        
        public boolean hasPrevious() {
            while (this.prev == null) {
                if (!((ListIterator)this.innerIt).hasPrevious())
                    return false;
                this.prev = (Entry) ((ListIterator)this.innerIt).previous();
                if (! this.prev.isValid()) {
                    this.innerIt.remove();
                    this.prev.dispose();
                    this.prev = null;
                }
            }
            return true;
        }
        
        public Object previous() {
            while (this.prev == null) {
                this.prev = (Entry) ((ListIterator)this.innerIt).previous();
                if (! this.prev.isValid()) {
                    this.innerIt.remove();
                    this.prev.dispose();
                    this.prev = null;
                }
            }
            this.last = this.prev;
            this.top = prev = null;
            try {
                Object result = map(this.key, this.last.getValue(), this.repos);
                this.position--;
                return result;
            }catch (StorageException se) {
                throw new DebugException(se.toString());
            }
        }
        
        public int nextIndex() {
            return ((ListIterator)this.innerIt).nextIndex();
        }
        
        public int previousIndex() {
            return ((ListIterator)this.innerIt).previousIndex();
        }
        
        public void add(Object o) {
            if (this.repos != null)
                throw new UnsupportedOperationException();
            if (this.last == null)
                throw new IllegalStateException();
            if (unique && this.collection.contains(o))
                throw new IllegalStateException("Object already contained in unique index.");
            try {
                Entry e = new Entry(this.key, o);
                ((ListIterator)innerIt).add(e);
                txlog.push(new CompensatingTransaction.AddOrderedCTx(this.key, o, this.position));
            }catch (StorageException se) {
                throw new DebugException();
            }
        }
        
        public void set(Object o) {
            if (this.repos != null)
                throw new UnsupportedOperationException();
            if (this.last == null)
                throw new IllegalStateException();
            if (unique && this.collection.contains(o))
                throw new IllegalStateException("Object already contained in unique index.");
            try {
                Entry e = new Entry(this.key, o);
                ((ListIterator)innerIt).set(e);
                txlog.push( new CompensatingTransaction.RemoveOrderedCTx(this.key,this.last.getValue(),this.position));
                txlog.push( new CompensatingTransaction.AddOrderedCTx(this.key, o, this.position));
            } catch (StorageException se) {
                throw new DebugException();
            }
        }
        
    }
    
    
    public TransientMultivaluedOrderedIndex(MdrStorage storage, String name, Storage.EntryType keyType, Storage.EntryType valueType, boolean unique) {
        super(storage, name, keyType, valueType, unique);
    }
    
    /** Inserts the specified element at the specified position in the list of values
     * associated with the specified key.
     * Throws StorageBadRequestException if the index is out of range.
     * @param key
     * @param index
     * @param value
     */
    public void add(Object key, int index, Object value) throws StorageException {
        if (this.map == null) {
            this.map = new HashMap();
        }
        // Never expungeStaleEntries here, changes index
        List c = (List) this.map.get(key);
        if (c == null) {
            c = new ArrayList();
            this.map.put(key, c);
        }
        this.addToList(c, index, key, value);
        this.txlog.push(new CompensatingTransaction.AddOrderedCTx(key, value, index));
    }
    
    protected SlotCollection createSlotCollection (Object key, Collection c) {
        return new OrderedSlotCollection (key, c);
    }
    
    protected SlotCollection createSlotCollection (Object key, Collection c, SinglevaluedIndex repos) {
        return new OrderedSlotCollection (key, c, repos);
    }
    
    /** Returns a list view of the values assosiated in the index with specified key.
     * Returned collection is read only and may not be modified.
     * If there are no values associated with the key empty collection is returned.
     * @param key
     * @return
     */
    public java.util.List getItemsOrdered(Object key) throws StorageException {
        return (List) this.getItems(key);
    }
    
    /** Like getItemsOrdered, but if the index contains keys, this returns the objects
     * corresponding to the key
     * @return
     * @param key
     * @throws StorageException
     */
    public Collection getObjectsOrdered(Object key, SinglevaluedIndex repos) throws StorageException {
        return this.getObjects(key, repos);
    }
    
    /** Removes the element at the specified position in the list of values
     * associated with the specified key.
     * @return true if this index changed as a result of this call
     * @param key
     * @param index
     */
    public boolean remove(Object key, int index) throws StorageException {
        if (this.map == null)
            return false;
        // Never expungeStaleEntries here, changes index
        List list = (List) this.map.get(key);
        Object result = this.removeFromList(list, index);
        if (result != null)
            this.txlog.push(new CompensatingTransaction.RemoveOrderedCTx(key, result, index));
        return result != null;
    }
    
    /** Replaces the element at the specified position in the list of values
     * associated with the specified key with the specified element.
     * Throws StorageBadRequestException if the index is out of range.
     * @param key
     * @param index
     * @param element
     * @throws StorageException
     */
    public void replace(Object key, int index, Object element) throws StorageException {
        if (this.map == null)
            throw new StorageBadRequestException();
        // Never expungeStaleEntries here, changes index
        List list = (List) this.map.get(key);
        Object result = this.setInList(list, index, key, element);
        if (result != null) {
            this.txlog.push(new CompensatingTransaction.RemoveOrderedCTx(key, result, index));
        }
        this.txlog.push(new CompensatingTransaction.AddOrderedCTx(key, result, index));
    }
    
    protected void addNoTx (Object key, Object value, int index) throws StorageException {
        if (map == null)
            return; // Healing
        List l = (List) this.map.get (key);
        this.addToList (l, index, key, value);
    }
    
    protected Object removeNoTx (Object key, Object value, int index) throws StorageException {
        if (map == null)
            return null; //Healing
        List l = (List) this.map.get (key);
        return this.removeFromList (l, index);
    }
    
    private void addToList(List c, int index, Object key, Object value) throws StorageException {
        if (this.unique) {
            for (Iterator it = c.iterator(); it.hasNext();) {
                Entry e = (Entry) it.next();
                if (value.equals(e.getValue()))
                    throw new StorageBadRequestException("Value: "+value+" is already contained.");
            }
        }
        Entry e = new Entry(key, value);
        c.add(index, e);
    }
    
    private Object removeFromList(List list, int index) throws StorageException {
        if (list == null)
            return null;
        Entry e = (Entry) list.remove(index);
        Object value = e.getValue();
        e.dispose();
        return value;
    }
    
    private Object setInList(List list, int index,Object key, Object value) throws StorageException {
        if (list == null)
            throw new StorageBadRequestException();
        Entry newEntry = new Entry(key, value);
        Entry old = (Entry) list.set(index, newEntry);
        if (old != null) {
            old.dispose();
            return old.getValue();
        }
        else
            return null;
    }
    
}
