/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.storagemodel;

import javax.jmi.reflect.*;
import java.util.*;
import java.io.*;

import org.netbeans.mdr.handlers.*;
import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.util.*;
import org.netbeans.mdr.handlers.gen.TagSupport;

/**
 * @author  Petr Hrebejk, Martin Matula
 * @version 0.2
 */
public class StorablePackage extends StorableBaseObject implements Streamable {
    private String context;
    private Map packageProxies;
    private Map assocProxies;
    private Map classProxies;
    private Map datatypes;
    private Class packageSuperclass = null;
    
    /** Replaces MOFIDs using the passed map. (This method is used when rebuilding metaobjects.)
     * @param table Map of old MOFIDs to new MOFIDs.
     */    
    protected void replaceValues(Map table) {
        objectWillChange();
        super.replaceValues(table);
        objectChanged();
    }

    public StorablePackage() {
        super();
    }
    
    public StorablePackage(MdrStorage mdrStorage, org.netbeans.mdr.persistence.MOFID packageId, org.netbeans.mdr.persistence.MOFID metaId,
        String mofContext, Map datatypes, String storageId) throws StorageException {
        super (mdrStorage, packageId, metaId, storageId);
        objectWillChange();
        this.packageProxies = new HashMap();
        this.assocProxies = new HashMap();
        this.classProxies = new HashMap();
        this.datatypes = datatypes;
        this.getMdrStorage().addObject(this);
        if (packageId == null && mofContext != null) {
            this.context = mofContext;
            this.getMdrStorage().createContext(mofContext, this.getMofId());
        }
        else {
            this.context = null;
        }
        this.initFinished = true;
        objectChanged ();
    }
    
    /** Creates new StorablePackage */
    public StorablePackage(MdrStorage mdrStorage, org.netbeans.mdr.persistence.MOFID packageId, org.netbeans.mdr.persistence.MOFID metaId,
        String mofContext, Map datatypes) throws StorageException {
        this(mdrStorage, packageId, metaId, mofContext, datatypes, null);
    }
    
    public synchronized Class getPackageSuperclass() throws StorageException, ClassNotFoundException {
        objectWillChange();
        packageSuperclass = resolvePackageSuperclass();
        objectChanged();

        return packageSuperclass;
    }
    
    private Class resolvePackageSuperclass() throws StorageException, ClassNotFoundException {
        try {
            return BaseObjectHandler.resolveImplementation(TagSupport.getImplFullName(getMetaObject(), TagSupport.PACKAGE));
        } catch (ClassNotFoundException e) {
            return PackageProxyHandler.class;
        }
    }

    public Class getPackageCustomImpl() {
        try {
            Class sup = getPackageSuperclass();
            return sup == PackageProxyHandler.class ? null : sup;
        } catch (Exception e) {}
        return null;
    }

    /**
     * @param pkgMeta
     * @param clustered
     * @throws StorageException  
     */    
    public void clusterPackage(String clusteredMetaName, org.netbeans.mdr.persistence.MOFID clusteredId) throws StorageException {
        addPackage(clusteredMetaName, clusteredId);
    }
    
    public void addPackage(String metaName, org.netbeans.mdr.persistence.MOFID mofId) {
        objectWillChange();
        packageProxies.put(metaName, mofId);
        objectChanged();
    }

    public void addClass(String metaName, org.netbeans.mdr.persistence.MOFID mofId) {
        objectWillChange();
        classProxies.put(metaName, mofId);
        objectChanged();
    }

    public void addAssociation(String metaName, org.netbeans.mdr.persistence.MOFID mofId) {
        objectWillChange();
        assocProxies.put(metaName, mofId);
        objectChanged();
    }

    /**
     * @param meta
     * @throws StorageException
     * @return  
     */    
    public StorablePackage getPackage(String metaName) throws StorageException {
        return (StorablePackage) getMdrStorage().getObject((org.netbeans.mdr.persistence.MOFID) packageProxies.get(metaName));
    }

    /**
     * @param meta
     * @throws StorageException
     * @return  
     */    
    public StorableClass getClassProxy(String metaName) throws StorageException {
        return (StorableClass) getMdrStorage().getObject((org.netbeans.mdr.persistence.MOFID) classProxies.get(metaName));
    }

    /**
     * @param meta
     * @throws StorageException
     * @return  
     */    
    public StorableAssociation getAssociation(String metaName) throws StorageException {
        return (StorableAssociation) getMdrStorage().getObject((org.netbeans.mdr.persistence.MOFID) assocProxies.get(metaName));
    }

    /**
     * @throws StorageException
     * @return  
     */    
    public Collection getAllPackages() throws StorageException {
        return new CachedCollection(getMdrStorage(), packageProxies.values());
    }

    /**
     * @throws StorageException
     * @return  
     */    
    public Collection getAllClasses () throws StorageException {
        return new CachedCollection(getMdrStorage(), classProxies.values());
    }

    /**
     * @throws StorageException
     * @return  
     */    
    public Collection getAllAssociations () throws StorageException {
        return new CachedCollection(getMdrStorage(), assocProxies.values());
    }
    
    public String getContext() {
        return context;
    }
    
    public DatatypeDescriptor getDatatypeDesc(String name) {
        return (DatatypeDescriptor) datatypes.get(name);
    }

    /**
     * @throws StorageException  
     */    
    public void delete() throws StorageException {
        if (getImmediatePackageId() == null) {
            getMdrStorage().dropContext(context, getMofId());
            deleteRecursive();
        } else {
            throw new InvalidCallException(getMdrStorage().getRepository().getHandler(this), null, "Not an outermost package.");
        }
    }
    
    protected void deleteRecursive() throws StorageException {
        StorablePackage sp;

        for (Iterator it = packageProxies.values().iterator(); it.hasNext();) {
            try {
                sp = (StorablePackage) getMdrStorage().getObject((org.netbeans.mdr.persistence.MOFID) it.next());
            } catch (StorageBadRequestException e) {
                sp = null;
            }
            if (sp != null) sp.deleteRecursive();
        }

        for (Iterator it = classProxies.values().iterator(); it.hasNext();) {
            StorableClass sc = (StorableClass) getMdrStorage().getObject((org.netbeans.mdr.persistence.MOFID) it.next());
            sc.deleteRecursive();
        }

        for (Iterator it = assocProxies.values().iterator(); it.hasNext();) {
            StorableAssociation sa = (StorableAssociation) getMdrStorage().getObject((org.netbeans.mdr.persistence.MOFID) it.next());
            sa.deleteRecursive();
        }
        
        packageProxies.clear();
        classProxies.clear();
        assocProxies.clear();
        
        super.deleteRecursive();
    }
    
    /**
     * @param outputStream 
     */    
    public void write(java.io.OutputStream outputStream) {
//        if (packageProxies == null || classProxies == null || assocProxies == null) throw new NullPointerException();
        
        super.write (outputStream);
        
        try {
            IOUtils.write (outputStream, meta, this);
            IOUtils.write (outputStream, immediatePackage, this);
            IOUtils.writeString(outputStream, context);
            IOUtils.write(outputStream, packageProxies, this);
            IOUtils.write(outputStream, classProxies, this);
            IOUtils.write(outputStream, assocProxies, this);
            
            IOUtils.writeInt(outputStream, datatypes.size());
            for (Iterator it = datatypes.keySet().iterator(); it.hasNext();) {
                String name = (String) it.next();
                IOUtils.writeString(outputStream, name);
                ((DatatypeDescriptor) datatypes.get(name)).write(outputStream);
            }
        } catch (java.io.IOException e) {
            Logger.getDefault().notify(Logger.INFORMATIONAL, e);
        }
    }

    /**
     * @param inputStream  
     */    
    public void read(java.io.InputStream inputStream) {
        super.read (inputStream);
        
        try {
            meta = (org.netbeans.mdr.persistence.MOFID) IOUtils.read (inputStream, this);
            immediatePackage = (org.netbeans.mdr.persistence.MOFID) IOUtils.read (inputStream, this);
            context = IOUtils.readString(inputStream);
            packageProxies = (Map) IOUtils.read(inputStream, this);
            classProxies = (Map) IOUtils.read(inputStream, this);
            assocProxies = (Map) IOUtils.read(inputStream, this);

            int count = IOUtils.readInt(inputStream);
            datatypes = new HashMap(count, 1);
            for (int i = 0; i < count; i++) {
                String name = IOUtils.readString(inputStream);
                datatypes.put(name, DatatypeDescriptor.readResolve(inputStream, this));
            }
        } catch (IOException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
}
