/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.storagemodel;

import org.netbeans.mdr.handlers.AssociationHandler;
import org.netbeans.mdr.handlers.BaseObjectHandler;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.mdr.persistence.MultivaluedIndex;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.persistence.Streamable;
import org.netbeans.mdr.util.DebugException;
import org.netbeans.mdr.util.IOUtils;
import org.netbeans.mdr.util.Logger;
import javax.jmi.reflect.JmiException;
import javax.jmi.reflect.RefFeatured;
import javax.jmi.reflect.RefObject;
import javax.jmi.reflect.WrongSizeException;
import java.util.*;

/**
 *
 * @author  Martin Matula, Pavel Buzek
 * @version 0.2
 */
public class StorableObject extends StorableFeatured implements Streamable {

    /** For each attr holds the value.
     */
    protected Object values[];
    
    static final String INDEX_KEY_DELIMITER = "@";
    static final String INDEX_KEY_DELIMITER_2 = "#";
    static final String NULL_VALUE_SUBSTITUTE = "NULL";
    
    private MOFID classProxyId;
    private transient StorableClass classProxy = null;
    /** MOFID of this object's immediate composite in case when the object is a value
     * of an attribute, otherwise null. */
    private MOFID attribComposite = null;
    
    protected void replaceValues(Map table) {
        objectWillChange();
        super.replaceValues(table);
        objectChanged();
    }
    
    /** Creates new StorableObject
     * Called during boot sequence only.
     */
    public StorableObject(MdrStorage mdrStorage, org.netbeans.mdr.persistence.MOFID immediatePackage, org.netbeans.mdr.persistence.MOFID meta, org.netbeans.mdr.persistence.MOFID classProxy) throws StorageException {
        this(mdrStorage, immediatePackage, meta, classProxy, null, null);
    }
    
    public StorableObject (MdrStorage mdrStorage, org.netbeans.mdr.persistence.MOFID immediatePackage, org.netbeans.mdr.persistence.MOFID meta, org.netbeans.mdr.persistence.MOFID classProxy, Object params[]) throws StorageException {
        this (mdrStorage, immediatePackage, meta, classProxy, params, null);
    }
    
    /** Creates new StorableObject */
    public StorableObject(MdrStorage mdrStorage, org.netbeans.mdr.persistence.MOFID immediatePackage, org.netbeans.mdr.persistence.MOFID meta, org.netbeans.mdr.persistence.MOFID classProxy, Object params[], String storageId) throws StorageException {
        super(mdrStorage, immediatePackage, meta, storageId);
        this.classProxyId = classProxy;
        this.classProxy = (StorableClass) mdrStorage.getObject(classProxyId);
        
        if (params != null) {
            check();
            for (int i = 0; i < params.length; i++) {
                StorableClass.AttributeDescriptor desc = getClassProxy().getAttrDesc(i);
                values[i] = getInitialValue(desc, params[i]);
                // modifyIndex(i, null, this.values[i]);
            } // for
        } // if
        this.addInstance ();
        if (params != null) {
            modifyIndex (getClassProxy().getIndexDescriptors(), false);
        }
        initFinished = true;
    }

    public StorableObject(StorableObject storable) throws StorageException {
        super(storable.getMdrStorage(), storable.getImmediatePackageId(), storable.getMetaObjectId(), null);
        this.classProxyId = storable.getClassProxyId();
        this.classProxy = storable.getClassProxy();
        copyValues(storable);
        this.addInstance();
        modifyIndex(getClassProxy().getIndexDescriptors(), false);
        StorableFeatured composite = storable.getImmediateComposite();
        if (composite != null) {
            attribComposite = composite.getMofId();
        }
        initFinished = true;
    }

    public StorableObject() {
        super();
    }

    protected void copyValues(StorableObject storable) {
        if (storable.values == null) return;
        this.values = new Object[storable.values.length];
        for (int i = 0; i < values.length; i++) {
            values[i] = storable.values[i];
            if (values[i] instanceof AttrCollection) {
                ((AttrCollection) values[i]).mdrObject = this;
            }
        }
    }

    public void setAttribute(String featureName, Object value) throws StorageException {
        //if (value == null) Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException());
        setAttribute(getClassProxy().getAttrIndex(featureName), value);
    }

    public void setAttribute(int attrIndex, Object value) throws StorageException {
        Object oldValue = getAttribute(attrIndex);
        StorableClass.AttributeDescriptor attribute = getClassProxy().getAttrDesc(attrIndex);

//        if (value == oldValue || (value != null && value.equals(oldValue))) {
//            return;
//        }

        objectWillChange();
        values[attrIndex] = value;
        
        if (!attribute.isMultivalued() && (value instanceof RefObject)) {
            StorableObject storableObj = (StorableObject) ((BaseObjectHandler) value)._getDelegate();
            storableObj.setComposite(getMofId(), storableObj.getMofId(), attribute.getMofId());
            if (oldValue != null) {
                storableObj = (StorableObject) ((BaseObjectHandler) oldValue)._getDelegate();
                storableObj.clearComposite();
            }
        }
        
        modifyIndex(attrIndex, oldValue, value); // hold index only in case of singlevalued attributes

        objectChanged();
    }

    public Object getAttribute(String featureName) throws StorageException {
        return getAttribute(getClassProxy().getAttrIndex(featureName));
    }
    
    public Object getAttribute(int attrIndex) throws StorageException {
        check();
        Object result = values[attrIndex];
        if (result instanceof MOFID) {
            result = getMdrStorage().getRepository().getByMofId((MOFID) result);
        }
        return result;
    }
    
    public void verify(Collection violations) throws StorageException {
        int count = getClassProxy().getAttrCount();
        for (int i = 0; i < count; i++) {
            StorableClass.AttributeDescriptor desc = getClassProxy().getAttrDesc(i);
            Object result = getAttribute(i);
            if ((desc.getMaxSize() == 1 && desc.getMinSize() == 1 && result == null) ||
                (desc.getMaxSize() != 1 && ((Collection) result).size() < desc.getMinSize()))
                    violations.add(new WrongSizeException(
                        (RefObject) getMdrStorage().getRepository().getHandler(getMdrStorage().getObject(desc.getMofId()))
                    ));
        }
        for (Iterator it = getClassProxy().getAllReferenceDescriptors().iterator(); it.hasNext();) {
            StorableClass.ReferenceDescriptor desc = (StorableClass.ReferenceDescriptor) it.next();
            desc.getAssociation().verifyEnd(violations, desc.getEndName(), getMofId());
        }
    }

    /** Adds a new reference.
     * This method is called only during boot sequence, thus it does not need to be synchronized.
     * @param referencedObjectId
     * @throws StorageException  
     */
    void addReference(String referenceName, org.netbeans.mdr.persistence.MOFID referencedObjectId) throws StorageException {
        org.netbeans.mdr.persistence.MOFID objA;
        org.netbeans.mdr.persistence.MOFID objB;
        StorableClass.ReferenceDescriptor reference = getClassProxy().getReferenceDescriptor(referenceName);
        StorableAssociation assocObj = (StorableAssociation) getMdrStorage().getObject(reference.getAssociationId());

        if (reference.getEndName().equals(assocObj.getEnd1Name())) {
            objB = referencedObjectId;
            objA = getMofId();
        } else {
            objA = referencedObjectId;
            objB = getMofId();
        }
        
        assocObj.addLink(objA, objB);
    }
    
    /** This method is only using for booting and for querying the MOF layer.
     * @return 
     * @throws StorageException  */
    public Object getReference(String referenceName) throws StorageException {
        StorableAssociation asocObj1;
        StorableClass.ReferenceDescriptor reference = getClassProxy().getReferenceDescriptor(referenceName);
        asocObj1 = (StorableAssociation) getMdrStorage().getObject(reference.getAssociationId());
        return asocObj1.queryObjects(reference.getEndName(), getMofId());
    }
    
    protected void check() throws StorageException {
        if (values == null) {
            values = new Object[getClassProxy().getAttrCount()];
        }
    }
    
    public MOFID getImmediateCompositeId() {
        return attribComposite;
    }

    /** Returns mofid of the immediate object in which the this object is contained.
     * @return result of this method is valid only if the immediate coposite was previously set
     * @throws StorageException
     */
    public StorableFeatured getImmediateComposite() throws StorageException {
        if (attribComposite != null) {
            StorableFeatured sf = (StorableFeatured) getMdrStorage().getObject(attribComposite);
            return sf;
        } else {
            return null;
        }
    }

    public StorableFeatured getOutermostComposite() throws StorageException {
        StorableFeatured result = getImmediateComposite();
        StorableFeatured composite = this;

        while (result != null) {
            composite = result;
            if (result instanceof StorableObject) {
                result = ((StorableObject) result).getImmediateComposite();
            } else {
                result = null;
            }
        }

        return composite;
    }
    
    public void clearComposite() throws StorageException {
        setComposite((org.netbeans.mdr.persistence.MOFID) null, null, null);
    }
    
    public void setComposite(org.netbeans.mdr.persistence.MOFID composite, org.netbeans.mdr.persistence.MOFID objectId, org.netbeans.mdr.persistence.MOFID elementId) throws StorageException {
        setComposite(getMdrStorage().getObject(composite), objectId, elementId);
    }
    
    /**
     * Sets (resp. unsets) immediate composite in case the object is (resp. was)
     * value of an attribute.
     *
     */
    public void setComposite(StorableBaseObject composite, org.netbeans.mdr.persistence.MOFID objectId, org.netbeans.mdr.persistence.MOFID elementId) throws StorageException {
        objectWillChange();
        if (composite == null) {
            attribComposite = null;
        } else {
            org.netbeans.mdr.persistence.MOFID id = composite.getMofId();
            if (!id.equals(attribComposite)) {
                // check for Composition Violation
                if (attribComposite != null) {
                    throw new javax.jmi.reflect.CompositionViolationException(getMdrStorage().getRepository().getHandler(this), (RefObject) getMdrStorage().getRepository().getHandler(getMdrStorage().getObject(elementId)));
                }
                // check for Composition Cycle
                if ((composite instanceof StorableObject) && (((StorableObject) composite).getOutermostComposite().equals(this))) {
                    throw new javax.jmi.reflect.CompositionCycleException(getMdrStorage().getRepository().getHandler(getMdrStorage().getObject(objectId)), (RefObject) getMdrStorage().getRepository().getHandler(getMdrStorage().getObject(elementId)));
                }
                // check for Composition Closure
                if (!composite.getOutermostPackageId().equals(getOutermostPackageId())) {
                    throw new javax.jmi.reflect.ClosureViolationException(getMdrStorage().getRepository().getHandler(getMdrStorage().getObject(objectId)), (RefObject) getMdrStorage().getRepository().getHandler(getMdrStorage().getObject(elementId)));
                }
                attribComposite = id;
            }
        }
        objectChanged ();
    }
    
    public void deleteInstance() throws StorageException {
        modifyIndex (getClassProxy().getIndexDescriptors(), true);
        getMdrStorage().removeInstance(this);
    }
    
    /**
     * @throws StorageException
     */
    public void delete() throws StorageException {
            deleteAttributes();
            deleteLinksAndComponents();
            deleteFromComposite();
            deleteInstance();
    }
    
    protected void deleteFromComposite() throws StorageException {
        StorableFeatured composite = getImmediateComposite();
        if (composite != null) {
            StorableClass cls = composite.getClassProxy();
            for (int i = 0; i < cls.getAttrCount(); i++) {
                StorableClass.AttributeDescriptor desc = cls.getAttrDesc(i);
                if (RefObject.class.isAssignableFrom(desc.getType())) {
                    RefFeatured obj = (RefFeatured) getMdrStorage().getRepository().getHandler(composite);
                    Object value = obj.refGetValue(desc.getName());
                    if (this.equals(value)) {
                        obj.refSetValue(desc.getName(), null);
                    } else if (value instanceof Collection && ((Collection) value).contains(this)) {
                        ((Collection) value).remove(this);
                    }
                }
            }
        }
    }
    
    protected void deleteLinksAndComponents() throws StorageException {
        RefObject thisObject = (RefObject) getMdrStorage().getRepository().getHandler(this);
        ArrayList associationEnds = new ArrayList();
        getClassProxy().collectAssociationEnds(associationEnds, new HashSet());
        for (Iterator it = associationEnds.iterator(); it.hasNext();) {
            StorableClass.AssocEndDescriptor item = (StorableClass.AssocEndDescriptor) it.next();
            StorableAssociation assocStorable = (StorableAssociation) getMdrStorage().getObject(item.mofId);
            if (assocStorable == null) continue;
            AssociationHandler assoc = (AssociationHandler) getMdrStorage().getRepository().getHandler(assocStorable);
            Object temp;
            try {
                temp = assoc._query(item.endName, thisObject);
            } catch (ClassCastException e) {
                temp = null;
            } catch (JmiException e) {
                temp = assoc._handleQuery(item.endName, thisObject);
            }
            if (temp != null) {
                if (temp instanceof Collection) {
                    for (Iterator itt = ((Collection)temp).iterator(); itt.hasNext();) {
                        RefObject obj = (RefObject) itt.next();
                        itt.remove();
                        if (item.isAggregate)
                            obj.refDelete();
                    }
                } else {
                    if (item.endName.equals(assocStorable.getEnd2Name())) {
                        assoc.refRemoveLink((RefObject) temp, thisObject);
                    } else {
                        assoc.refRemoveLink(thisObject, (RefObject) temp);
                    if (item.isAggregate)
                        ((RefObject) temp).refDelete();
                    }
                }
            }
        }
    }

    /**
     * @return
     */
    public org.netbeans.mdr.persistence.MOFID getClassProxyId() {
        return classProxyId;
    }
    
    /**
     * @throws StorageException
     * @return
     */
    public StorableClass getClassProxy() throws StorageException {
        return classProxy;
    }
    
    /**
     * @param outputStream
     */
    public void write(java.io.OutputStream outputStream) {
        super.write(outputStream);
        try {
            IOUtils.write (outputStream, classProxyId, this);
            IOUtils.write (outputStream, attribComposite, this);
            if (values == null) {
                IOUtils.writeInt(outputStream, 0);
            } else {
                IOUtils.writeInt(outputStream, values.length + 1);
                for (int i = 0; i < values.length; i++) {
                    IOUtils.write(outputStream, values[i], this);
                }
            }
        } catch (java.io.IOException e) {
            Logger.getDefault().notify(Logger.INFORMATIONAL, e);
        }
    }
    
    /**
     * @param inputStream
     */
    public void read(java.io.InputStream inputStream) {
        super.read(inputStream);
        try {
            classProxyId = (org.netbeans.mdr.persistence.MOFID) IOUtils.read (inputStream, this);
            attribComposite = (org.netbeans.mdr.persistence.MOFID) IOUtils.read (inputStream, this);
            // init value of "meta" (not serialized in case of StorableObject)
        } catch (java.io.IOException e) {
            throw (RuntimeException) Logger.getDefault().annotate(new RuntimeException(e.getMessage()), e);
        }
        // init value of "meta" (not serialized in case of StorableObject)
        try {
            classProxy = (StorableClass) getMdrStorage().getObject(classProxyId);
            meta = classProxy.getMetaObjectId();
            immediatePackage = classProxy.getImmediatePackageId();
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }

        try {
            int objCount = IOUtils.readInt(inputStream);
            if (objCount != 0) {
                int count = objCount - 1;
                values = new Object[count];
                for (int i = 0; i < count; i++) {
                    values[i] = IOUtils.read(inputStream, this, getClassProxy().getAttrDesc(i).getType().getName());
                    //Logger.getDefault().log(getClassProxy().getAttrDesc(i).getName() + " = " + values[i]);
                }
            }
        } catch (Exception e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    
    protected void addInstance () throws StorageException {
        this.getMdrStorage().addInstance(this);
    }
    
    /**
     * @param oldValue
     * @param newValue
     * @throws StorageException
     */
    protected void modifyIndex(int attrIdx, Object oldValue, Object newValue) throws StorageException {
        MultivaluedIndex attrIndex = null;
        StorableClass proxy = getClassProxy();
        StorableClass.AttributeDescriptor attrDesc = proxy.getAttrDesc(attrIdx);
        org.netbeans.mdr.persistence.MOFID attribId = attrDesc.getMofId();
        Collection indexes;
        if (attrDesc.isIndexed() && (indexes = proxy.getIndexes(attribId)) != null) {
            
            Iterator iter = indexes.iterator();
            org.netbeans.mdr.persistence.MOFID outermostPackageId = getOutermostPackageId();
            org.netbeans.mdr.persistence.MOFID mofId = getMofId();
            
            while (iter.hasNext()) {
                StorableClass.IndexDescriptor indexDesc = (StorableClass.IndexDescriptor) iter.next();
                try {
                    attrIndex = getMdrStorage().acquireAdditionalIndex(outermostPackageId, indexDesc.getName());

                    if (attrIndex == null) {
                        return;
                    }
                    
                    if ((oldValue == null) && (newValue == null)) {
                        return;
                    }
                    if ((oldValue != null) && oldValue.equals(newValue)) {
                        return;
                    }
                    
                    StorableClass.IndexDescriptor.Field [] fields = indexDesc.getFields ();
                    int currPosition = -1;
                    for (int x = 0; x < fields.length; x++) {
                        if (fields[x] instanceof StorableClass.IndexDescriptor.Attrib) {
                            if (fields[x].getId().equals(attribId)) {
                                currPosition = x;
                                break;
                            }
                        }
                    }
                    
                    String oldValStr = objectToString(oldValue, fields[currPosition].isOrdered());
                    String newValStr = objectToString(newValue, fields[currPosition].isOrdered());
                    String prefix = valuesToKey(fields, 0, currPosition - 1);
                    String suffix = valuesToKey(fields, currPosition + 1, fields.length - 1);
                    
                    StringBuffer buf = new StringBuffer();
                    if (prefix.length() > 0) {
                        buf.append(prefix);
                        buf.append(INDEX_KEY_DELIMITER);
                    }
                    buf.append(oldValStr);
                    if (suffix.length() > 0) {
                        buf.append(INDEX_KEY_DELIMITER);
                        buf.append(suffix);
                    }
                    attrIndex.remove(buf.toString(), mofId);
                    //Logger.getDefault().log("remove: " + buf.toString ());
                    
                    buf = new StringBuffer();
                    if (prefix.length() > 0) {
                        buf.append(prefix);
                        buf.append(INDEX_KEY_DELIMITER);
                    }
                    buf.append(newValStr);
                    if (suffix.length() > 0) {
                        buf.append(INDEX_KEY_DELIMITER);
                        buf.append(suffix);
                    }
                    attrIndex.add(buf.toString(), mofId);
                    //Logger.getDefault().log("add: " + buf.toString ());
                    
                } finally {
                    if (attrIndex != null) getMdrStorage().releaseAdditionalIndex();
                }
            } // while
            
        } // if
    }

    /**
     * @throws StorageException  
     */    
    protected void deleteAttributes() throws StorageException {
        for (int i = 0; i < values.length; i++) {
            //modifyIndex(i, values[i], null);
            Object value = getAttribute(i);
            if (value instanceof RefObject) {
                RefObject obj = (RefObject) value;
                setAttribute(i, null);
                obj.refDelete();
            } else if (value instanceof Collection) {
                for (Iterator it = ((Collection)value).iterator(); it.hasNext();) {
                    Object obj = it.next();
                    if (!(obj instanceof RefObject))
                        break;
                    it.remove();
                    ((RefObject)obj).refDelete();
                }
            }
        }
    }

    void addToIndex (org.netbeans.mdr.persistence.MOFID id) {
        try {
            StorableClass proxy = getClassProxy();        
            modifyIndex (proxy.getIndexes(id), false);
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }

    void removeFromIndex (org.netbeans.mdr.persistence.MOFID id) {
        try {
            StorableClass proxy = getClassProxy();        
            modifyIndex (proxy.getIndexes(id), true);
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    private void modifyIndex(List descriptors, boolean remove) throws StorageException {
        if (descriptors == null)
            return;
        MultivaluedIndex attrIndex = null;
        org.netbeans.mdr.persistence.MOFID outermostPackageId = getOutermostPackageId();
        org.netbeans.mdr.persistence.MOFID mofId = getMofId();
        Iterator iter = descriptors.iterator();
        while (iter.hasNext()) {
            StorableClass.IndexDescriptor desc = (StorableClass.IndexDescriptor) iter.next();
            StorableClass.IndexDescriptor.Field [] fields = desc.getFields();
            try {
                attrIndex = getMdrStorage().acquireAdditionalIndex(outermostPackageId, desc.getName());
                if (remove) {
                    //Logger.getDefault().log("Remove: " + valuesToKey (fields, 0, fields.length - 1));
                    attrIndex.remove(valuesToKey(fields, 0, fields.length - 1), mofId);
                } else {
                    //Logger.getDefault().log("Add: " + valuesToKey (fields, 0, fields.length - 1));
                    attrIndex.add(valuesToKey(fields, 0, fields.length - 1), mofId);
                }
            } finally {
                if (attrIndex != null) getMdrStorage().releaseAdditionalIndex();
            }
        } // while
    }
    
    private static String objectToString(Object o, boolean isOrdered) {
        if (o == null)
            return NULL_VALUE_SUBSTITUTE;
        if (o instanceof RefObject)
            return ((RefObject) o).refMofId ();
        if (o instanceof StorableObject)
            return ((StorableObject) o).getMofId().toString ();
        if (o instanceof Collection) {
            StringBuffer buf = new StringBuffer();
            Iterator iter = ((Collection) o).iterator();
            List list = new LinkedList ();
            while (iter.hasNext ()) {
                list.add (objectToString(iter.next(), true));
            }            
            if (!isOrdered) {
                Collections.sort (list);
            }
            iter = list.iterator ();
            if (iter.hasNext())
                buf.append(iter.next());
            while (iter.hasNext()) {
                buf.append(INDEX_KEY_DELIMITER);
                buf.append(iter.next());
            } // while
            return buf.toString();            
        } // if
        return o.toString();
    }
    
    private String valuesToKey (StorableClass.IndexDescriptor.Field [] fields, int pos_1, int pos_2) throws StorageException {
        StringBuffer buf = new StringBuffer ();
        for (int x = pos_1; x <= pos_2; x++) {
            String res;
            if (fields[x] instanceof StorableClass.IndexDescriptor.Attrib) {
                String name = fields[x].getName();
                res = objectToString(values[getClassProxy().getAttrIndex(name)], fields[x].isOrdered());
                //Logger.getDefault().log("valuesToKey, Attrib: " + res);
            } else {
                StorableClass.IndexDescriptor.AssocEnd endDesc = (StorableClass.IndexDescriptor.AssocEnd)fields[x];
                StorableAssociation sa = (StorableAssociation) getMdrStorage().getObject(endDesc.getAssociation());
                String otherEndName  = sa.getEnd1Name ();
                if (otherEndName.equals (endDesc.getName()))
                    otherEndName = sa.getEnd2Name ();
                // [PENDING] rewrite this so that also pre/post processed and derived associations work
                // (instead of calling queryObjects method, _query method on
                // AssociationHandler should be called)
                res = objectToString(sa.queryObjects(otherEndName, getMofId()), fields[x].isOrdered());
                //Logger.getDefault().log("valuesToKey, AssocEnd: " + res);
            }
            buf.append(res);
            if (x < pos_2)
                buf.append(INDEX_KEY_DELIMITER_2);
        }
        return buf.toString();
    }

    public static String valuesToKey (Map map, StorableClass.IndexDescriptor.Field [] fields) {
        StringBuffer buf = new StringBuffer ();
        if (map.size () != fields.length) {
            throw new DebugException ("Wrong query on additional index, incorrect number of passed parameters.");
        }
        for (int x = 0; x < fields.length; x++) {
            String name = fields[x].getName ();
            if (!map.containsKey (name)) {
                throw new DebugException ("Wrong query on additional index, value of field " + name + " not specified.");
            }
            Object value = map.get (name);            
            String str = objectToString (value, fields[x].isOrdered());            
            buf.append(str);
            if (x < fields.length - 1)
                buf.append(INDEX_KEY_DELIMITER_2);
        } // for
        return buf.toString ();        
    }
    
    public static String valueToKey (Object value, StorableClass.IndexDescriptor.Field [] fields) {
        if (fields.length != 1) {
            throw new DebugException ("Wrong query on additional index, more than one parameter expected.");
        }
        return objectToString (value, fields[0].isOrdered());
    }
    
    public byte[] getClassFile() {
        return (byte[]) getSlot1();
    }
    
    public void setClassFile(byte[] bytecode) {
        setSlot1(bytecode);
    }
    
    public byte[] getInstanceClassFile() {
        return (byte[]) getSlot2();
    }
    
    public void setInstanceClassFile(byte[] bytecode) {
        setSlot2(bytecode);
    }
    
}
