/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.storagemodel;

import java.util.*;
import java.io.IOException;

import javax.jmi.model.*;
import javax.jmi.reflect.*;

import org.netbeans.mdr.util.*;
import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.handlers.*;
import org.netbeans.mdr.handlers.gen.TagSupport;

/** Class representing a class proxy.
 *
 * @author Martin Matula, Petr Hrebejk
 * @version 0.2
 */
public class StorableClass extends StorableFeatured {
    
    private AttributeDescriptor attrDescs[];
    private Map clAttrDescs;
    private Map clAttrValues;
    private Map datatypes;
    private List superclasses;
    private List subclasses;
    private Map references;
    private Collection associationEnds;
    private boolean classDerived;
    private boolean instanceDerived;
    private String classSuperclass = null;
    private String instanceSuperclass = null;
    private boolean singletonClass;
    private boolean abstractClass;
    private IndexDescriptor[] indexDescs;

    // transient variables
    private final Hashtable refCache = new Hashtable();

    /**
     * Names of all instance-level attributes of the class corresponding to 
     * this classproxy collected from
     * the metaclass and its superclasses. Order of the names match exactly
     * order of attribute descriptors in the attributeDecs array and the
     * array of attr values in instances.
     */
    private List attributes;
    
    /**
     * Collected attribute descriptors from this classproxy's metaclass and
     * its superclasses. They're stored as array because of convenient fast access
     * by index.
     */
    private AttributeDescriptor[] attributeDescs;

    /**
     * Collected additional index descriptors from this classproxy's metaclass and
     * its superclasses.
     */
    private List indexDescriptors;

    /**
     * Maps attribute's ant assoc. end's ids to list of related additional indexes.
     */
    private HashMap indexMap;
    
    /**
     * Map of all additional indexes by name.
     */
    private HashMap indexesByName;
    
    /** Replaces MOFIDs using the passed map. (This method is used when rebuilding metaobjects.)
     * @param table Map of old MOFIDs to new MOFIDs.
     */
    protected void replaceValues(Map table) {
        objectWillChange();
        super.replaceValues(table);

        // rebuild all attributes
        for (int i = 0; i < attrDescs.length; i++) {
            attrDescs[i].replaceValues(table);
        }

        if (clAttrDescs != null) {
            for (Iterator it = clAttrDescs.values().iterator(); it.hasNext();) {
                ((AttributeDescriptor) it.next()).replaceValues(table);
            }
        }

        // rebuild all references
        for (Iterator it = references.values().iterator(); it.hasNext();) {
            ((ReferenceDescriptor) it.next()).replaceValues(table);
        }
        objectChanged();
    }

    /** Creates a new StorableClass.
     * This default constructor is to be called only when deserializing class proxy.
     * <code>read</code> method is called immediately after this constructor.
     */
    public StorableClass() {
        super();
    }

    /** Creates new StorableClass providing all needed arguments.
     * @param mdrStorage parent storage
     * @param packageId MOFID of immediate package
     * @param metaId MOFID of metaobject
     * @param attrDescriptors mapping from attribute names to attribute mofids
     * @param clAttrDescriptors mapping from attribute names to attribute mofids (for classifier-level attributes)
     * @throws StorageException problem in storage
     */
    public StorableClass(MdrStorage mdrStorage, org.netbeans.mdr.persistence.MOFID packageId, org.netbeans.mdr.persistence.MOFID metaId,
        List attrDescs, List clAttrDescs, Map datatypes, boolean classDerived,
        boolean instanceDerived, boolean isSingleton, boolean isAbstract) throws StorageException {

        super(mdrStorage, packageId, metaId);

        this.attrDescs = (AttributeDescriptor[]) attrDescs.toArray(new AttributeDescriptor[attrDescs.size()]);
        
        int size = clAttrDescs.size();
        if (size > 0) {
            this.clAttrDescs = new HashMap(size * 2);
            this.clAttrValues = new HashMap(size * 2);
            for (Iterator it = clAttrDescs.iterator(); it.hasNext();) {
                AttributeDescriptor desc = (AttributeDescriptor) it.next();
                this.clAttrDescs.put(desc.getName(), desc);
                this.clAttrValues.put(desc.getName(), getInitialValue(desc, null));
            }
        }

        this.datatypes = datatypes;
        subclasses = new ArrayList();
        superclasses = new ArrayList();
        references = new HashMap();
        associationEnds = new ArrayList();

        this.classDerived = classDerived;
        this.instanceDerived = instanceDerived;
        this.singletonClass = isSingleton;
        this.abstractClass = isAbstract;
        
        getMdrStorage().addObject(this);
        initFinished = true;
    }

    public void buildAdditionalIndexes(List indexTags, Map associationProxies) throws StorageException {
        objectWillChange();
        indexDescs = new IndexDescriptor [indexTags.size ()];
        Iterator iter = indexTags.iterator ();
        for (int y = 0; iter.hasNext (); y++) {
            Tag tag = (Tag) iter.next ();
            List values = tag.getValues ();
            if ((values == null) || (values.size () == 0)) {
                throw new DebugException ("Cannot create unnamed additional index.");
            }
            String indexName = (String) values.get (0);
            List elements = new ArrayList (tag.getElements ());
            Iterator elementsIter = elements.iterator ();            
            List collectedFields = new ArrayList (elements.size ());
            
            while (elementsIter.hasNext()) {
                ModelElement elem = (ModelElement) elementsIter.next ();
                if (((org.netbeans.mdr.handlers.InstanceHandler)elem)._getDelegate().getMofId().equals (getMetaObjectId()))
                    continue;                                                                
                if (elem instanceof Attribute) {                   
                    collectedFields.add (new IndexDescriptor.Attrib (((org.netbeans.mdr.handlers.InstanceHandler)elem)._getDelegate().getMofId(), elem.getName (), ((Attribute)elem).getMultiplicity().isOrdered()));
                } else if (elem instanceof AssociationEnd) {
                    org.netbeans.mdr.persistence.MOFID assocId = (org.netbeans.mdr.persistence.MOFID) associationProxies.get (((org.netbeans.mdr.handlers.InstanceHandler)elem.getContainer ())._getDelegate().getMofId());
                    collectedFields.add (new IndexDescriptor.AssocEnd (((org.netbeans.mdr.handlers.InstanceHandler)elem)._getDelegate().getMofId(), elem.getName(), ((AssociationEnd)elem).getMultiplicity().isOrdered(), assocId));
                } else {
                    throw new DebugException ("Cannot create additional index " + indexName + ", element " + elem.getName() + " is not an attribute or an association end.");
                }
            } // for
            IndexDescriptor indexDesc = new IndexDescriptor (indexName, (IndexDescriptor.Field []) collectedFields.toArray (new IndexDescriptor.Field [collectedFields.size()]));
            indexDescs [y] = indexDesc;
            getMdrStorage().createAdditionalIndex(getOutermostPackageId(), indexName, getMofId ());
        } // for
        objectChanged ();
    }

    /**
     * Creates map storing all additional indexes related to attribute's or assoc. end's id.
     */
    private void buildIndexMap () throws StorageException {
        indexMap = new HashMap();
        indexesByName = new HashMap();
        Iterator iter = indexDescriptors.iterator();
        while (iter.hasNext()) {
            IndexDescriptor desc = (IndexDescriptor) iter.next();
            indexesByName.put (desc.getName(), desc);
            Object [] fields = desc.getFields();
            for (int x = 0; x < fields.length; x++) {
                org.netbeans.mdr.persistence.MOFID id;
                if (fields[x] instanceof IndexDescriptor.Attrib) {
                    id = ((IndexDescriptor.Attrib) fields[x]).getId();
                    String name = ((IndexDescriptor.Attrib) fields[x]).getName();
                    attributeDescs [getAttrIndex(name)].setIndexed(true);
                } else
                    id = ((IndexDescriptor.AssocEnd) fields[x]).getId();
                List list = (List) indexMap.get(id);
                if (list == null)
                    indexMap.put(id, list = new LinkedList());
                list.add(desc);
            } // for
        } // while        
    }

    List getIndexes(org.netbeans.mdr.persistence.MOFID mofId) throws StorageException {
        checkAttributes();
        return (List) indexMap.get(mofId);
    }
    
    public IndexDescriptor getAdditionalIndex(String indexName) throws StorageException {
        checkAttributes();
        return (IndexDescriptor) indexesByName.get (indexName);
    }

    public DatatypeDescriptor getDatatypeDesc(String name) {
        return (DatatypeDescriptor) datatypes.get(name);
    }

    /** Returns immutable, live collection of all instances of MOF class
     * represented by this proxy.
     *
     * @param subclasses if <code>true</code> also the instances of sub-classes
     *     will be returned, otherwise only the instances of the current class
     * @return collection of instances of this class (and its sub-classes)
     * @throws StorageException problem in storage
     */
    public Collection allObjects(boolean subclasses) throws StorageException {
        if (subclasses) {
            CompositeCollection result = new CompositeCollection();
            collectObjects(result, new HashSet());
            return result;
        } else {
            MdrStorage storage = getMdrStorage();
            org.netbeans.mdr.persistence.MOFID mofId = getMofId();
            return new IndexImmutSet(storage, storage.getInstancesIndex(mofId), mofId);
        }
    }

    /**
     *  Adds the instances of this class and its super-classes to <code>result</code>
     *  and adds the MOF IDs of all visited class proxies to <code>visited</code>.
     *  Instances of a class already being a member of <code>visited</code> are
     *  not added to <code>result</code> a second time. 
     *
     * @param result the collection where to add the instances
     * @param visited MOF IDs of class proxies already visited
     */
    private void collectObjects(CompositeCollection result, Set visited) throws StorageException {
        if (visited.add(getMofId())) {
            result.addCollection(allObjects(false));
            for (Iterator it = subclasses.iterator(); it.hasNext();) {
                ((StorableClass) getMdrStorage().getObject((org.netbeans.mdr.persistence.MOFID) it.next())).collectObjects(result, visited);
            }
        }
    }
    
    public void addAssociationEnd(org.netbeans.mdr.persistence.MOFID mofId, String endName, boolean aggregate) {
        objectWillChange();
        associationEnds.add(new AssocEndDescriptor(mofId, endName, aggregate));
        objectChanged();
    }
    
    // this is always called from a synchronized section
    public void addSubclass(org.netbeans.mdr.persistence.MOFID mofId) {
        objectWillChange();
        subclasses.add(mofId);
        objectChanged();
    }

    // this is always called from a synchronized section
    public void addSuperclass(org.netbeans.mdr.persistence.MOFID mofId) {
        if (attributes != null) {
            throw new DebugException("bad thing in: " + getMofId());
        }
        objectWillChange();
        superclasses.add(mofId);
        objectChanged();
    }

    // always synchronized
    public void addReferenceDescriptor(org.netbeans.mdr.persistence.MOFID mofId, String name, org.netbeans.mdr.persistence.MOFID assocProxyId, String endName) {
        objectWillChange();
        references.put(name, new ReferenceDescriptor(mofId, assocProxyId, endName));
        objectChanged();
    }
    
    public void verify(Collection violations) {
        // [PENDING] should check all components and classifier attributes
    }
    
    /** Returns value of a given classifier-level attribute
     * @param name attribute name.
     * @throws StorageException
     */
    public Object getAttribute(String name) throws StorageException {
        StorableClass cls = getClsForAttr(name);
        if (cls == null) throw new DebugException("Classifier-level attribute '" + name + "' not found.");
        return cls.clAttrValues.get(name);
    }
    
    /** Sets value of a given classifier-level attribute
     * @param name attribute name
     * @param value attribute value
     */
    public void setAttribute(String name, Object value) throws StorageException {
        StorableClass cls = getClsForAttr(name);
        if (cls == null) throw new DebugException("Classifier-level attribute '" + name + "' not found.");
        
        cls.objectWillChange();
        AttributeDescriptor attribute = (AttributeDescriptor) cls.clAttrDescs.get(name);
        Object oldValue = cls.clAttrValues.put(name, value);

        if (!attribute.isMultivalued() && (value instanceof RefObject)) {
            StorableObject storableObj = (StorableObject) ((BaseObjectHandler) value)._getDelegate();
            storableObj.setComposite(getMofId(), storableObj.getMofId(), attribute.getMofId());
            if (oldValue != null) {
                storableObj = (StorableObject) ((BaseObjectHandler) oldValue)._getDelegate();
                storableObj.clearComposite();
            }
        }
        cls.objectChanged();
    }

    private StorableClass getClsForAttr(String name) throws StorageException {
        if (clAttrDescs == null || clAttrDescs.get(name) == null) {
            StorableClass result = null;
            for (Iterator it = superclasses.iterator(); it.hasNext() && result == null;) {
                result = ((StorableClass) getMdrStorage().getObject((org.netbeans.mdr.persistence.MOFID) it.next())).getClsForAttr(name);
            }
            return result;
        } else {
            return this;
        }
    }
    
    public StorableClass getClassProxy() throws StorageException {
        return this;
    }
    
    public final int getAttrIndex(String attributeName) throws StorageException {
        checkAttributes();

        int result = attributes.indexOf(attributeName);
        if (result < 0) {
            Logger.getDefault().log("attribute not found: " + attributeName);
            Logger.getDefault().log("collected attributes of " + getMofId() + ":");
            for (Iterator it = attributes.iterator(); it.hasNext();) {
                Logger.getDefault().log("\t" + it.next());
            }
            Logger.getDefault().log("registered superclasses:");
            for (Iterator it = superclasses.iterator(); it.hasNext();) {
                Logger.getDefault().log("\t" + it.next());
            }
            Logger.getDefault().log("registered subclasses:");
            for (Iterator it = subclasses.iterator(); it.hasNext();) {
                Logger.getDefault().log("\t" + it.next());
            }
            throw new DebugException();
        }
        return result;
    }

    public final int getAttrCount() throws StorageException {
        checkAttributes();
        return attributeDescs.length;
    }

    public final AttributeDescriptor getAttrDesc(int attrIndex) throws StorageException {
        checkAttributes();
        return attributeDescs[attrIndex];
    }
    
    public final AttributeDescriptor getAttrDesc(String attrName) throws StorageException {
        try {
            int index = getAttrIndex(attrName);
            return getAttrDesc(index);
        } catch (DebugException e) {
            StorableClass cls = getClsForAttr(attrName);
            if (cls == null) throw new DebugException("Attribute '" + attrName + "' not found.");
            return (AttributeDescriptor) cls.clAttrDescs.get(attrName);
        }
    }
    
    public ReferenceDescriptor getReferenceDescriptor(String referenceName) throws StorageException {
        ReferenceDescriptor result = (ReferenceDescriptor) refCache.get(referenceName);

        if (result == null) {
            result = (ReferenceDescriptor) references.get(referenceName);

            if (result == null) {
                for (Iterator it = superclasses.iterator(); it.hasNext();) {
                    result = ((StorableClass) getMdrStorage().getObject((org.netbeans.mdr.persistence.MOFID) it.next())).getReferenceDescriptor(referenceName);
                    if (result != null) {
                        break;
                    }
                }
            }
            if (result != null) refCache.put(referenceName, result);
        }

        return result;
    }
    
    public Collection getAllReferenceDescriptors() throws StorageException {
        return collectReferences(new ArrayList(), new HashSet());
    }
    
    private Collection collectReferences(Collection descs, Set visited) throws StorageException {
        descs.addAll(references.values());
        visited.add(getMofId());
        for (Iterator it = superclasses.iterator(); it.hasNext();) {
            org.netbeans.mdr.persistence.MOFID mofId = (org.netbeans.mdr.persistence.MOFID) it.next();
            if (!visited.contains(mofId)) {
                ((StorableClass) getMdrStorage().getObject(mofId)).collectReferences(descs, visited);
            }
        }
        return descs;
    }
    
    private synchronized void checkAttributes() throws StorageException {
        if (attributes == null) {
            //Logger.getDefault().log("collecting attributes for: " + getMofId());
            attributes = new ArrayList();
            List attribDescs = new ArrayList();
            indexDescriptors = new ArrayList();
            collectAttributes(attributes, attribDescs, indexDescriptors, new HashSet());
            this.attributeDescs = (AttributeDescriptor[])attribDescs.toArray(
                new AttributeDescriptor[attribDescs.size()]);
            buildIndexMap();
        }
    }
    
    void collectAssociationEnds(Collection result, Set visited) throws StorageException {
        org.netbeans.mdr.persistence.MOFID mofId;
        for (Iterator it = superclasses.iterator(); it.hasNext();) {
            mofId = (org.netbeans.mdr.persistence.MOFID) it.next();
            if (!visited.contains(mofId)) {
                ((StorableClass) getMdrStorage().getObject(mofId)).collectAssociationEnds(result, visited);
            }
        }
        result.addAll(associationEnds);
        visited.add(getMofId());
    }

    private void collectAttributes(List attrs, List descs, List indexes, Set visited) throws StorageException {
        org.netbeans.mdr.persistence.MOFID mofId;
        for (Iterator it = superclasses.iterator(); it.hasNext();) {
            mofId = (org.netbeans.mdr.persistence.MOFID) it.next();
            if (!visited.contains(mofId)) {
                ((StorableClass) getMdrStorage().getObject(mofId)).collectAttributes(attrs, descs, indexes, visited);
            }
        }

        for (int i = 0; i < attrDescs.length; i++) {
            attrs.add(attrDescs[i].getName());
            descs.add(attrDescs[i]);
        }
        if (indexDescs != null) {
            for (int i = 0; i < indexDescs.length; i++) {
                indexes.add(indexDescs[i]);
            }
        }
        visited.add(getMofId());
    }

    /**
     * Returns the super-class for generated handler classes.
     */
    public Class getClassSuperclass() throws StorageException, ClassNotFoundException {
        return resolveClass(classSuperclass);
    }
    
    public Class getClassCustomImpl() {
        try {
            return getClassSuperclass();
        } catch (Exception e) {}
        return null;
    }

    public Class getInstanceSuperclass() throws StorageException, ClassNotFoundException {
        return resolveClass(instanceSuperclass);
    }

    public Class getInstanceCustomImpl() {
        try {
            Class sup = getInstanceSuperclass();
            return sup == InstanceHandler.class ? null : sup;
        } catch (Exception e) {}
        return null;
    }

    public boolean isAbstract() {
        return abstractClass;
    }

    public boolean isSingleton() {
        return singletonClass;
    }
    
    public boolean isTransient () {
        return false;
    }

    /**
     * Returns the implementation super-class for the generated handler.
     *
     * <p>If the class has derived or inceptable members, {@link
     * TagSupport#getImplFullName(StorableObject, int)} determines the class
     * name.</p>
     *
     * <p>If this model class has model super-classes then the return value
     * depends on if there is a most specific implementation class among the
     * implementation classes of those super-classes. If yes, this most
     * specific implementation class is returned, otherwise {@link
     * TagSupport#getImplFullName(StorableObject, int)} determines the class
     * name.</p>
     *
     * <p>If non of the cases above applies, <code>ClassProxyHandler.class</code>
     * is returned as default value.</p>
     */
    public Class initClassSuperclass(Class iface[]) throws StorageException, ClassNotFoundException {
        Class result = null;
        if (classSuperclass == null) {
            try {
                result = resolveClass(this, TagSupport.CLASS);
    //            iface[0] = resolveInterface(this, TagSupport.CLASS);
            } catch (ClassNotFoundException e) {
    //            if (classDerived)
    //                throw e;
    //            Class current;
    //            Class currentIface[] = new Class[1];
    //            for (Iterator it = superclasses.iterator(); it.hasNext();) {
    //                current = ((StorableClass) getMdrStorage().getObject((org.netbeans.mdr.persistence.MOFID) it.next())).initClassSuperclass(currentIface);
    //                if (result == null || iface[0] == null || iface[0].isAssignableFrom(currentIface[0])) {
    //                    result = current;
    //                    iface[0] = currentIface[0];
    //                } else if (!currentIface[0].isAssignableFrom(iface[0])) {
    //                    throw e;
    //                }
    //            }
    //            if (result == null) {
                    result = ClassProxyHandler.class;
    //                iface[0] = RefClass.class;
    //            }
            }
            objectWillChange();
            classSuperclass = result.getName();
            objectChanged();
//        } else {
//            result = resolveClass(classSuperclass);
        }
        return result;
    }

    private static Class resolveClass(StorableClass cls, int type) throws ClassNotFoundException, StorageException {
        return resolveClass(TagSupport.getImplFullName(cls.getMetaObject(), type));
    }

    private static Class resolveClass(String className) throws ClassNotFoundException {
        return BaseObjectHandler.resolveImplementation(className);
    }
    
    private static Class resolveInterface(StorableClass cls, int type) throws ClassNotFoundException, StorageException {
        return BaseObjectHandler.resolveInterface(TagSupport.getTypeFullName(cls.getMetaObject(), type));
    }
    
    public Class initInstanceSuperclass(Class iface[]) throws StorageException, ClassNotFoundException {
        Class result = null;
        try {
            result = resolveClass(this, TagSupport.INSTANCE);
            iface[0] = resolveInterface(this, TagSupport.INSTANCE);
        } catch (ClassNotFoundException e) {
//            if (instanceDerived)
//                throw e;
            Class current;
            Class currentIface[] = new Class[1];
            for (Iterator it = superclasses.iterator(); it.hasNext();) {
                current = ((StorableClass) getMdrStorage().getObject((org.netbeans.mdr.persistence.MOFID) it.next())).initInstanceSuperclass(currentIface);
                if (result == null || iface[0] == null || iface[0].isAssignableFrom(currentIface[0])) {
                    result = current;
                    iface[0] = currentIface[0];
                } else if (!currentIface[0].isAssignableFrom(iface[0]))
                    throw e;
            }
            if (result == null) {
                result = InstanceHandler.class;
                iface[0] = RefObject.class;
            }
        }
        if (instanceSuperclass == null) {
            objectWillChange();
            this.instanceSuperclass = result.getName();
            objectChanged();
        }
        return result;
    }

    protected void deleteRecursive() throws StorageException {
        MOFID thisMofId = this.getMofId();
        MdrStorage storage = this.getMdrStorage ();
        MultivaluedIndex instancesIndex = storage.getInstancesIndex(thisMofId);
        Collection instances = instancesIndex.getItems (thisMofId);
        for (Iterator it = instances.iterator(); it.hasNext();) {
            MOFID instanceMofId = (MOFID) it.next();
            storage.removeObject (instanceMofId);
            it.remove ();
        }
        super.deleteRecursive();
    }

    public List getIndexDescriptors () throws StorageException {
        checkAttributes();
        return indexDescriptors;
    }

    /**
     * @param outputStream
     */
    public void write(java.io.OutputStream outputStream) {
        super.write(outputStream);
        try {
            IOUtils.write (outputStream, meta, this);
            IOUtils.write (outputStream, immediatePackage, this);

            if (attrDescs == null) {
                IOUtils.writeInt(outputStream, 0);
            } else {
                IOUtils.writeInt(outputStream, attrDescs.length + 1);
                for (int i = 0; i < attrDescs.length; i++) {
                    attrDescs[i].write(outputStream);
                }
            }
            
            IOUtils.writeInt(outputStream, datatypes.size());
            for (Iterator it = datatypes.keySet().iterator(); it.hasNext();) {
                String name = (String) it.next();
                IOUtils.writeString(outputStream, name);
                ((DatatypeDescriptor) datatypes.get(name)).write(outputStream);
            }

            // can be null
            IOUtils.write(outputStream, subclasses, this);
            // can be null
            IOUtils.write(outputStream, superclasses, this);
            // can be null
            IOUtils.write(outputStream, references, this);

            IOUtils.writeInt(outputStream, associationEnds.size());
            for (Iterator it = associationEnds.iterator(); it.hasNext();) {
                AssocEndDescriptor item = (AssocEndDescriptor) it.next();
                IOUtils.writeMOFID(outputStream, item.mofId, getMdrStorage(), getMofId());
                IOUtils.writeString(outputStream, item.endName);
                IOUtils.writeBoolean(outputStream, item.isAggregate);
            }

            IOUtils.writeBoolean(outputStream, classDerived);
            IOUtils.writeBoolean(outputStream, instanceDerived);
            IOUtils.writeBoolean(outputStream, singletonClass);
            IOUtils.writeBoolean(outputStream, abstractClass);

            IOUtils.writeString(outputStream, classSuperclass);
            IOUtils.writeString(outputStream, instanceSuperclass);
            
            if (indexDescs == null) {
                IOUtils.writeInt(outputStream, 0);
            } else {
                IOUtils.writeInt(outputStream, indexDescs.length);
                for (int i = 0; i < indexDescs.length; i++) {
                    indexDescs[i].write(outputStream,this);
                }
            }
            
            if (clAttrDescs == null) {
                IOUtils.writeInt(outputStream, 0);
            } else {
                IOUtils.writeInt(outputStream, clAttrDescs.size());
                for (Iterator it = clAttrDescs.values().iterator(); it.hasNext();) {
                    AttributeDescriptor desc = (AttributeDescriptor) it.next();
                    desc.write(outputStream);
                    IOUtils.write(outputStream, clAttrValues.get(desc.getName()), this);
                }
            }

        } catch (java.io.IOException e) {
            Logger.getDefault().notify(Logger.INFORMATIONAL, e);
        }
    }

    /**
     * @param inputStream
     */
    public void read(java.io.InputStream inputStream) {
        super.read(inputStream);
        try {
            meta = (org.netbeans.mdr.persistence.MOFID) IOUtils.read(inputStream, this);
            immediatePackage = (org.netbeans.mdr.persistence.MOFID) IOUtils.read (inputStream, this);

            int objCount = IOUtils.readInt(inputStream);
            if (objCount != 0) {
                int arrayLength = objCount - 1;
                attrDescs = new AttributeDescriptor[arrayLength];

                for (int i = 0; i < arrayLength; i++) {
                    attrDescs[i] = AttributeDescriptor.readResolve(inputStream, this);
                }
            }

            objCount = IOUtils.readInt(inputStream);
            datatypes = new HashMap(objCount, 1);
            for (int i = 0; i < objCount; i++) {
                String name = IOUtils.readString(inputStream);
                datatypes.put(name, DatatypeDescriptor.readResolve(inputStream, this));
            }

            subclasses = (List) IOUtils.read(inputStream, this);
            superclasses = (List) IOUtils.read(inputStream, this);
            references = (Map) IOUtils.read(inputStream, this);
            
            objCount = IOUtils.readInt(inputStream);
            associationEnds = new ArrayList(objCount);
            for (int i = 0; i < objCount; i++) {
                associationEnds.add(new AssocEndDescriptor(
                    IOUtils.readMOFID(inputStream, getMdrStorage(), getMofId()), 
                    IOUtils.readString(inputStream),
                    IOUtils.readBoolean(inputStream)
                ));
            }

            classDerived = IOUtils.readBoolean(inputStream);
            instanceDerived = IOUtils.readBoolean(inputStream);
            singletonClass = IOUtils.readBoolean(inputStream);
            abstractClass = IOUtils.readBoolean(inputStream);

            classSuperclass = IOUtils.readString(inputStream);
            instanceSuperclass = IOUtils.readString(inputStream);
            
            objCount = IOUtils.readInt(inputStream);
            if (objCount != 0) {
                indexDescs = new IndexDescriptor[objCount];
                for (int i = 0; i < objCount; i++) {
                    indexDescs[i] = IndexDescriptor.readResolve(inputStream, this);
                }
            }
            
            objCount = IOUtils.readInt(inputStream);
            if (objCount > 0) {
                clAttrDescs = new HashMap(objCount * 2);
                clAttrValues = new HashMap(objCount * 2);
                for (int i = 0; i < objCount; i++) {
                    AttributeDescriptor desc = AttributeDescriptor.readResolve(inputStream, this);
                    clAttrDescs.put(desc.getName(), desc);
                    Object value = IOUtils.read(inputStream, this, desc.getType().getName());
                    if (value instanceof MOFID) {
                        value = getMdrStorage().getRepository().getByMofId((MOFID) value);
                    }
                    clAttrValues.put(desc.getName(), value);
                }
            }
        } catch (IOException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }

    public final class ReferenceDescriptor {
        private org.netbeans.mdr.persistence.MOFID mofId;
        private final org.netbeans.mdr.persistence.MOFID proxyId;
        private final String endName;
        private StorableAssociation proxy = null;

        void replaceValues(Map table) {
            if (mofId != null) {
                mofId = (org.netbeans.mdr.persistence.MOFID) table.get(mofId);
            }
        }

        public ReferenceDescriptor(org.netbeans.mdr.persistence.MOFID mofId, org.netbeans.mdr.persistence.MOFID proxyId, String endName) {
            this.mofId = mofId;
            this.proxyId = proxyId;
            this.endName = endName;
        }
        
        public org.netbeans.mdr.persistence.MOFID getMofId() {
            return mofId;
        }

        public org.netbeans.mdr.persistence.MOFID getAssociationId() {
            return proxyId;
        }

        public String getEndName() {
            return endName;
        }

        public synchronized StorableAssociation getAssociation() {
            if (proxy == null) {
                try {
                    proxy = (StorableAssociation) getMdrStorage().getObject(proxyId);
                } catch (StorageException e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                }
            }
            return proxy;
        }

        public boolean isFirstEnd() {
            return getAssociation().getEnd2Name().equals(endName);
        }

    }

    public static final class AttributeDescriptor {
        private final int typeIndex;
        private final int maxSize;
        private final int minSize;
        private final boolean unique;
        private final boolean ordered;
        private final boolean changeable;
        private org.netbeans.mdr.persistence.MOFID mofId;
        private final String name;

        private final transient MdrStorage mdrStorage;
        private transient boolean indexed = false;
        private transient Class type = null;
        private transient String storageId = null;

        public AttributeDescriptor(MdrStorage storage, org.netbeans.mdr.persistence.MOFID mofId, String name, Class type, int minSize, int maxSize, boolean isUnique, boolean isOrdered, boolean isChangeable, String storageId) {
            this(storage, mofId, name, storage.storageValues(storageId).store(type.getName()), minSize, maxSize, isUnique, isOrdered, isChangeable, storageId);
            this.type = type;
        }

        public AttributeDescriptor(MdrStorage storage, org.netbeans.mdr.persistence.MOFID mofId, String name, int typeIndex, int minSize, int maxSize, boolean isUnique, boolean isOrdered, boolean isChangeable, String storageId) {
            this.mdrStorage = storage;
            this.typeIndex = typeIndex;
            this.minSize = minSize;
            this.maxSize = maxSize;
            this.unique = isUnique;
            this.ordered = isOrdered;
            this.changeable = isChangeable;
            this.mofId = mofId;
            this.name = name;
            this.storageId = storageId;
        }

        void replaceValues(Map table) {
            if (mofId != null) {
                mofId = (org.netbeans.mdr.persistence.MOFID) table.get(mofId);
            }
        }

        public Class getType() {
            if (type == null) {
                try {
                    type = BaseObjectHandler.resolveInterface((String) mdrStorage.storageValues(this.storageId).resolve(typeIndex));
                } catch (ClassNotFoundException e) {
                    throw new DebugException();
                }
            }
            return type;
        }

        public int getTypeIndex() {
            return typeIndex;
        }

        public int getMinSize() {
            return minSize;
        }

        public int getMaxSize() {
            return maxSize;
        }

        public boolean isMultivalued() {
            return (maxSize > 1) || (maxSize == -1);
        }

        public boolean isUnique() {
            return unique;
        }

        public boolean isOrdered() {
            return ordered;
        }

        public boolean isChangeable() {
            return changeable;
        }

        public org.netbeans.mdr.persistence.MOFID getMofId() {
            return mofId;
        }

        public String getName() {
            return name;
        }

        public void setIndexed(boolean indexed) {
            this.indexed = indexed;
        }

        public boolean isIndexed() {
            return indexed;
        }

        static AttributeDescriptor readResolve(java.io.InputStream stream, StorableClass storable) throws IOException {
            return new AttributeDescriptor(storable.getMdrStorage(), IOUtils.readMOFID(stream, storable.getMdrStorage(),storable.getMofId()), IOUtils.readString(stream), IOUtils.readInt(stream), IOUtils.readInt(stream), IOUtils.readInt(stream), IOUtils.readBoolean(stream), IOUtils.readBoolean(stream), IOUtils.readBoolean(stream), MdrStorage.getStorageIdFromMofId (storable.getMofId()));
        }

        void write(java.io.OutputStream outputStream) throws IOException {
            IOUtils.writeMOFID (outputStream, mofId, mdrStorage.getStorageById(storageId));
            IOUtils.writeString(outputStream, name);
            IOUtils.writeInt(outputStream, typeIndex);
            IOUtils.writeInt(outputStream, minSize);
            IOUtils.writeInt(outputStream, maxSize);
            IOUtils.writeBoolean(outputStream, unique);
            IOUtils.writeBoolean(outputStream, ordered);
            IOUtils.writeBoolean(outputStream, changeable);
        }
    }
    
    static final class AssocEndDescriptor {
        final MOFID mofId;
        final String endName;
        final boolean isAggregate;
        
        AssocEndDescriptor(MOFID mofId, String endName, boolean aggregate) {
            this.mofId = mofId;
            this.endName = endName;
            this.isAggregate = aggregate;
        }
    }

    public static final class IndexDescriptor {

        private static final int ATTRIB_ID = 0;
        private static final int ASSOC_END_ID = 1;
        
        private final String indexName;
        private final Field [] fields;

        public IndexDescriptor (String name, Field [] fields) {
            this.indexName = name;
            this.fields = fields;
        }

        public String getName () {
            return indexName;
        }

        public Field [] getFields () {
            return fields;
        }

        void write (java.io.OutputStream stream, StorableBaseObject storable) throws IOException {            
            IOUtils.writeString(stream, indexName);
            IOUtils.writeInt(stream, fields.length);
            for (int i = 0; i < fields.length; i++) {
                // Since reading of field id is not part of Field.read() method, we exclude it from write method too.
                if (fields[i] instanceof Attrib) {
                    IOUtils.writeInt(stream, ATTRIB_ID);                    
                } else {
                    IOUtils.writeInt(stream, ASSOC_END_ID);                    
                }
                fields[i].write (stream, storable);
            } // for
        }
        
        static IndexDescriptor readResolve(java.io.InputStream stream, StorableBaseObject storable) throws IOException {
            String indexName = IOUtils.readString(stream);
            int fieldsCount = IOUtils.readInt(stream);
            Field [] fields = new Field [fieldsCount];
            for (int i = 0; i < fieldsCount; i++) {
                int id = IOUtils.readInt(stream);
                switch (id) {
                    case ATTRIB_ID:                        
                        fields[i] = new Attrib(null, null, false);                        
                    break;
                    case ASSOC_END_ID:
                        fields[i] = new AssocEnd(null, null, false, null);
                } // switch
                fields[i].read (stream, storable);
            } // for
            return new IndexDescriptor (indexName, fields);            
        }
    
        public static abstract class Field {
            protected org.netbeans.mdr.persistence.MOFID id;
            protected String name;
            protected boolean isOrdered;
            
            public Field (org.netbeans.mdr.persistence.MOFID id, String name, boolean ordered) {
                this.id = id;
                this.name = name;
                this.isOrdered = ordered;
            }
            
            public org.netbeans.mdr.persistence.MOFID getId () {
                return id;
            }
            
            public String getName () {
                return name;
            }
            
            public boolean isOrdered () {
                return isOrdered;
            }
            
            public void write (java.io.OutputStream stream, StorableBaseObject storable) throws IOException {
                IOUtils.writeMOFID(stream, id, storable.getMdrStorage(), storable.getMofId());
                IOUtils.writeString(stream, name);
                IOUtils.writeBoolean(stream, isOrdered);
            }
            
            public void read (java.io.InputStream stream, StorableBaseObject storable) throws IOException {
                id = IOUtils.readMOFID(stream, storable.getMdrStorage(), storable.getMofId());
                name = IOUtils.readString(stream);
                isOrdered = IOUtils.readBoolean(stream);
            }
        } // class Field
        
        
        public static final class Attrib extends Field {
            
            public Attrib (org.netbeans.mdr.persistence.MOFID id, String name, boolean ordered) {
                super (id, name, ordered);
            }
            
            // no additional features required at the moment ...
            
        } // class Attrib

        
        public static final class AssocEnd extends Field {
            
            private org.netbeans.mdr.persistence.MOFID assocId;

            public AssocEnd (org.netbeans.mdr.persistence.MOFID id, String name, boolean ordered, org.netbeans.mdr.persistence.MOFID assocId) {
                super (id, name, ordered);
                this.assocId = assocId;                
            }

            public org.netbeans.mdr.persistence.MOFID getAssociation () {
                return assocId;
            }
            
            public void write (java.io.OutputStream stream, StorableBaseObject storable) throws IOException {
                super.write (stream, storable);
                IOUtils.write (stream, assocId, storable);                
            }
            
            public void read (java.io.InputStream stream, StorableBaseObject storable) throws IOException {
                super.read (stream, storable);
                assocId = (org.netbeans.mdr.persistence.MOFID) IOUtils.read(stream, storable);                
            }
            
        } // class AssocEnd

    } // IndexDescriptor
    
}
