/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.storagemodel;

import java.util.*;
import java.io.*;
import java.lang.reflect.Array;

import javax.jmi.reflect.*;

import org.netbeans.mdr.handlers.BaseObjectHandler;
import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.util.*;

/**
 * Instances of <code>CompositeCollection</code> are immutable collections uniting
 * several other collections. The methods of this class must be called in the following
 * order:
 *
 * <p><ol>
 *    <li>the constructor {@link #CompositeCollection()} to create an instance</li>
 *    <li>{@link addCollection(Collection} to add a collection forming a part of
 *         the current collection</li>
 *    <li>any accessor method</li>
 * </ol></p>
 *
 * <p>It is not allowed to add further collections after any of the accessors has
 * been called.</p>
 *
 * <p>Because the collections added using @link addCollection(Collection} are added
 * as is and their members are accessed only, when to be returned by methods of
 * this class resp. its {@link Iterator}, instances of <code>CompositeCollection</code>
 * are live if and only if all of the composed collections are live.</p>
 *
 * <p><b>[XXX]:</b> Put this class into <code>org.netbeans.mdr.util</code>
 *  package.</p>
 *
 * @author Martin Matula
 */
public class CompositeCollection implements Collection {

    private final ArrayList innerCollections = new ArrayList();
    private volatile boolean canChange = true;
    
    public CompositeCollection() {
    }
    
    /**
     * Adds a new collection. Can only be called before any accessor call.
     */
    public void addCollection(Collection collection) {
        if (canChange) {
            innerCollections.add(collection);
        } else {
            throw new DebugException("Set of collections cannot be changed after it was accessed.");
        }
    }
    
    public int size() {
        int size = 0;
        canChange = false;

        for (Iterator it = innerCollections.iterator(); it.hasNext();) {
            size += ((Collection) it.next()).size();
        }
        
        return size;
    }
    
    public boolean contains(Object obj) {
        canChange = false;

        for (Iterator it = innerCollections.iterator(); it.hasNext();) {
            if (((Collection) it.next()).contains(obj)) return true;
        }
        
        return false;
    }
    
    public Iterator iterator() {
        canChange = false;
        ArrayList innerIterators = new ArrayList();
        
        for (Iterator it = innerCollections.iterator(); it.hasNext();) {
            innerIterators.add(((Collection) it.next()).iterator());
        }
        
        return new CompositeIterator(innerIterators.iterator());
    }
    
    public boolean isEmpty() {
        canChange = false;

        for (Iterator it = innerCollections.iterator(); it.hasNext();) {
            if (!((Collection) it.next()).isEmpty()) return false;
        }
        
        return true;
    }
    
    public boolean containsAll(Collection collection) {
        for (Iterator it = collection.iterator(); it.hasNext();) {
            if (!contains(it.next())) return false;
        }
        
        return true;
    }
    
    public Object[] toArray(Object[] obj) {
        canChange = false;
        ArrayList value = new ArrayList();
        
        for (Iterator it = innerCollections.iterator(); it.hasNext();) {
            value.addAll((Collection) it.next());
        }
        
        return value.toArray(obj);
    }
    
    public Object[] toArray() {
        return toArray(new Object[0]);
    }

    /**
     * operation not supported
     */
    public void clear() {
        throw new UnsupportedOperationException();
    }
    
    /**
     * operation not supported
     */
    public boolean addAll(Collection collection) {
        throw new UnsupportedOperationException();
    }
    
    /**
     * operation not supported
     */
    public boolean remove(Object obj) {
        throw new UnsupportedOperationException();
    }
    
    /**
     * operation not supported
     */
    public boolean add(Object obj) {
        throw new UnsupportedOperationException();
    }
    
    /**
     * operation not supported
     */
    public boolean retainAll(Collection collection) {
        throw new UnsupportedOperationException();
    }
    
    /**
     * operation not supported
     */
    public boolean removeAll(Collection collection) {
        throw new UnsupportedOperationException();
    }

    /* --------------------------------------------------------------------- */
    /* -- CompositeIterator (inner class) ---------------------------------- */
    /* --------------------------------------------------------------------- */
    
    protected class CompositeIterator implements Iterator {
        private final Iterator innerIterators;
        private Iterator currentIterator;
        
        protected CompositeIterator(Iterator innerIterators) {
            this.innerIterators = innerIterators;
            this.currentIterator = (Iterator) innerIterators.next();
            hasNext();
        }
        
        public boolean hasNext() {
            while (innerIterators.hasNext()) {
                if (currentIterator.hasNext()) {
                    return true;
                } else {
                    currentIterator = (Iterator) innerIterators.next();
                }
            }
            return currentIterator.hasNext();
        }
        
        public Object next() {
            Object result = currentIterator.next();
            hasNext();
            return result;
        }
        
       /**
        * operation not supported
        */
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}
