/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.storagemodel;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;

import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.util.ListIterator;

import javax.jmi.reflect.*;

import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.mdr.handlers.BaseObjectHandler;
import org.netbeans.mdr.util.*;

/**
 *
 * @author Martin Matula
 */
public class AttrCollection implements Collection {
    protected Collection inner = new ArrayList();
    protected String attrName;

    protected transient StorableFeatured mdrObject;
    protected transient int maxSize;
    protected transient Class type;
    protected transient boolean isRefObject;
    protected transient MOFID metaMofId;
    protected transient boolean isIndexed = false;
    
    protected boolean needsUnwrap = false;
    
    public AttrCollection() {
    }

    AttrCollection(StorableFeatured mdrObject, StorableClass.AttributeDescriptor desc) throws StorageException {
        this(mdrObject, desc, null);
    }

    protected AttrCollection(StorableFeatured mdrObject, StorableClass.AttributeDescriptor desc, Collection values) throws StorageException {
        this.mdrObject = mdrObject;
        this.attrName = desc.getName();
        
        cacheValues(desc);

        if (values != null && !values.isEmpty()) {
            checkMaxSize(values.size());
            for (Iterator it = values.iterator(); it.hasNext();) {
                Object value = it.next();
                checkType(value);
                inner.add(value);
                if (isRefObject) {
                    setAttribComposite((RefObject) value);
                }
            }
        }
    }

    protected AttrCollection(StorableFeatured mdrObject, List values, int maxSize, Class type, String attrName, boolean isRefObject, MOFID metaMofId) {
        this.mdrObject = mdrObject;
        this.inner = values;
        this.maxSize = maxSize;
        this.type = type;
        this.attrName = attrName;
        this.isRefObject = isRefObject;
        this.metaMofId = metaMofId;
    }
    
    protected synchronized void checkUnwrap() {
        if (needsUnwrap) {
            for (ListIterator it = ((List) inner).listIterator(); it.hasNext();) {
                Object temp = it.next();
                if (temp instanceof MOFID) {
                    temp = mdrObject.getMdrStorage().getRepository().getByMofId((MOFID) temp);
                    if (temp != null) {
                        it.set(temp);
                    } else {
                        // recovery: remove the object if it cannot be resolved
                        it.remove();
                        Logger.getDefault().log(Logger.WARNING, "Invalid element found in attr. collection - removing.");
                    }
                } else if (temp == null) {
                    // recovery: remove the element if it is null
                    it.remove();
                    Logger.getDefault().log(Logger.WARNING, "Null found in attr. collection - removing.");
                }
            }
            needsUnwrap = false;
        }
    }

    public void read(InputStream stream, StorableFeatured storable) throws IOException {
        int size = IOUtils.readInt(stream);

        mdrObject = storable;
        
        try {
            attrName = IOUtils.readString(stream);
            cacheValues(storable.getClassProxy().getAttrDesc(attrName));
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }

        for (int i = 0; i < size; i++) {
            inner.add(IOUtils.read(stream, storable, type.getName()));
        }
        needsUnwrap = this.isRefObject;
    }
    
    public void write(OutputStream stream) throws IOException {
        IOUtils.writeInt(stream, inner.size());
        IOUtils.writeString(stream, attrName);
        for (Iterator it = inner.iterator(); it.hasNext();) {
            IOUtils.write(stream, it.next(), mdrObject);
        }
    }
    
    private void cacheValues(StorableClass.AttributeDescriptor attrDesc) throws StorageException {
        this.type = attrDesc.getType();
        this.maxSize = attrDesc.getMaxSize();
        this.isRefObject = RefObject.class.isAssignableFrom(this.type);
        this.metaMofId = attrDesc.getMofId();
        this.isIndexed = (mdrObject instanceof StorableObject) && attrDesc.isIndexed ();        
    }
    
    protected RefObject getMetaElement() {
        try {
            return (RefObject) mdrObject.getMdrStorage().getRepository().getByMofId(metaMofId);
        } catch (Exception e) {
            return null;
        }
    }

    public boolean add(Object obj) {
        checkType(obj);
        checkMaxSize(1);
        checkUnwrap();
        mdrObject.objectWillChange();
        if (isIndexed)
            ((StorableObject) mdrObject).removeFromIndex (metaMofId);
        boolean result = inner.add(obj);
        if (result) {
            if (isRefObject) {
                try {
                    setAttribComposite((RefObject) obj);
                } catch (StorageException e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                }
            }
            mdrObject.objectChanged();
        }
        if (isIndexed)
            ((StorableObject) mdrObject).addToIndex (metaMofId);
        return result;
    }
    
    public boolean addAll(Collection collection) {
        // should never be called
        throw new DebugException();
    }
    
    public void clear() {
        // should never be called
        throw new DebugException();
    }
    
    public Iterator iterator() {
        checkUnwrap();
        return new AttrIterator(inner.iterator());
    }
    
    public boolean removeAll(Collection collection) {
        // should never be called
        throw new DebugException();
    }
    
    public boolean retainAll(Collection collection) {
        // should never be called
        throw new DebugException();
    }
    
    public int size() {
        checkUnwrap();
        return inner.size();
    }
    
    public boolean contains(Object obj) {
        checkUnwrap();
        return inner.contains(obj);
    }
    
    public boolean containsAll(Collection collection) {
        checkUnwrap();
        return inner.containsAll(collection);
    }
    
    public boolean isEmpty() {
        checkUnwrap();
        return inner.isEmpty();
    }
    
    public boolean remove(Object obj) {
        checkUnwrap();
        mdrObject.objectWillChange();
        if (isIndexed)
            ((StorableObject) mdrObject).removeFromIndex (metaMofId);        
        boolean result = inner.remove(obj);
        if (result) {
            if (isRefObject) {
                try {
                    clearAttribComposite((RefObject) obj);
                } catch (StorageException e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                }
            }
            mdrObject.objectChanged();
        }
        if (isIndexed)
            ((StorableObject) mdrObject).addToIndex (metaMofId);
        return result;
    }
    
    public Object[] toArray() {
        checkUnwrap();
        return inner.toArray();
    }
    
    public Object[] toArray(Object[] obj) {
        checkUnwrap();
        return inner.toArray(obj);
    }
    
    public boolean equals(Object o) {
        checkUnwrap();
        return inner.equals(o);
    }
    
    public int hashCode() {
        checkUnwrap();
        return inner.hashCode();
    }
    
    protected void checkType(Object obj) {
        if (obj == null) {
            throw new NullPointerException();
        }
        
        if (!type.isInstance(obj)) {
            throw new TypeMismatchException(type, obj, getMetaElement(), "Expected type: " + type + ", supplied type: " + obj.getClass().getName());
        }
    }
    
    protected void checkMaxSize(int size) {
        if (maxSize != -1) {
            if (maxSize < (size + size())) throw new WrongSizeException(getMetaElement());
        }
    }

    protected void setAttribComposite(RefObject object) throws StorageException {        
        StorableObject storable = (StorableObject) ((BaseObjectHandler) object)._getDelegate();
        storable.setComposite(mdrObject, storable.getMofId(), metaMofId);
    }
    
    protected void clearAttribComposite(RefObject object) throws StorageException {
        StorableObject storable = (StorableObject) ((BaseObjectHandler) object)._getDelegate();
        storable.clearComposite();
    }
    
    protected class AttrIterator implements Iterator {
        private final Iterator inner;
        protected Object lastRead = null;
        
        public AttrIterator(Iterator inner) {
            this.inner = inner;
        }
        
        public boolean hasNext() {
            return inner.hasNext();
        }
        
        public Object next() {
            return (lastRead = inner.next());
        }
        
        public void remove() {
            mdrObject.objectWillChange();
            if (isIndexed)
                ((StorableObject) mdrObject).removeFromIndex (metaMofId);            
            inner.remove();
            if (isRefObject) {
                try {
                    clearAttribComposite((RefObject) lastRead);
                } catch (StorageException e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                }
            }
            if (isIndexed)
                ((StorableObject) mdrObject).addToIndex (metaMofId);
            mdrObject.objectChanged();
        }
        
        public boolean equals(Object o) {
            return inner.equals(o);
        }
        
        public int hashCode() {
            return inner.hashCode();
        }
    }
}
