/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreestorage;

import java.io.*;
import java.util.*;
import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.persistence.btreeimpl.btreeindex.*;

/**
* An implementation of List which uses a btree database as a backing store.
* This allows arrays which which not otherwise fit in memory.  To use a
* Virtual List:
* <ol>
* <li>
* First create it with VirtualList(String name).  This will create 
* a btree database with the given name to hold the list elements.
* <li>
* After an entry changes, call VirtualList.changed(index) to ensure that the
* changed version is written out to the repository.
* <li>
* When done, call VirtualArray.delete() to delete the backing repository
* </ol>
* Note that objects in the list must be Streamable.
*/
public class VirtualList extends AbstractList {
    private ArrayList keys = new ArrayList();
    private BtreeDatabase store;
    private String name;
    private BtreeStorage storage;

    /** Create a VirtualList
    * @param nm The name of the btree repository to create
    */
    public  VirtualList (String fileName, ObjectResolver resolver) {
        this (defaultMap (fileName), resolver);
    }
    
    public VirtualList(Map properties, ObjectResolver resolver) {
	try  {
	    this.storage = (BtreeStorage) (new BtreeFactory()).createStorage(properties);
	    this.storage.create(true, resolver);
	    this.store = (BtreeDatabase)this.storage.getPrimaryIndex();
   	} 
	catch (StorageException ex) {
	    throw new RuntimeStorageException(ex);
	}
    }

    /** Delete the VirtualList's backing repository
    */
    public void delete() {
	try  {
	    store.close();
	    store.delete(name);
   	} 
	catch (StorageException ex) {
	    throw new RuntimeStorageException(ex);
	}
    }

    /** Get the specified element
    * @param index The index of the element to get
    * @return the element at the specified index
    */
    public synchronized Object get(int index) {
	try  {
	    MOFID key = (MOFID)keys.get(index);
	    return store.get(key);
   	} 
	catch (StorageException ex) {
	    throw new RuntimeStorageException(ex);
	}
    }

    /** Get the number of elements in the list
    */

    public int size() {
    	return keys.size();
    }

    /** Set the element at the specified index
    * @param index the index to set
    * @param element the element to set the index to
    * @return the element formerly at that index
    */
    public synchronized  Object set(int index, Object element) {
	try  {
	    MOFID key = (MOFID)keys.get(index);
	    Object retval = store.getIfExists(key);
	    store.put(key, element);
	    return retval;
   	} 
	catch (StorageException ex) {
	    throw new RuntimeStorageException(ex);
	}
    }

    /** Add a new element to the list
    * @param the index at which to ass it
    * @param element the element to add
    */
    public synchronized  void add(int index, Object element) {
	try  {
	    MOFID key = new MOFID(this.storage);
	    keys.add(index, key);
	    store.add(key, element);
   	} 
	catch (StorageException ex) {
	    keys.remove(index);
	    throw new RuntimeStorageException(ex);
	}
    }

    /** Remove ana element from the list
    * @param index Where to remove the element
    * @return the element removed.
    */
    public synchronized  Object remove(int index) {
	try  {
	    MOFID key = (MOFID)keys.get(index);
	    Object o = store.get(key);
	    store.remove(key);
	    keys.remove(index);
	    return o;
   	} 
	catch (StorageException ex) {
	    throw new RuntimeStorageException(ex);
	}
    }

    /** mark the elemenet at the given index changed
    * @param the index of the changed element.
    */
    public synchronized void changed(int index) {
	try  {
	    store.objectStateChanged((MOFID)keys.get(index));
   	} 
	catch (StorageException ex) {
	    throw new RuntimeStorageException(ex);
	}
    }
    
    
    private static Map defaultMap (String fileName) {
        java.util.Hashtable map = new java.util.Hashtable ();
        map.put (org.netbeans.mdr.persistence.btreeimpl.btreestorage.BtreeFactory.STORAGE_FILE_NAME,fileName);
        return map;
    }
}
