/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.mdr.persistence.btreeimpl.btreestorage;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.Set;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.mdr.persistence.SinglevaluedIndex;
import org.netbeans.mdr.persistence.Storage;
import org.netbeans.mdr.persistence.StorageBadRequestException;
import org.netbeans.mdr.persistence.StorageException;


/**
 *
 * @author Tomas Hurka
 */
public class PrimaryIndex implements SinglevaluedIndex {

    private FileCache	fileCache;	// source of pages from file
    private int		fileId;		// file where our pages are stored,
    					// for making FileCache requests    
    private Storage.EntryType	keyType;
    private Storage.EntryType	dataType;
    private String 		indexName;
    private int                 valuesInPage;
    private final long          firstMofIdSerial;
    private static final int    NO_VALUE=0;
    private static final int    sizeof_int=4;
    private static final int    reservedValues=FileHeader.HEADER_SIZE/sizeof_int;
    
    private class ValInfo {
        private int index;
        private CachedPage page;
        
        ValInfo(Object mofId) throws StorageException {
            MOFID mid=(MOFID)mofId;
            long serial=mid.getSerialNumber();
            
            if (serial>=BtreeFactory.FIRST_EXTERNAL_ID) {
                assert serial>=firstMofIdSerial;
                serial-=firstMofIdSerial-BtreeFactory.FIRST_EXTERNAL_ID;
            }
            serial+=reservedValues;
            long pageNum=serial/valuesInPage;
            page=fileCache.getPage(fileId, (int)pageNum);
            
            index=(int)(serial%valuesInPage)*sizeof_int;
        }
    }
    
    PrimaryIndex(String name, int id, FileCache cache, long firstSerial, Storage.EntryType key, Storage.EntryType data) {
        indexName=name;
        fileId=id;
        fileCache=cache;
        firstMofIdSerial=firstSerial;
        keyType=key;
        dataType=data;
        valuesInPage=cache.getPageSize()/sizeof_int; 
        
    }

    public void add(Object key, Object value) throws StorageException {
        put(key, value);
    }

    public Object get(Object key) throws StorageException, StorageBadRequestException {
        Object result = getIfExists(key);
        if (result == null) {
            throw new StorageBadRequestException (
                MessageFormat.format("Key {0} not found in index",
                    new Object[] {key}));
        }
        return result;
    }

    public Object getIfExists(Object key) throws StorageException {
        ValInfo info=new ValInfo(key);
        int value = Converter.readInt(info.page.contents,info.index);
        
        fileCache.unpin(info.page);
        if (value==NO_VALUE)
            return null;
        return new Integer(value);
   }

    public Storage.EntryType getKeyType() throws StorageException {
        return keyType;
    }

    public String getName() throws StorageException {
        return indexName;
    }

    public Object getObject(Object key, SinglevaluedIndex repos) throws StorageException {
        Object result = getObjectIfExists(key, repos);
        if (result == null) {
            throw new StorageBadRequestException (
                MessageFormat.format("Key {0} not found in index",
                    new Object[] {key}));
        }
        return result;
    }

    public Object getObjectIfExists(Object key, SinglevaluedIndex repos) throws StorageException {
        throw new UnsupportedOperationException();
    }

    public Storage.EntryType getValueType() throws StorageException {
        return dataType;
    }

    public Set keySet() throws StorageException {
        throw new UnsupportedOperationException();
    }

    public boolean put(Object key, Object value) throws StorageException {
        ValInfo info=new ValInfo(key);
        int oldVal = Converter.readInt(info.page.contents,info.index);
        int val=((Integer)value).intValue();
        
        if (oldVal!=val) {
            fileCache.setWritable(info.page);
            Converter.writeInt(info.page.contents,info.index, val);
        }
        fileCache.unpin(info.page);
        return oldVal!=NO_VALUE;
    }

    public Collection queryByKeyPrefix(Object prefix, SinglevaluedIndex repos) throws StorageException {
        throw new UnsupportedOperationException();
    }

    public boolean remove(Object key) throws StorageException {
        ValInfo info=new ValInfo(key);
        int val=Converter.readInt(info.page.contents,info.index);
        
        if (val!=NO_VALUE) {
            fileCache.setWritable(info.page);
            Converter.writeInt(info.page.contents,info.index, NO_VALUE);
        }
        fileCache.unpin(info.page);
        return val!=NO_VALUE;
    }

    public void replace(Object key, Object value) throws StorageException, StorageBadRequestException {
        ValInfo info=new ValInfo(key);
        int oldVal = Converter.readInt(info.page.contents,info.index);
        
        if (oldVal==NO_VALUE) {
            throw new StorageBadRequestException (
                MessageFormat.format("Key {0} not found in index",
                    new Object[] {key}));
        }            
        int val=((Integer)value).intValue();
        fileCache.setWritable(info.page);
        Converter.writeInt(info.page.contents,info.index, val);
        fileCache.unpin(info.page);
    }

    public Collection values() throws StorageException {
        throw new UnsupportedOperationException();
    }
    
}
