/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreestorage;

import java.io.*;
import java.text.*;
import java.util.*;
import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.persistence.btreeimpl.btreeindex.*;

/**
* This is an index from names (Strings) to some object.
*/
abstract class NameIndex implements Streamable  {

    /* hash table */
    protected HashMap hashOnName;

    /* name of object */
    String name = "";

    /** Create a new NameIndex  */
    public NameIndex() {
        hashOnName = new HashMap();
    }

    /** give object a name
    * @param nm name of NameIndex
    */
    public void setName(String nm) {
    	name = nm;
    }

    /** return name set with setName
    */
    public String toString() {
    	return name;
    }

    /** get an object by its name.  If none exists, throw an exception
    * @param name the name associated with the object
    */
    protected synchronized Object getObj(String name)throws StorageException  {
        Object id = getObjIf(name);
        if (id == null) {
            throw new StorageBadRequestException(
                MessageFormat.format(
                    "There is no value for key \"{0}\"", 
                    new Object[] {name} ) );
        }
        return id;
    }

    /** get an object by its name.  If none exists, return null
    * @param name the name associated with the object
    */
    protected synchronized Object getObjIf(String name)throws StorageException  {
        return hashOnName.get(name);
    }

    /**
    * Add a name-Object pair to the NameIndex.  If one already existed,
    * throw an exception.
    * @param name name of object to add to index
    * @param id object to add
    */
    protected synchronized void addObj(String name, Object id) throws StorageException {
        if (hashOnName.get(name) != null) {
            throw new StorageBadRequestException(
                MessageFormat.format(
                    "There is already a value for key \"{0}\"", 
                    new Object[] {name} ) );
        }
        hashOnName.put(name, id);
    }

    /** Remove a name-Object pair from the NameIndex.  If it is not present, 
    * throw an exception.
    * @param name name of object to rmove from index
    */
    public synchronized void remove(String name) throws StorageException {
        if (hashOnName.get(name) == null) {
            throw new StorageBadRequestException(
                MessageFormat.format(
                    "There is no value for key \"{0}\"", 
                    new Object[] {name} ) );
        }
        hashOnName.remove(name);
    }

    /**
    * Remove all name-Object pairs from the NameIndex.
    */
    public synchronized void clear() {
        hashOnName.clear();
    }

    /** return the names of all objects in the NameIndex, in no particular
    * order.  Note that this is a snapshot made at the time this method is 
    * called.
    * @return list of all names
    */
    public synchronized String[] listNames() {
        ArrayList list = new ArrayList();
        list.addAll(hashOnName.keySet());
        return (String[])list.toArray(new String[list.size()]);
    }

    /** iterate over entries
    */
    public synchronized Iterator iterator() {
    	return hashOnName.entrySet().iterator();
    }

    /** serialize the NameIndex to a stream.  The format is:
    * <p>
    * number of pairs
    * <p>
    * name of object 1 (in UTF-8)
    * <p>
    * Representation of object 1
    * <p>
    * ...
    * name of object N (in UTF-8)
    * <p>
    * Representation of object N
    * <p>
    *  @param strm stream to serialize to
    */
    public synchronized void write(OutputStream strm) throws StorageException{
    	if (strm instanceof DataOutputStream)
	    write((DataOutputStream)strm);
	else
	    write(new DataOutputStream(strm));
    }

    /** write to a DataOutputStream
    */
    public synchronized void write(DataOutputStream dstrm) throws StorageException{
        try {
            Set entries = hashOnName.entrySet();
            dstrm.writeInt(entries.size());
            Iterator itr = entries.iterator();
            while (itr.hasNext()) {
                Map.Entry entry = (Map.Entry)itr.next();
                dstrm.writeUTF((String)entry.getKey());
		writeObjectToStream(entry.getValue(), dstrm);
            }
            dstrm.flush();
        }
        catch (IOException ex) {
            throw new StorageIOException(ex);
        }
    }

    /** deserialize the NameIndex from a stream.  The format is as
    * described in write
    *  @param strm stream to deserialize from
    */
    public synchronized void read(InputStream strm) throws StorageException{
    	if (strm instanceof DataInputStream)
	    read((DataInputStream)strm);
	else
	    read(new DataInputStream(strm));
    }

    /** read from a DataOutputStream
    */
    public synchronized void read(DataInputStream dstrm) throws StorageException{
        try {
            clear();
            int size = dstrm.readInt();
            for (int i = 0; i < size; i++) {
                String key = dstrm.readUTF().intern();
		Object id = readObjectFromStream(dstrm);
                addObj(key, id);
            }
        }
        catch (IOException ex) {
            throw new StorageIOException(ex);
        }
    }

    /**
    * write object to stream.  Used by serialization.
    * @param obj object to write
    * @param strm stream to write it to
    */
    protected abstract void writeObjectToStream(
    	Object obj, DataOutputStream strm) throws StorageException ;

    /**
    * read object from stream.  Used by deserialization.
    * @param strm stream to read from
    * @return obj object read
    */
    protected abstract Object readObjectFromStream(DataInputStream strm)
    	throws StorageException;
}



