/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreestorage;

import java.io.*;
import java.text.*;
import java.util.*;

import org.netbeans.mdr.persistence.*;

/** The transaction log file used by the FileCache.
* <p>
* The scheme used is before-image logging.  That is, before a page is modified,
* its contents are written to the log file.  The log file itself contains two 
* sorts of pages: these before-image pages, and map pages, which keep track
* of the PageID of each before-image file.
* <p>
* At the successful completion of each transaction, the log file is deleted.
* If, on cache creation, the log file exists, this indicates that a previous
* transaction did not complete successfully. Recovery is attempted, using the
* contents of the log file to back out the partially completed transaction.
*/
class LogFile {

    /* page size for the cache */
    private int pageSize;

    /* number of files in the cache */
    private int numFiles;

    /* name of the log file */
    private final String baseName;

    /* name of the log file */
    private String name;

    /* bitmaps showing which pages are currently logged */
    private BitSet pageBitmaps[];

    /* the current map page */
    private MapPage currentMap;

    /* owning cache */
    private FileCache cache;

    /* file ID */
    long fileId;

    /* counts for forced failure during regreession tests */
    private int beforeWriteFailure = -1;
    private int afterCommitFailure = -1;
    private int recoveryFailure = -1;

    /** Create the log file.  If the file already exists on disk, 
    * attempt recovery.  Note that exceptions pnly occur during recovery.
    * @param fCache the cache which owns this log file
    * @param baseName the name of the log file without the suffix - i.e. name of the repository
    * @param pgSz the page size
    * @param files the number of files being logged
    * @param id the fileId for the files being logged
    * @exception StorageException I/O error during recovery
    * @exception BadParameterException the log file is not consistent with
    * the files being recovered
    * @exception ConsistencyException the log file is corrupt on disk
    */
    LogFile(FileCache fCache, String baseName, int pgSz, int files, long id) 
            throws StorageException {
        // Size must be a power of 2 and >= 4096
        // no more than 4096 files
        cache = fCache;
        pageSize = pgSz;
        this.baseName = baseName;
        numFiles = files;
        fileId = id;
        pageBitmaps = new BitSet[files];
        for (int i = 0; i < files; i++)
            pageBitmaps[i] = new BitSet();

        if (new File(BtreeDatabase.getFileName(baseName, BtreeDatabase.LFL)).exists())
            recover();
    }

    /* returns true if page is already logged */
    private boolean isPageLogged(CachedPage page) {
        return pageBitmaps[page.key.fileIndex].get(page.key.offset/pageSize);
    }

    /* create the log file */
    private RandomAccessFile createPhysicalLog() throws StorageException {
        String name = BtreeDatabase.getFileName(baseName, BtreeDatabase.LFL);
        RandomAccessFile file = null;
        try {
            file = FileCache.getFile(name);
            file.setLength(0);
            writeMap(file);
            return file;
        } catch (IOException ex) {
            throw new StorageIOException(ex);
        } finally {
            if (file != null) {
                this.name = name;
            }
        }
    }

    /* add a new page to the log file 
    * @param page the page to add
    * @exception StorageException error loggin the page to the before file
    */
    void addPageToLog(CachedPage page) throws StorageException {

        if (page.key.offset >= currentMap.getEOF(page.key.fileIndex)) {
                return;
        }

        if (isPageLogged(page))
            return;

        RandomAccessFile file;
        try {
            file = name == null ? createPhysicalLog() : FileCache.getFile(name);
            file.seek(currentMap.nextPageOffset());
            file.write(page.contents);
        }
        catch (IOException ex) {
            throw new StorageIOException(ex);
        }

        beforeWriteFailure = 
            cache.checkForForcedFailure(
        "org.netbeans.mdr.persistence.btreeimpl.btreestorage.LogFile.beforeWriteFailure", 
                beforeWriteFailure);
        pageBitmaps[page.key.fileIndex].set(page.key.offset/pageSize);

        /* add to map */
        currentMap.add(page);
        if (currentMap.isFull()) {

            // map is full; need to write it
            writeMap(file);
        }

        /* prevent from being written until log is flushed */
        cache.holdForLog(page);
    }

    /* write the current map page to the log */
    private void writeMap(RandomAccessFile file) throws StorageException{
        flushFile(file);
        currentMap.write(file);
        flushFile(file);
        if (currentMap.isFull())
            currentMap = new MapPage(currentMap);
    }

    /* sync the file to disk */
    private void flushFile(RandomAccessFile file) throws StorageException{
        try {
            file.getFD().sync();
        }
        catch (IOException ex) {
            throw new StorageIOException(ex);
        }
    }

    /** make the log consistent on disk 
    * @exception StorageException I/O error writing the log
    */
    void flush() throws StorageException{
        try {
            writeMap(FileCache.getFile(name));
            cache.logWasFlushed();
        } catch (IOException e) {
            throw new StorageIOException(e);
        }
    }

    /** begin a new transaction
    * @param files the files being logged
    * @param timeStamp the timestamp for the previously comitted transaction
    * @param long newTimeStamp the timestamp for the new transaction
    * @exception StorageException I/O error accessing the files
    */
    void begin(String fileNames[], long timeStamp, long newTimeStamp) 
                throws StorageException {

        name = null;
        currentMap = new MapPage(pageSize, numFiles, 0);
        RandomAccessFile files[];
        try {
            files = FileCache.getFiles(fileNames);
        } catch (IOException e) {
            throw new StorageIOException(e);
        }
        currentMap.setEOFs(files);
        for (int i = 0; i < numFiles; i++) {
            pageBitmaps[i].xor(pageBitmaps[i]);
        }
        currentMap.setTimeStamps(timeStamp, newTimeStamp);
        currentMap.setFileID(fileId);
    }

    /** commit the current transaction
    * @exception StorageException I/O error closing or deleting the log
    */
    void commit() throws StorageException{
        try {
            if (name != null) {
                FileCache.closeFile(name);
                new File(name).delete();
            }
        }
        catch (IOException ex) {
            throw new StorageIOException(ex);
        }
        name = null;
        afterCommitFailure = 
            cache.checkForForcedFailure(
        "org.netbeans.mdr.persistence.btreeimpl.btreestorage.LogFile.afterCommitFailure", 
            afterCommitFailure);
    }

    /** close the log file 
    * @exception StorageException I/O error closing the log
    */
    void close() throws StorageException{
        try {
            if (name != null) {
                FileCache.closeFile(name);
        }
        } catch (IOException ex) {
            throw new StorageIOException(ex);
        }
    }

    /** return the current log file size */
    int fileSize() {
        return currentMap.nextPageOffset();
    }

    /** recover from an incomplete transaction 
    * @exception StorageException I/O error during recovery
    * @exception BadParameterException the log file is not consistent with
    * the files being recovered
    * @exception ConsistencyException the log file is corrupt on disk
    */
    void recover() throws StorageException {
        RandomAccessFile files[] = null;
        RandomAccessFile logFile = null;

        try {
            try {
                logFile = new RandomAccessFile(BtreeDatabase.getFileName(baseName, BtreeDatabase.LFL), "r");
                files = cache.getFiles();
                if (files.length != numFiles) {
                    throw new StorageBadRequestException(
                        MessageFormat.format(
                            "Log file contains {0} files; {1} were requested",
                            new Object[] { 
                                new Integer(numFiles), 
                                new Integer(files.length)}));
                }

                int numPages = (int)logFile.length() / pageSize;
                if (numPages > 0) {
                    byte pageBuffer[] = new byte[pageSize];

                    MapPage page = new MapPage(logFile, 0, pageSize);
                    page.checkParameters(pageSize,  numFiles);

                    for (int i = 0; i < numFiles; i++) {
                        page.checkFileHeader(files[i]);

                    }
                    while (true) {
                        page.recover(files, logFile, numPages, pageBuffer);

                        recoveryFailure = 
                            cache.checkForForcedFailure(
                "org.netbeans.mdr.persistence.btreeimpl.btreestorage.LogFile.recoveryFailure", 
                                recoveryFailure);
                        MapPage newPage = page.getNext(logFile, numPages);
                        if (newPage != null && !newPage.isEmpty()) {
                            page = newPage;
                        }
                        else {
                            break;
                        }
                    } 

                    page.truncateFiles(files);
                    logFile.close();
                    (new File(BtreeDatabase.getFileName(baseName, BtreeDatabase.LFL))).delete();
                }
            }
            finally  {
                if (logFile != null) {
                    logFile.close();

                }
            }
        }   
        catch (IOException ex) {
            throw new StorageIOException(ex);
        }
    }
}


