/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreestorage;

import java.io.*;
import java.util.*;
import org.netbeans.mdr.util.Logger;

/**
* A ContinuationBtreeExtent in an extent which is the second or subsequent
* extent in a record.
*
*   <p>
*   Disk format:
*   <ol>
*   <li>
*   bytes 0-7               generic BtreeExtent header
*   <li>
*   bytes 8-11              data length (DL)
*   <li>
*   bytes 12-11+DL          data
*   </ol>
*/
class ContinuationBtreeExtent extends ActiveBtreeExtent {

    /** size of fixed part of header */
    static final int CONTINUATION_FIXED_LENGTH = 12;

    /** data which will fit in biggest continuation extent */
    static final int MAX_CONTINUATION_DATA_LENGTH = 
        BtreeDataFile.MAX_BYTES_IN_EXTENT - CONTINUATION_FIXED_LENGTH;

    /** initialize a new ContinuationBtreeExtent
    * @param file the BtreeDataFile this extent will belong to
    * @param chunkNum where this extent begins
    * @param numChunks the size of the extent
    */
    ContinuationBtreeExtent(BtreeDataFile file, int chunkNum, short numChunks) {
        super(file, chunkNum, numChunks);
        dataStart = CONTINUATION_FIXED_LENGTH;
    }

    /** Convert a deleted extent to an active one.  The deleted extent has
    * already been removed from its chain
    * @param del the extent to convert
    * @param length data length 
    */
    ContinuationBtreeExtent(DeletedBtreeExtent del, int length) {
        super(del);
        setMyDataLength(length);
        dataStart = CONTINUATION_FIXED_LENGTH;
    }

    /** create a new ContinuationBtreeExtent 
    * @param file the BtreeDataFile this extent will belong to
    * @param chunkNum where this extent begins
    * @param numChunks the size of the extent
    * @param dLen how much data this extent will contain
    */
    ContinuationBtreeExtent(
        BtreeDataFile file, int chunkNum, short numChunks, int dLen) {

        super(file, chunkNum, numChunks);
        dataStart = CONTINUATION_FIXED_LENGTH;
        setMyDataLength(dLen);
        headerIsDirty = true;
    }

    /** read the type-specific parts of the extent header from the
    * buffer.
    * @param buffer the buffer to read from
    * @param offset the offset to being reading at
    */
    void readHeaderFromPage(byte buffer[], IntHolder offset) {
        dataLength = Converter.readInt(buffer, offset);
        if (dataLength > getAvailableDataLength()) {            
            Logger.getDefault().log("Bad data length read: ----------------------");
            Logger.getDefault().log("chunk number: " + myChunkNum);
            Logger.getDefault().log("dataLength: " + dataLength);            
            Logger.getDefault().log("available: " + getAvailableDataLength());            
        }
    }

    /** write the type-specific part of the header to the file cache
    * @param page the page to write to
    * @param offset the offset to begin an
    */
    protected void writeHeaderToPage(CachedPage page, int offset) {
        Converter.writeInt(page.contents, offset, dataLength);
    }

    /** get the amount of data contained in this extent
    * @return amount of data
    */
    int getMyDataLength() {
        return dataLength;
    }

    /** get how much data this extent could contain
    * @return maximum amount of data which would fit
    */
    int getAvailableDataLength() {
        return (chunks * BtreeDataFile.BTREE_CHUNK_SIZE) - 
                CONTINUATION_FIXED_LENGTH;
    }

    /** Get the magic number for this type of extent
    * @return the magic number
    */
    short getMagic() {
      return CONTINUATION_MAGIC;
    }

    /** return the number of chunks an extent with this much data would
    * occupy.  If an extent of maximum size wouldn't hold this much,
    * return the maximum extent size.
    * @param dataLength amount of data to hold.
    * @return size of extent in chunks
    */
    static int getNumChunks(int dataLength) {
        int size = (dataLength + CONTINUATION_FIXED_LENGTH - 1) /
                            BtreeDataFile.BTREE_CHUNK_SIZE + 1;
        return Math.min(size, BtreeDataFile.MAX_CHUNKS_IN_EXTENT);
    }

    /** set the amount of data contained in this extent
    * @param length amount of data
    */
    int setMyDataLength(int length) {
        int oldDatalength = dataLength;
        dataLength = Math.min(length, getAvailableDataLength());
        if (oldDatalength != dataLength) {
            headerIsDirty = true;
        }
        return dataLength;
    }

    /** is the extent already full of data
    * @return true if the extent has no room for more data
    */
    boolean isMaximum() {
        return dataLength == MAX_CONTINUATION_DATA_LENGTH;
    }

    /** return type of extent
    * @return IS_CONTINUATION
    */
    byte getType() {
        return IS_CONTINUATION;
    }

    /** return name of type of extent
    * @return CONTINUATION_NAME
    */
    String getTypeName() {
        return CONTINUATION_NAME;
    }
}
