/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreeindex;

import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.util.MapEntryImpl;
import java.util.*;
import java.text.*;

/**
 * Btree implementation of SinglevaluedIndex
 *
 * @author	Dana Bergen
 * @version	1.0
 */
public class SinglevaluedBtree extends Btree implements SinglevaluedIndex {

    public SinglevaluedBtree(String name, Storage.EntryType keyType, 
    			     Storage.EntryType dataType, 
			     BtreePageSource pageSource) 
			     throws StorageException {
	super(name, keyType, dataType, pageSource);
    }

    protected void init() throws StorageException {
        uniqueKeys = true;
	uniqueValues = false;
	super.init();
    }

    /**
     * No-argument constructor for reconstructing via read().  Only used with
     * Btree's whose pageSource is a BtreeMDRSource.
     */
    public SinglevaluedBtree() {
    }

    /*
     *
     * SinglevaluedIndex interface methods
     *
     */

    /**
     * Return a single object associated with the specified key.
     *
     * @param	key	key for lookup
     *
     * @return	the data stored in the index for that key, or null if
     *		not found
     *
     * @exception StorageException	If there was a problem reading 
     *					pages
     */
    public Object getIfExists(Object key)
    	throws StorageException {
   	return getIfExistsInternal(key, null);
    }

    /** Like getIfExists, but if the index contains keys, this returns the object 
     * corresponding to the key
     * @return
     * @param key
     * @throws StorageException
     */
    public Object getObjectIfExists(Object key, SinglevaluedIndex repos) 
    	throws StorageException {
	return getIfExistsInternal(key, repos);
    }

    /* get item or object */
    private Object getIfExistsInternal(Object key, SinglevaluedIndex repos) 
    	throws StorageException {

        beginRead();
	try { 
	    byte[] keyBuffer;
	    byte[] dataBuffer;
	    Object result;

	    if ((keyBuffer = keyInfo.toBuffer(key)) == null) {
		throw new StorageBadRequestException(
		  MessageFormat.format(
		  "Invalid key type for this index: {0} received, {1} expected",
			    new Object[] {
				key.getClass().getName(),
				keyInfo.typeName()} ));
	    }	

	    BtreePage root = pageSource.getPage(rootPageId, this);
	    dataBuffer = root.get(keyBuffer);
	    if (dataBuffer != null) {
	    	if (repos == null) {
		    result = dataInfo.fromBuffer(dataBuffer);
		}
		else {
		    result = dataInfo.objectFromBuffer(dataBuffer, repos);
		}
	    } else {
		result = null;
	    }
	    pageSource.unpinPage(root);
	    return result;
	} finally {
	    endRead();
	}
    }

    /**
     * Return a single object associated with the specified key.
     *
     * @param	key	key for lookup
     *
     * @return	the data stored in the index for that key
     *
     * @exception StorageException	If no matching item was found, or
     * 					there was a problem reading pages
     */
    public Object get(Object key) throws StorageException {
        beginRead();
	try { 
	
	    Object result = getIfExists(key);
	    if (result == null) {
		throw new StorageBadRequestException (
		    MessageFormat.format("Key {0} not found in index",
			new Object[] {key}));
	    }
	    return result;
	} finally {
	    endRead();
	}
    }

    /** Like get, but if the index contains keys, this returns the object 
     * corresponding to the key
     * @return
     * @param key
     * @throws StorageException
     */
    public Object getObject (Object key, SinglevaluedIndex repos) throws StorageException {
        beginRead();
	try { 
	
	    Object result = getObjectIfExists(key, repos);
	    if (result == null) {
		throw new StorageBadRequestException (
		    MessageFormat.format("Key {0} not found in index",
			new Object[] {key}));
	    }
	    return result;
	} finally {
	    endRead();
	}
    }

    /**
     * Add a new entry to the index. 
     *
     * @param	key	key to insert	
     * @param	data	data associated with key
     *
     * @return	true if an existing entry was replaced
     *
     * @exception StorageException
     *			If a problem was encountered accessing storage
     */
    public boolean put(Object key, Object data) throws StorageException {        
        beginWrite();
	try { 
	    replaced = false;
	    btreePut(key, data, REPLACE_IF_EXISTS, 0);
	    return replaced;
	} finally {
	    endWrite();
	}
    }
    
    /**
     * Replace an index entry.
     *
     * @param	key	key whose entry is to be replaced
     * @param	data	new data to be associated with key
     *
     * @exception StorageBadRequestException 
     * 			If key or data are incorrect type, or if no entry
     *			exists matching key.
     * @exception StorageException
     *			If a problem was encountered accessing storage
     */
    public void replace(Object key, Object data) throws StorageException {
        beginWrite();
	try { 
	    failed = false;
	    btreePut(key, data, REPLACE, 0);
	    if (failed) {
		throw new StorageBadRequestException (
		    MessageFormat.format("Key {0} not found in index",
			new Object[] {key}));
	    }
	} finally {
	    endWrite();
	}
    }

    /**
     * Return a collection view of this btree's values.
     *
     * @return	A collection
     * @exception StorageException
     */
    public Collection values() throws StorageException {
        return new BtreeCollection(this);
    }
    
    /**
     * Returns {@link BtreeEntryImpl} key-value pairs, where the key contains the queried prefix.
     */
    public synchronized Collection queryByKeyPrefix (Object prefix, SinglevaluedIndex primaryIndex) throws StorageException {
        if (keyType != Storage.EntryType.STRING) {
            throw new UnsupportedOperationException ("Key type must be EntryType.STRING");
        }
        
        List result = new LinkedList ();
        byte [] prefixBytes = keyInfo.toBuffer (prefix);
        SearchResult location = getLocation (prefixBytes);
        if (location.entryNum == location.page.numEntries())
            BtreePage.getNext (null, location);
        
        while ((location.entryNum < location.page.numEntries()) &&
            isPrefix (prefixBytes, location.page.getKey (location.entryNum))) {
            byte [] key = location.page.getKey (location.entryNum);
            byte [] data = location.page.getData (location.entryNum);
            Object entry = new MapEntryImpl (
                keyInfo.objectFromBuffer (key, primaryIndex), 
                dataInfo.objectFromBuffer (data, primaryIndex)
            );
            result.add (entry);
            BtreePage.getNext (null, location);
        } // while                
        return result;
    }
    
    static boolean isPrefix (byte [] prefix, byte [] key) {
        if (prefix.length > key.length)
            return false;
        for (int x = 0; x < prefix.length; x++) {
            if (prefix [x] != key [x])
                return false;
        }
        return true;
    }
    
}
