/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreeindex;

import org.netbeans.mdr.persistence.*;
import java.util.*;
import java.text.*;

/**
 * Btree implementation of the MultivaluedOrderedIndex interface.
 *
 * @author	Dana Bergen
 * @version	1.0
 */
public class MultivaluedOrderedBtree extends MultivaluedBtree 
				implements MultivaluedOrderedIndex {

    public MultivaluedOrderedBtree(String name, Storage.EntryType keyType, 
    			     Storage.EntryType dataType, 
			     boolean uniqueValues,
			     BtreePageSource pageSource) 
			     throws StorageException {
	super(name, keyType, dataType, uniqueValues, pageSource);
    }

    public MultivaluedOrderedBtree() {}

    /** Returns a list view of the values associated in the index 
     * with specified key.
     * Returned collection is read only and may not be modified.
     * If there are no values associated with the key empty collection is returned.
     * @param key
     * @return
     */
    public List getItemsOrdered(Object key) throws StorageException {
        return (List) getItems (key); // supertype always returns List, typecasting is legal
    }

      /** Like getItemsOrdered, but if the index contains keys, this returns the objects 
     * corresponding to the key
     * @return
     * @param key
     * @throws StorageException
     */
   public java.util.Collection getObjectsOrdered (Object key, SinglevaluedIndex repos) throws StorageException {
   	return getObjects (key, repos);
    }

    /** Inserts the specified element at the specified position in the list of values
     * associated with the specified key.
     * Throws StorageBadRequestException if the index is out of range.
     * @param key
     * @param index
     * @param value
     */
    public void add(Object key, int index, Object data) throws StorageException {
        beginWrite();
	try { 
            if (index < 0) {
                failed = true;
            } else {
                failed = false;
                btreePut(key, data, ADD, index);
                updateKeyModCount (key);
            }
	    if (failed) {
		throw new StorageBadRequestException(
		   MessageFormat.format("Index {0} is out of range for key {1}",
			new Object[] {new Integer(index), key}));
	    }
	} finally {
	    endWrite();
	}
    }

    /**
     * Add an item to the end of the list of values for this key.
     *
     * @param	key
     * @param	value
     */
    public void add(Object key, Object data) throws StorageException {
        /* An index of Integer.MAX_VALUE signifies the end of the list */
        add(key, Integer.MAX_VALUE, data);
    }

    /** Removes the element at the specified position in the list of values
     * associated with the specified key.
     * @return true if this index changed as a result of this call
     * @param key
     * @param index
     */
    public boolean remove(Object key, int index) throws StorageException {
        beginWrite();
	try { 
	    boolean result;
	    byte[] keyBuffer;

	    if ((keyBuffer = keyInfo.toBuffer(key)) == null) {
		throw new StorageBadRequestException(
		  MessageFormat.format(
		  "Invalid key type for this index: {0} received, {1} expected",
			    new Object[] {
				key.getClass().getName(),
				keyInfo.typeName()} ));
	    }

	    BtreePage root = pageSource.getPage(rootPageId, this);
	    result = root.remove(keyBuffer, index);
            if (result)
                updateKeyModCount (key);
	    pageSource.unpinPage(root);
	    return result;
	} finally {
	    endWrite();
	}
    }

    /** Replaces the element at the specified position in the list of values
     * associated with the specified key with the specified element.
     * Throws StorageBadRequestException if the index is out of range.
     * @param key
     * @param index
     * @param element
     * @throws StorageException
     */
    public void replace(Object key, int index, Object data) 
    						throws StorageException {                                                    
        beginWrite();
	try { 
	    failed = false;
	    btreePut(key, data, REPLACE, index);
            updateKeyModCount (key);
	    if (failed) {
		throw new StorageBadRequestException(
		   MessageFormat.format("Index {0} is out of range for key {1}",
			new Object[] {new Integer(index), key}));
	    }
	} finally {
	    endWrite();
	}
    }
}
