/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreeindex;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.persistence.Storage;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.mdr.persistence.btreeimpl.btreestorage.*;

/**
 * In-memory page source for unit testing
 */
public class BtreeCacheSource implements BtreePageSource {

    private MDRCache		cache;
    private int		pageSize;
    private EntryTypeInfo	pageIdInfo;
    private byte[]	noPageId;
    private long 	nextID = 0;
    private BtreeStorage     storage;

    public BtreeCacheSource(MDRCache cache, int pageSize, BtreeStorage storage)
			    throws StorageException {

	this.cache = cache;
	this.pageSize = pageSize;
        this.storage = storage;
	pageIdInfo = 
		EntryTypeInfo.getEntryTypeInfo(Storage.EntryType.MOFID, storage);
	noPageId = this.storage.getMOFIDData (BtreeFactory.nullMOFID);
    }

    public BtreePage getPage(byte[] pageId, Btree btree) throws StorageException {
        MOFID pageMofId = this.storage.readMOFIDData (new ByteArrayInputStream (pageId));
        return (BtreePage) cache.get(pageMofId);
    }

    public BigKeyPage newBigKeyPage(Btree btree) throws StorageException {
        BigKeyPage page = new BigKeyPage();
	initNewPage(page, btree);
	return page;
    }

    public BtreePage newPage(Btree btree) throws StorageException {

	BtreePage	page;

	page = btree.pageFactory();
	initNewPage(page, btree);
	return page;
    }

    private void initNewPage(BtreePage page, Btree btree) throws StorageException {
	MOFID pageMOFID = new MOFID(this.storage);
        byte[] mofidBytes = this.storage.getMOFIDData (pageMOFID);
	page.init(btree, mofidBytes, new byte[pageSize], true);
	cache.put(pageMOFID, page);
	cache.setDirty(pageMOFID);
    }

    public void unpinPage(BtreePage page) {
        /* nothing we need to do here */
    }

    public void unpinPage(BigKeyPage page) {
        /* nothing we need to do here */
    }

    public void dirtyPage(BtreePage page) throws StorageException {
        MOFID dirtyPageMOFId = this.storage.readMOFIDData (new ByteArrayInputStream(page.pageId));
        cache.setDirty(dirtyPageMOFId);
    }

    public BtreePage getRootPage(Btree btree) throws StorageException {
        return newPage(btree);
    }

    public EntryTypeInfo getPageIdInfo() {
        return pageIdInfo;
    }

    public int getPageIdLength() {
        return pageIdInfo.getLength();
    }

    public int getPageSize() {
        return pageSize;
    }

    /**
     * Set the passed-in pageId to contain the special value noPageId
     */
    public void setNoPage(byte[] pageId) {

        System.arraycopy(noPageId, 0, pageId, 0, pageId.length);
    }

    /**
     * Test whether the passed-in pageId contains the special value noPageId
     */
    public boolean isNoPage(byte[] pageId) {
	
	for (int i = 0; i < pageId.length; i++) {
	    if (pageId[i] != noPageId[i]) {
	        return false;
	    }
	}
	return true;
    }

    public synchronized long getNextMofid() {
		nextID++;
		return nextID;
    }

    public String getMofidPrefix() {
	    return "01234567-89AB-CDEF-GHIJ-KLMNOPQRSTUV";
    }
    
    public BtreeStorage getStorage() {
        return storage;
    }

}
