/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence;

import java.util.Collections;
import java.util.Arrays;

/** Set of indexes (Singlevalued and Multivalued) that are all updated
 * in a single transaction. Every index holds keys and every key in the index
 * is connected with a value or an ordered/unordered values.
 * The type of objects that can be stored in the index is resticted
 * by the {@link Storage.Entrytype}.
 * @author Pavel Buzek
 * @version 1.0
 */
public interface Storage {
    /** Type of values and keys stored in index.
     */
    public static final class EntryType {
        /** Values are fixed length alphanumeric strings */
	public static final EntryType MOFID = new EntryType ((byte) 1, "MOFID");

        /** Values are objects that implement the Streamable interface. 
         * This type of values may not be used by user to create a new index,
         * it is used only in the primary index created by the Storage.
         */
        public static final EntryType STREAMABLE =new EntryType ((byte) 2, "STREAMABLE");

        /** Values are variable length Strings. */
        public static final EntryType STRING =new EntryType ((byte) 3, "STRING");

        /** Values are integers. */
        public static final EntryType INT =new EntryType ((byte) 4, "INT");

	private static final EntryType[] all = new EntryType[] {MOFID, STREAMABLE, STRING, INT};

        private final byte id;
	private final String textId;
        
	private EntryType(byte id, String textId) {
	    this.id = id;
	    this.textId = textId;
	}
	
        /** Returns list of all EntryTypes.
         * @return Collection view of all available EntryTypes. 
         */
        public static java.util.Collection getEntryTypes () {
	    return Collections.unmodifiableCollection(Arrays.asList(all));
        }

        public static Storage.EntryType decodeEntryType (String name) {
            for (int i = 0; i < all.length; i++){
                if (all[i].textId.equals(name)) {
                    return all[i];
                }
            }
            return null;
        }

        public static Storage.EntryType decodeEntryType (byte code) {
	    if (code > 0 && code <= 4) return all[code-1];
            return null;
        }

	public String toString() {
            return textId;
        }
        
	public byte encode() {
            return id;
        }
    }

    public String getName();
    
    public String getStorageId ();
    
    public long getSerialNumber ();
    
    /** Reads a MOFID from Storage, must be called in the
     *  streamable read context.
     *  @return MOFID red object
     *  @exception StorageException 
     */
    public MOFID readMOFID (java.io.InputStream inputStream) throws StorageException;

    /** Writes a MOFID into Storage, must be called in the
     *  streamable write context.
     *  @param  MOFID mofid, object to be written
     *  @exception StorageException 
     */
    public void writeMOFID (java.io.OutputStream outputStream, MOFID mofid) throws StorageException;

    /** Returns the primary index in this Storage. There is exactly one primary
     * index in every storage, primary index has valueType STREAMABLE.
     */
    public SinglevaluedIndex getPrimaryIndex() throws StorageException;
    
    /* Returns true if the Storage of the given name exists */
    public boolean exists() throws StorageException;
    
    /** Delete any persistent resources associated with the Storage.
     * @return true if and only if the Storage is succesfully deleted, false otherwise
     */
    public boolean delete() throws StorageException;
    
    /* Create a new Storage instance.  For btree, this creates a new btree repository.  
     * For an RDBMS implementation, this might create a new database, or new tables and
     * indexes within an existing database.  For an implementation which uses a JMI
     * service, it might create a top-level package containing all MDR pacakges.
     * After the repository is created, it is opened.
     * If the repository already exists, and "replace" is false, an exception is
     * thrown. If replace is true, the existing repository is deleted and a new one created.
     * // PENDING: should return primary index or nothing ?
     */
    public void create (boolean replace, ObjectResolver resolver) throws StorageException;
    
    /* Open an existing repository.  If createOnNoExist is true, and the Storage
     * doesn't exist, create it and then open it.
     * // PENDING: should return primary index or nothing ?
     */
    public void open(boolean createOnNoExist, ObjectResolver resolver) throws StorageException;
    
    /* Close the Storage */
    public void close() throws StorageException;

    /** Create index that holds exactly one value for each key.
     * @return created index
     * @param name name of the index
     * @param keyType type of keys in the index
     * @param valueType type of values in the index (any type except STREAMABLE)
     */
    public SinglevaluedIndex createSinglevaluedIndex(String name, EntryType keyType, EntryType valueType) throws StorageException;
    /** Create index that holds sorted set of values for each key.
     * @return created index
     * @param name name of the index
     * @param keyType type of keys in the index
     * @param valueType type of values in the index (any type except STREAMABLE)
     * @param unique true if values associated with one key do not contain duplicates
     */
    public MultivaluedOrderedIndex createMultivaluedOrderedIndex(String name, EntryType keyType, EntryType valueType, boolean unique) throws StorageException;
    /** Create index that hold a set of values for each key. Elements in one Multivalued are
     * not sorted. Set does not contain duplicate values.
     * @return created index
     * @param name name of the index
     * @param keyType type of keys in the index
     * @param valueType type of values in the index (any type except STREAMABLE)
     * @param unique true if values associated with one key do not contain duplicates
     */
    public MultivaluedIndex createMultivaluedIndex(String name, EntryType keyType, EntryType valueType, boolean unique) throws StorageException;

    /** Retrieve index by name.
     * @param name name of the index
     * @return index of the specified name
     */
    public Index getIndex(String name) throws StorageException;
    /** Retrieve index by name.
     * @param name name of the index
     * @return index of the specified name and type
     */
    public SinglevaluedIndex getSinglevaluedIndex(String name) throws StorageException;
    /** Retrieve index by name.
     * @param name name of the index
     * @return index of the specified name and type
     */
    public MultivaluedIndex getMultivaluedIndex(String name) throws StorageException;
    /** Retrieve index by name.
     * @param name name of the index
     * @return index of the specified name and type
     */
    public MultivaluedOrderedIndex getMultivaluedOrderedIndex(String name) throws StorageException;
    /** Delete index.
     * @param name name of the index
     */
    public void dropIndex(String name) throws StorageException;

    /** Notify the Storage that state of the object associated with this key
     * will be changed.  This must be called before the change.
     * Storage can use this notification to handle information needed to
     * perform rollback. Note that the notification does not imply object
     * change will perform necessary (the planned change can be canceled), 
     * however every performed change (notified by {@link Storage#objectStateChanged})
     * is supposed to be precedesed by this notification.
     * @param key key of the object that will be changed
     */
    public void objectStateWillChange (Object key) throws StorageException;
    
    /** Notify the Storage that state of the object associated with this key
     * was changed.  This must be called after the change is made to ensure 
     * that the changed state is comitted correctly.
     * @param key key of the object that was changed and must be saved
     */
    public void objectStateChanged (Object key) throws StorageException;            
    
    /** Save all objects changed since this method was last call.
     * This operation implements transactions on the storage.
     * It must either whole complete or whole fail.
     */
    public void commitChanges() throws StorageException;

    /** Discard all changes since commitChanges() method was last call.
     * This operation implements transactions on the storage.
     * It must either whole complete or whole fail.
	 * Note that, after this method completes, the persistent MDR and
	 * any modified objects in memory are inconsistent, so it should
	 * be followed shortly by program exit.
     */
    public void rollBackChanges () throws StorageException;
    
    /**
     * Performs operations needed on exit.
     */
    public void shutDown() throws StorageException;
    
//
// Method removed, storage has always only one primary index, that is returned by 
// getPrimaryIndex method
// 
//    /** Returns true if the storage supports more than one index with type
//     * {@link Entrytype.STREAMABLE}
//     * @return true if the storage supports more than one index with type
//     * {@link Entrytype.STREAMABLE}
//     */
//    public boolean supportsMultipleStorableIndexes() throws StorageException;
}
