/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence;

import java.io.*;
import java.lang.ref.WeakReference;
import java.text.*;
import java.util.*;
import org.netbeans.mdr.persistence.*;

/**
 * MOFID implemented as a UUID for the repository plus a long for the ID.
 */
public final class MOFID extends Object {
    
    public static final int LENGTH = 8;
    public static final int STRING_LENGTH = 16;
    
    private long serialNumber;
    private WeakReference string;
    private String storageId = null;

    
    public MOFID(Storage storage) {
        this.storageId = storage.getStorageId ();
        this.serialNumber = storage.getSerialNumber ();
    }

    /** This constructor is just for internal use by storage
     *  to create MOFID after deserialization and should be
     *  newer called by user code.
     */ 
    public MOFID(long serialNumber, String storageId) {
        this.serialNumber = serialNumber;
        if (storageId == null)
            throw new IllegalArgumentException ();
        else
            this.storageId = storageId;
    }

    /** get the serial number
    */
    public long getSerialNumber() {
    	return serialNumber;
    }
    
    /** Getter for storage ID.
     * @return ID of the home storage of this MOFID.
     */
    public String getStorageID() {
        return storageId;
    }

    /** compare MOFIDs as values
    */
    public boolean equals(Object o) {
        if (o == this) return true;
	if (!(o instanceof MOFID))
	    return false;
	if (this.serialNumber != ((MOFID)o).serialNumber)
            return false;
        return this.getStorageID().equals (((MOFID)o).getStorageID());
    }

    /*** Helper Methods ***/
    
    /** MOFIDs with equal values will hash equal
    */
    public int hashCode() {
	return (int)serialNumber;
    }

    private static char makeStringBuffer[];
    /**
     * Construct the external form of a MOFID
     */
    public synchronized static String makeString(String prefix, long id) {
	int size = prefix.length();
	int totalSize = size + 1 + STRING_LENGTH;
	if (makeStringBuffer == null || makeStringBuffer.length < totalSize) {	
	    makeStringBuffer = new char[totalSize];
	}
	prefix.getChars(0, size, makeStringBuffer, 0);
	makeStringBuffer[size] = ':';
	fromLong(id, makeStringBuffer, size + 1);
	return new String(makeStringBuffer, 0, totalSize);
    }

    /**
     * Convert to external form.
     */
    public String toString() {
        String result = string == null ? null : (String) string.get();
        if (result == null) {
            result = makeString(storageId, serialNumber);
            string = new WeakReference(result);
        }
        return result;
    }

    private static int[] longwords;
    /** Convert long to 16 hex digits, to be placed in a character array.
    * @param value value to convert to hex digits
    * @param buffer array into which to place hex digits
    * @param beginning offset in array
    */
    public synchronized static void fromLong(long value, char buffer[], int offset) {
        int b;
	if (longwords == null)
	    longwords = new int[2];
	longwords[0] = (int)((value >> 32) & 0xFFFFFFFF);
	longwords[1] = (int)(value & 0xFFFFFFFF);
	for (int j = 0; j < 2; j++) {
	    for (int shift = 28; shift >= 0; shift -= 4) {
		b = (int)((longwords[j] >> shift) & 0xF);

		if (b <= 9)
		    b += 0x30;
		else
		    b += 0x37;

		buffer[offset++] = (char)b;
	    }
	}
    }	
    
    public static MOFID fromString(String mofId) {
        int pos = mofId.lastIndexOf(':');
        if (pos >= 0) try {
            MOFID result = new MOFID(Long.parseLong(mofId.substring(pos + 1), 16), mofId.substring(0, pos));
            result.string = new WeakReference(mofId);
            return result;
        } catch (NumberFormatException e) {
        } catch (IndexOutOfBoundsException e) {
        }
        return null;
    }
}
