/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.handlers;

import org.netbeans.api.mdr.events.*;

import javax.jmi.model.*;
import javax.jmi.reflect.*;

import java.util.*;

import org.netbeans.mdr.storagemodel.*;
import org.netbeans.mdr.util.DebugException;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.util.Logger;

/** Invocation handler for Instances
 *
 * @author Petr Hrebejk, Martin Matula
 * @version 2.0
 */
public abstract class InstanceHandler extends FeaturedHandler implements RefObject {

    /* -------------------------------------------------------------------- */
    /* -- Constructor ----------------------------------------------------- */
    /* -------------------------------------------------------------------- */

   /** Creates new ClassProxy */
    protected InstanceHandler(StorableObject storable) {
        super(storable);
    }

    /* -------------------------------------------------------------------- */
    /* -- Helper methods -------------------------------------------------- */
    /* -------------------------------------------------------------------- */

    private StorableObject getInstanceDelegate() {
        return (StorableObject) _getDelegate();
    }

    private void _setReference(String name, Object value) {
        Object extraInfo = null;
        boolean fail = true;
        try {
            extraInfo = _preSetR(name, value);
            _handleSetR(name, value);
            fail = false;
        } catch (NullPointerException e) {
            // reference descriptor was not found
            throw new InvalidNameException(name);
        } finally {
            _postSetR(extraInfo, fail);
        }
    }        
    
    private Object _getReference(String name) {
        Object extraInfo = null;
        Object result = null;
        boolean fail = true;
        try {
            extraInfo = _preGetR(name);
            result = _handleGetR(name);
            fail = false;
            return result;
        } catch (NullPointerException e) {
            // reference descriptor was not found
            throw new InvalidNameException(name);
        } finally {
            _postGetR(result, extraInfo, fail);
        }
    }
    
    private boolean _isReference(String name) {
        try {
            return getInstanceDelegate().getClassProxy().getReferenceDescriptor(name) != null;
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }

    /* -------------------------------------------------------------------- */
    /* -- Methods to be called by generated handlers (_pre, _handle, _post) */
    /* -------------------------------------------------------------------- */

    protected final Object _preGetR(String featureName) {
        _lock(false);
        return null;
    }
    
    protected final Object _handleGetR(String featureName) {
        StorableClass.ReferenceDescriptor reference = null;
        try {
            reference = getInstanceDelegate().getClassProxy().getReferenceDescriptor(featureName);
            AssociationHandler assoc = (AssociationHandler) _getRepository().getHandler(reference.getAssociation());
            //Logger.getDefault().log("getting reference " + featureName + " for end: " + reference.getEndName());
            return assoc._query(reference.getEndName(), (RefObject) this);
        } catch (ClassCastException e) {
            // this will throw TypeMismatchException or DebugException if the mismatch is not found
            if (reference.isFirstEnd()) {
                reference.getAssociation().checkType(null, this);
            } else {
                reference.getAssociation().checkType(this, null);
            }    
            return null;
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    protected final void _postGetR(Object result, Object extraInfo, boolean fail) {
        _unlock();
    }

    protected final Object _preSetR(String featureName, Object newValue) {
        _lock(true);
        // events firing will be handled in the association proxy
        return null;
    }
    
    protected final void _handleSetR(String featureName, Object newValue) {
        try {
            StorableClass.ReferenceDescriptor reference = getInstanceDelegate().getClassProxy().getReferenceDescriptor(featureName);
            AssociationHandler assoc = (AssociationHandler) _getRepository().getHandler(reference.getAssociation());
            
            // get old value of the reference
            Object oldValue = _getReference(featureName);

            // remove old link and add the new one
            if (reference.isFirstEnd()) {
                if (oldValue != null) try {
                    assoc._remove((RefObject) oldValue, (RefObject) this);
                } catch (ClassCastException e) {
                    // this will throw TypeMismatchException or DebugException if the mismatch is not found
                    ((StorableAssociation) assoc._getDelegate()).checkType(oldValue, this);
                    // this is here only to make the compiler happy
                    return;
                }
                if (newValue != null) try {
                    assoc._add((RefObject) newValue, (RefObject) this);
                } catch (ClassCastException e) {
                    // this will throw TypeMismatchException or DebugException if the mismatch is not found
                    ((StorableAssociation) assoc._getDelegate()).checkType(newValue, this);
                    // this is here only to make the compiler happy
                    return;
                }
            } else {
                if (oldValue != null) try {
                    assoc._remove((RefObject) this, (RefObject) oldValue);
                } catch (ClassCastException e) {
                    // this will throw TypeMismatchException or DebugException if the mismatch is not found
                    ((StorableAssociation) assoc._getDelegate()).checkType(this, oldValue);
                    // this is here only to make the compiler happy
                    return;
                }
                if (newValue != null) try {
                    assoc._add((RefObject) this, (RefObject) newValue);
                } catch (ClassCastException e) {
                    // this will throw TypeMismatchException or DebugException if the mismatch is not found
                    ((StorableAssociation) assoc._getDelegate()).checkType(this, newValue);
                    // this is here only to make the compiler happy
                    return;
                }
            }
            
            // [PENDING] should add replace operation into an association
            // and call it in case when both oldValue and newValue are not null so that
            // the operation will be atomic and replace event can be thrown instead
            // of add and remove events
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    protected final void _postSetR(Object extraInfo, boolean fail) {
        // events firing was handled in the association proxy
        _unlock();
    }

    protected final Object _handleGet(int attrIndex) {
        Object result;
        
        try {
            result = getInstanceDelegate().getAttribute(attrIndex);
            if (result instanceof Collection) {
                if (result instanceof AttrList) {
                    result = new AttrListWrapper(this, attrIndex, null);
                } else if (result instanceof AttrCollection) {
                    result = new AttrCollWrapper(this, attrIndex, null);
                } else if (result instanceof AttrImmutList) {
                    result = new AttrImmutListWrapper(_getMdrStorage(), this, attrIndex, null);
                }
            }
            return result;
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    protected final void _handleSet(int attrIndex, Object newValue) {
        try {
            getInstanceDelegate().setAttribute(attrIndex, newValue);
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }

    // MOF specific Handling methods ---------------------------------------------

    /* -------------------------------------------------------------------- */
    /* -- Implementation of javax.jmi.reflect.RefObject ------------------- */
    /* -------------------------------------------------------------------- */

    public final RefFeatured refOutermostComposite() {
        _lock(false);
        try {
            return _outermostComposite();
        } finally {
            _unlock();
        }
    }
    
    protected RefFeatured _outermostComposite() {
        try {
            return (RefFeatured) _getRepository().getHandler(getInstanceDelegate().getOutermostComposite());
        } catch ( StorageException e ) {
            throw new DebugException("Storage exception: " + e);
        }
    }

    public final RefFeatured refImmediateComposite() {
        _lock(false);
        try {
            return _immediateComposite();
        } finally {
            _unlock();
        }
    }

    protected RefFeatured _immediateComposite() {
        try {
            return (RefFeatured) _getRepository().getHandler(getInstanceDelegate().getImmediateComposite());
        } catch ( StorageException e ) {
            throw new DebugException("Storage exception: " + e);
        }
    }
    
    public final boolean refIsInstanceOf(RefObject objType, boolean considerSubtypes) {
        _lock(false);
        try {
            return isInstanceOf((GeneralizableElement) refMetaObject(), objType, considerSubtypes);
        } finally {
            _unlock();
        }
    }

    private final boolean isInstanceOf(GeneralizableElement metaObject, RefObject objType, boolean considerSubtypes) {
        if (metaObject.equals(objType)) return true;
        if (considerSubtypes) {
            boolean isInstance;
            Iterator it = metaObject.getSupertypes().iterator();
            while (it.hasNext()) {
                if (isInstanceOf(((GeneralizableElement) it.next()), objType, true)) {
                    return true;
                }
            }
        }
        return false;
    }

    public final RefClass refClass() {
        _lock(false);
        try {
            return (RefClass) _getRepository().getHandler(getInstanceDelegate().getClassProxy());
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        } finally {
            _unlock();
        }
    }

    public final void refDelete() {
        boolean fail = true;
        InstanceEvent event = null;
        _lock(true);
        try {
            if (_getMdrStorage().eventsEnabled()) {
                event = new InstanceEvent(
                    this,
                    InstanceEvent.EVENT_INSTANCE_DELETE,
                    null,
                    this
                );

                _getMdrStorage().getEventNotifier().INSTANCE.firePlannedChange(this, event);
            }
            _delete();
            Exception deletionStackTrace = new Exception("mofId: " + refMofId() + ", class: " + getClass().getName());
            BaseObjectHandler.deletedInstances.put(this, deletionStackTrace);
            fail = false;
        } finally {
            _unlock(fail);
        }
        _getRepository().removeHandler(_getMofId());
    }
    
    protected void _delete() {
        try {
            getInstanceDelegate().delete();
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    };

    public final void refSetValue(String featureName, java.lang.Object value) {
        _lock(true);
        try {
            if (_isReference(featureName)) {
                _setReference(featureName, value);
            } else {
                super.refSetValue(featureName, value); 
            }
        } finally {
            _unlock();
        }
    }
    
    public final void refSetValue(RefObject feature, Object value) {
        if (feature instanceof Attribute) {
            super.refSetValue(feature, value);
        } else {
            _lock(true);
            try {
                StorableClass cls = getInstanceDelegate().getClassProxy();
                try {
                    StorableClass.ReferenceDescriptor desc = cls.getReferenceDescriptor(((Reference) feature).getName());
                    if (!desc.getMofId().equals(((BaseObjectHandler)feature)._getDelegate().getMofId())) throw new InvalidCallException(null, feature);
                } catch (DebugException e) {
                    throw new InvalidCallException(null, feature);
                }
                _setReference(((Reference) feature).getName(), value);
            } catch (InvalidNameException e) {
                throw new InvalidCallException(null, feature);
            } catch (ClassCastException e) {
                throw new InvalidCallException(null, feature);
            } catch (StorageException e) {
                throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
            } finally {
                _unlock();
            }
        }
    }

    public final Object refGetValue(String featureName) {
        _lock(false);
        try {
            if (_isReference(featureName)) {
                return _getReference(featureName);
            } else {
                return super.refGetValue(featureName);
            }
        } finally {
            _unlock();
        }
    }
    
    public final Object refGetValue(RefObject feature) {
        if (feature instanceof Attribute) {
            return super.refGetValue(feature);
        } else {
            _lock(false);
            try {
                StorableClass cls = getInstanceDelegate().getClassProxy();
                try {
                    StorableClass.ReferenceDescriptor desc = cls.getReferenceDescriptor(((Reference) feature).getName());
                    if (desc == null || !desc.getMofId().equals(((BaseObjectHandler)feature)._getDelegate().getMofId())) throw new InvalidCallException(null, feature);
                } catch (DebugException e) {
                    throw new InvalidCallException(null, feature);
                }
                return _getReference(((Reference) feature).getName());
            } catch (InvalidNameException e) {
                throw new InvalidCallException(null, feature);
            } catch (ClassCastException e) {
                throw new InvalidCallException(null, feature);
            } catch (StorageException e) {
                throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
            } finally {
                _unlock();
            }
        }
    }

    /* -------------------------------------------------------------------- */
    /* -- Implementation of org.netbeans.api.mdr.events.MDRChangeSource --- */
    /* -------------------------------------------------------------------- */
    
    /** Registers a listener for receiving all event notifications
     *  fired on this object.
     * @param listener Object that implements {@link MDRChangeListener} interface.
     */
    public void addListener(MDRChangeListener listener) {
        addListener(listener, MDRChangeEvent.EVENTMASK_ALL);
    }
    
    /** Registers a listener for receiving event notifications.
     * @param listener Object that implements {@link MDRChangeListener} interface.
     * @param mask bitmask to filter types of events the listener listens on
     */
    public void addListener(MDRChangeListener listener, int mask) {
        _getMdrStorage().getEventNotifier().INSTANCE.addListener(listener, mask, this);
    }
    
    /** Removes listener from the list of objects registered for event notifications.
     * @param listener Object that implements {@link MDRChangeListener} interface.
     */
    public void removeListener(MDRChangeListener listener) {
        _getMdrStorage().getEventNotifier().INSTANCE.removeListener(listener, this);
    }
    
    /** Removes listener from the list of objects registered for event notifications.
     * @param listener Object that implements {@link MDRChangeListener} interface.
     * @param mask determines type of the events the listeners stops to listen on
     */
    public void removeListener(MDRChangeListener listener, int mask) {
        _getMdrStorage().getEventNotifier().INSTANCE.removeListener(listener, mask, this);
    }

    /** Returns all listeners registered for event notification on this object.
     * @return collection of registered listeners
     */
    public Collection getListeners() {
        return _getMdrStorage().getEventNotifier().INSTANCE.getListeners(this);
    }
        
    /* ---------------------------------------------------------------- */
    /* -- Implementation of abstract methods from BaseObjectHandler --- */
    /* ---------------------------------------------------------------- */

    protected final Collection _recursiveVerify(Collection violations, Set visited) {
        _lock(false);
        try {
            _verify(violations);
            visited.add(this);
            // [PENDING] should verify all contained objects
            return violations;
        } finally {
            _unlock();
        }
    }    

    protected Collection _verify(Collection violations) {
        _lock(false);
        try {
            getInstanceDelegate().verify(violations);
            return violations;
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        } finally {
            _unlock();
        }
    }
}
