/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.handlers;

import java.util.*;
import java.lang.reflect.Array;
import javax.jmi.reflect.RefBaseObject;

import org.netbeans.mdr.storagemodel.*;
import org.netbeans.mdr.util.DebugException;

/**
 * Wraps collection of storage content with handlers.
 *
 * @author Martin Matula
 */
public class IndexSetWrapper implements Collection {
    protected final Collection inner;
    protected final MdrStorage storage;
    
    /** Creates new CollectionWrapper */
    public IndexSetWrapper(MdrStorage mdrStorage, Collection inner) {
        this.inner = inner;
        this.storage = mdrStorage;
    }
    
    protected void lock(boolean writeAccess) {
        storage.getRepository().beginTrans(writeAccess);
    }
    
    protected void unlock() {
        storage.getRepository().endTrans();
    }
    
    protected void unlock(boolean fail) {
        storage.getRepository().endTrans(fail);
    }
    
    protected static Object wrap(Object obj) {
        if (obj == null) {
            return null;
        } else if (obj instanceof StorableBaseObject) {
            return ((StorableBaseObject) obj).getMdrStorage().getRepository().getHandler((StorableBaseObject) obj);
        } else if (obj instanceof AssociationLink) {
            return AssociationLinkWrapper.wrapLink((AssociationLink) obj);
        } else {
            throw new DebugException("Invalid object in collection: " + obj.getClass().getName());
        }
    }
    
    protected static Object unwrap(Object obj) {
        if (obj == null) {
            return null;
        } else if (obj instanceof RefBaseObject) {
            return ((BaseObjectHandler) obj)._getDelegate().getMofId();
        } else if (obj instanceof AssociationLinkWrapper) {
            return ((AssociationLinkWrapper) obj).getInnerLink();
        } else {
            return obj;
        }
    }    
    
    protected static Object[] wrapArray(Object[] array) {
        for (int i = 0; i < array.length; i++) {
            array[i] = wrap(array[i]);
        }
        
        return array;
    }
    
    public boolean contains(Object obj) {
        lock(false);
        try {
            return inner.contains(unwrap(obj));
        } finally {
            unlock();
        }
    }
    
    public Iterator iterator() {
        lock(false);
        try {
            return new IndexIteratorWrapper(inner.iterator());
        } finally {
            unlock();
        }
    }
    
    public int size() {
        try {
            lock(false);
            return inner.size();
        } finally {
            unlock();
        }
    }
    
    public boolean isEmpty() {
        lock(false);
        try {
            return inner.isEmpty();
        } finally {
            unlock();
        }
    }
    
    public boolean containsAll(Collection collection) {
        lock(false);
        try {
            return inner.containsAll(new CollectionUnwrapper(collection));
        } finally {
            unlock();
        }
    }
    
    public Object[] toArray(Object[] obj) {
        lock(false);
        try {
            Object[] value = wrapArray(inner.toArray());
            Object[] result = obj;
            if (value.length > result.length) {
                if (value.getClass() == result.getClass()) {
                    return value;
                } else {
                    result = (Object[]) Array.newInstance(obj.getClass().getComponentType(), 
                        value.length);
                }
            } else if (value.length < result.length) {
                result[value.length] = null;
            }
            System.arraycopy(value, 0, result, 0, value.length);
            return result;
        } finally {
            unlock();
        }
    }
    
    public boolean equals(Object object) {
        if (object == this) return true;
        if (!(object instanceof Collection)) return false;
        lock(false);
        try {
            Iterator it1 = iterator();
            Iterator it2 = ((Collection) object).iterator();
            while(it1.hasNext() && it2.hasNext()) {
                Object o1 = it1.next();
                Object o2 = it2.next();
                if (!(o1==null ? o2==null : o1.equals(o2)))
                    return false;
            }
            return !(it1.hasNext() || it2.hasNext());
        } finally {
            unlock();
        }
    }
    
    public int hashCode() {
        lock(false);
        try {
            int hashCode = 1;
            for (Iterator it = iterator(); it.hasNext();) {
                Object obj = it.next();
                hashCode = 31*hashCode + (obj==null ? 0 : obj.hashCode());
            }
            return hashCode;
        } finally {
            unlock();
        }
    }

    public Object[] toArray() {
        return toArray(new Object[size()]);
    }
    
    public void clear() {
        throw new UnsupportedOperationException();
    }
    
    public boolean addAll(Collection collection) {
        throw new UnsupportedOperationException();
    }
    
    public boolean remove(Object obj) {
        throw new UnsupportedOperationException();
    }
    
    public boolean add(Object obj) {
        throw new UnsupportedOperationException();
    }
    
    public boolean retainAll(Collection collection) {
        throw new UnsupportedOperationException();
    }
    
    public boolean removeAll(Collection collection) {
        throw new UnsupportedOperationException();
    }
    
    protected class IndexIteratorWrapper implements Iterator {
        protected Iterator innerIterator;
        
        public IndexIteratorWrapper(Iterator innerIterator) {
            this.innerIterator = innerIterator;
        }
        
        public boolean hasNext() {
            lock(false);
            try {
                return innerIterator.hasNext();
            } finally {
                unlock();
            }
        }
        
        public Object next() {
            lock(false);
            try {
                return wrap(innerIterator.next());
            } finally {
                unlock();
            }
        }
        
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}
