/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdrshell;

import java.util.*;
import java.io.*;

import org.openide.util.Lookup;
import org.netbeans.api.mdr.*;

import javax.jmi.reflect.*;
import javax.jmi.model.ModelElement;

import org.netbeans.mdr.storagemodel.*;
import org.netbeans.mdr.handlers.BaseObjectHandler;

/** Shell for metadata repository.
 *
 * @author Petr Hrebejk
 * @version 
 */
public class Shell extends Object {
 
    private static final char CONTINUE_CHAR = '\\';
    private static final char NEW_LINE = '\n';
    
    /** Holds the preprocesor for this Shell */
    private Preprocessor preprocessor;
    
    /** Value indicating wether shell should exit */
    private boolean exit; 

    /** Stack of streams to read from */
    private Stack ioStack;
    
    /** Print stream used for this shell. Note that this variable can be
    * changed from inside the shell. 
    */
    public PrintStream out;
    
    /** Says wether the result of commands should be printed */
    private boolean resultPrint;
    
    /** Should the commands be echoed */
    private boolean commandEcho;
    
    /** String used for prompting the user */
    private String prompt;
    
    private Object current;

    /** Creates new shell 
    */
    public Shell() {
        preprocessor = new Preprocessor( this );
        exit = false;
        out = System.out;
        ioStack = new Stack();
        resultPrint = false;
        commandEcho = true;
        current = null;
    }
    
    private void execCommand(String command) throws Exception {
        String preprocessedLine = preprocessor.processLine( command.toString() );

        //Support.defaultPrint( 0, Interpreter.processLine( preprocessedLine ) );

        if ( commandEcho ) {
            out.println( preprocessedLine );
        }

        Object result = Interpreter.processLine( preprocessedLine );
        if ( resultPrint ) {
            Support.defaultPrint( out, result );
        }
    }

    // Methods to be used from inside the shell --------------------------------
    
    /** Exits the shell 
    */
    public void exit() {
        System.out.println( "Exiting" );
        exit = true;
    }
    
    /** Prints object in default format 
    */
    public void print( Object o ) {
        Support.defaultPrint( out, o );
    }
    
    public void print( boolean  what ) {
        out.println( what );
    }
    
    public void print( byte  what ) {
        out.println( what );
    }
    
    public void print( short  what ) {
        out.println( what );
    }
    
    public void print( char  what ) {
        out.println( what );
    }
    
    public void print( int  what ) {
        out.println( what );
    }
    
    public void print( long  what ) {
        out.println( what );
    }
    
    public void print( float  what ) {
        out.println( what );
    }
    
    public void print( double  what ) {
        out.println( what );
    }
    
    public Object setActive(Object object) throws Exception {
        retype(object);
        return object;
    }
    
    public void retype(Object object) throws Exception {
        execCommand("#define C$ ((" + object.getClass().getName() + ")CurrentObject)");
    }    
    
    public MDRepository getRepository() throws Exception {
        return MDRManager.getDefault().getDefaultRepository();
    }
    
    /** get the current prompt 
    */
    public String getPrompt() {
        return prompt;
    }
    
    
    /** Set the string for prompting 
    */
    public void setPrompt( String prompt ) {
        this.prompt = prompt;
    }
    
    /** Gets the of current command echoing 
    */
    public boolean isCommandEcho() {
        return commandEcho;
    }
    
    /** Sets the command echo 
    */
    public void setCommandEcho( boolean commandEcho ) {
        this.commandEcho = commandEcho;
    }
    
    /** Gets the value of result printing
    */
    public boolean isResultPrint() {
        return resultPrint;
    }
    
    /** Sets the command echo 
    */
    public void setResultPrint( boolean resultPrint ) {
        this.resultPrint = resultPrint;
    }
    
    
    // Public methods ----------------------------------------------------------
    
    /** Starts interpretting commands in the InputStream.
    * @param is InputStream to read commands from. If <code>null</code>
     * <code>System.in</code> is used.
    */
    public void run( InputStream is ) {
        run( new InputStream[] { is } );
    }
    
    /** Starts interpretting commands in the InputStreams.
    * @param iss InputStreams to read commands from. If <code>null</code>
    * <code>System.in</code> is used.
    */
    public void run( InputStream[] iss ) {

        if ( iss == null ) {
            pushInput(System.in);
        } else {
            for ( int i = iss.length-1; i >= 0; i-- ) {
                pushInput(iss[i]);
            }
        }
 
        DJava.initialize();
        DJava.declareVariable( "SHELL", this );
        
        StringBuffer commandBuffer = new StringBuffer();
        while ( !exit ) {
            try {                
                if ( getPrompt() != null && ioStack.size() == 1 ) {
                    out.print( getPrompt() );
                }
                
                String line = ((BufferedReader)ioStack.peek()).readLine();
                
                if ( line == null ) {
                    ioStack.pop();
                    continue;
                }
                if ( line.equals( "" ) || line.trim().startsWith( "//" ) ) {
                    continue;
                }
                
                commandBuffer.append( line );
                
                if ( line.charAt( line.length() - 1 ) == CONTINUE_CHAR ) {    
                    commandBuffer.deleteCharAt( commandBuffer.length() - 1 ).append( NEW_LINE );
                    continue;
                }
                
                String command = commandBuffer.toString();
                commandBuffer = new StringBuffer();
     
                execCommand(command.toString());
            }
            catch ( Throwable e ) {
                e.printStackTrace( out );
            }
           
        }
    }

    // Package private methods -------------------------------------------------

    void pushInput( Reader reader ) {
        ioStack.push( new BufferedReader( reader ) );
    }
    
    void pushInput( InputStream is ) {
        ioStack.push( new BufferedReader( new InputStreamReader( is ) ) );
    }
               
       
}
