/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.mdrxml.util;

import javax.jmi.reflect.*;
import xmlmodel.*;

import java.util.*;
import java.io.*;

public class XMLGenerator {
    
    private static final char QUOTE = '\'';
    private static final String INDENT = "  ";
    private static final int INDENT_LENGTH = INDENT.length ();
    /** An upper bound on the number of characters that can be printed to the output stream 
     * per one line. A new line is started every time when the number of characters in a line
     * (excluding indent spaces) after @link #startElement or @link #addAttribute exceeds MAX.
     */
    private static final int MAX = 70;
    
    // output stream
    private PrintStream stream;
    // true if the currently written XML element has at least one sub-element
    private boolean hasContent = true; // inited by true due to condition in @link #startElement
    // true if the currently written XML element has some characters in its content
    private boolean hasCharacters = false;
    // indentation spaces to be currently used
    private String indentSpaces = "";
    // number of charecters written on current line so far (excluding indent spaces)
    private int charsCount = 0;
    
    // **************************************************************************
    // Methods related to writing to the output stream.
    // **************************************************************************
    
    /**
     * Writes a string to the output string.
     *
     * @param text a string to be written in the output stream
     */
    private void write (String text) {
        stream.print (text);
    }
    
    /**
     * Writes end of line to the output stream.
     */
    private void writeln () {
        stream.println ();
        charsCount = 0;
    }

    /**
     * Locates occurences of special XML charecters (like '<', '&', etc.) in a string
     * and replaces them by sequences of the form &X...X;
     */
    private String replaceSpecialChars (String s) {
        int length = s.length ();
        char [] chars = new char [6 * length];
        int n = 0;
        for (int x = 0; x < length; x++) {
            char c = s.charAt (x);
            switch (c) {
                case '&':
                    chars [n] = '&'; n++;   chars [n] = 'a'; n++;
                    chars [n] = 'm'; n++;   chars [n] = 'p'; n++;
                    chars [n] = ';'; n++; 
                break;
                case '\'':
                    chars [n] = '&'; n++;   chars [n] = 'a'; n++;
                    chars [n] = 'p'; n++;   chars [n] = 'o'; n++;
                    chars [n] = 's'; n++;   chars [n] = ';'; n++;
                break;
                case '\"':
                    chars [n] = '&'; n++;   chars [n] = 'q'; n++;
                    chars [n] = 'u'; n++;   chars [n] = 'o'; n++;
                    chars [n] = 't'; n++;   chars [n] = ';'; n++;
                break;
                case '<':
                    chars [n] = '&'; n++;   chars [n] = 'l'; n++;
                    chars [n] = 't'; n++;   chars [n] = ';'; n++;
                break;
                case '>':
                    chars [n] = '&'; n++;   chars [n] = 'g'; n++;
                    chars [n] = 't'; n++;   chars [n] = ';'; n++;
                break;
                default:
                    chars [n] = c; n++;
            } // switch
        } // for
        return new String (chars, 0, n);
    }
    
    /**
     * Writes start of a XML element to the output stream.
     *
     * @param name name of the XML element to be written
     */
    private void startElement (String name) {
        if (!hasContent && !hasCharacters) {
            write (">");
            writeln ();
        }
        hasContent = false;
        hasCharacters = false;
        write (indentSpaces);
        write ("<" + name);        
        charsCount += name.length () + 1;
        indentSpaces = indentSpaces + INDENT;
    }
    
    /**
     * Writes end of an XML element to the output stream.
     *
     * @param name name of the XML element to be written
     */
    private void endElement (String name) {
        indentSpaces = indentSpaces.substring (0, indentSpaces.length () - INDENT_LENGTH);
        if (hasContent) {
            write (indentSpaces);
            write ("</" + name + ">");
        } else if (hasCharacters) {
            write ("</" + name + ">");
        } else
            write ("/>");
        writeln ();
        hasContent = true;
    }
    
    /**
     * Writes an attribute of the currenly written XML elemnt to the output stream.
     *
     * @param name attribute name
     * @param value value of the attribute
     */
    private void addAttribute (String name, String value) {
        value = replaceSpecialChars (value);
        // [PENDING] ??? can be special characters in name too ???
        if (charsCount > MAX) {
            writeln ();
            write (indentSpaces);
        } else {
            write (" ");
            charsCount++;
        }
        write (name + " = " + QUOTE + value + QUOTE);
        charsCount += name.length () + value.length () + 5;
    }
    
    /**
     * Writes characters into body of the currenly written XML elemnt.
     * Before the string is written, @link #replaceSpecialChars is called
     * on it to replace special XML characters.
     *
     * @param text string to be written
     */
    private void characters (String text) {
        text = replaceSpecialChars (text);
        if (!hasContent)
            write (">");
        write (text);        
        hasCharacters = true;
    }
    
    // **************************************************************************
        
    // XML Generation
    public void generateXML (OutputStream os, ElementNode elementNode) {
        stream = new PrintStream (os);
        write ("<?xml version = '1.0' encoding = 'ISO-8859-1' ?>\n");
        generateElement (elementNode);
    }
    
    private void generateElement (ElementNode element) {
        String name = element.getName ();
        startElement (name);
        List nodes = element.getNodes ();
        Iterator iter = nodes.iterator ();
        while (iter.hasNext ()) {
            Object subNode = iter.next ();
            if (subNode instanceof AttributeNode) {
                AttributeNode attrNode = (AttributeNode) subNode;
                addAttribute (attrNode.getName (), attrNode.getValue ());
            } // if
        } // while
        iter = nodes.iterator ();
        while (iter.hasNext ()) {
            Object subNode = iter.next ();
            if (subNode instanceof TextNode)
                characters (((TextNode) subNode).getName ());
            else if (subNode instanceof ElementNode)
                generateElement ((ElementNode) subNode);
        }
        endElement (name);
    }
    
}