/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.jdbcimpl;

import org.netbeans.mdr.persistence.btreeimpl.btreestorage.*;
import org.netbeans.mdr.persistence.*;

import java.io.*;
import java.text.*;
import java.util.*;

/**
 * JdbcPrimaryIndex implements primary index storage using JDBC.
 *
 * @author John V. Sichi
 * @version $Id: JdbcPrimaryIndex.java,v 1.5.58.1 2006/07/01 05:19:21 jtulach Exp $
 */
class JdbcPrimaryIndex
    extends JdbcSinglevaluedIndex implements MDRCache.OverflowHandler
{
    // NOTE:  around cache access sequences, we synchronize on
    // the parent storage, not on "this", to prevent deadlock
    private final MDRCache cache;

    // NOTE:  most of this class was ripped from BtreeDatabase
    static final int MDR_CACHE_SIZE = 1024;
    static final int MDR_CACHE_THRESHHOLD = 1000;
    
    JdbcPrimaryIndex()
    {
        cache = new MDRCache(
            MDR_CACHE_SIZE,
            this,
            MDR_CACHE_THRESHHOLD,
            null);
    }

    // implement MDRCache.OverflowHandler
    public void cacheThreshholdReached(MDRCache cache, int size) 
        throws StorageException
    {
        flushChanges();
    }

    void shutDown()
    {
        cache.shutDown();
    }

    void flushChanges()
        throws StorageException
    {
        // TODO:  optimize with batched JDBC calls where supported

        try {
            int modStatus = cache.getModStatus();
            MOFID id;
            Object value;
            while (modStatus != 0) {
                // First, persistently delete everything that needs it
                if ((modStatus & MDRCache.M_DELETED) != 0) {
                    Iterator delIter = cache.getDeleted().iterator();
                    while (delIter.hasNext()) {
                        id = (MOFID)delIter.next();
                        super.removeImpl(id);
                    }
                }

                // Next, modify what needs modification
                if ((modStatus & MDRCache.M_DIRTY) != 0) {
                    Iterator dirtyIter = cache.getDirty().iterator();
                    while (dirtyIter.hasNext()) {
                        Map.Entry entry = (Map.Entry) dirtyIter.next();
                        id = (MOFID) entry.getKey();
                        value = entry.getValue();
                        super.replaceImpl(id,value);
                    }
                }
            
                // Last, add what needs adding
                if ((modStatus & MDRCache.M_NEW) != 0) {
                    Iterator newIter = cache.getNew().iterator();
                    while (newIter.hasNext()) {
                        Map.Entry entry = (Map.Entry) newIter.next();
                        id = (MOFID) entry.getKey();
                        value = entry.getValue();
                        super.addImpl(id,value);
                    }
                }
                
                modStatus = cache.getModStatus();
            }
        } finally {
            cache.updateSize();
        }
    }

    private MOFID makeMOFID(Object key) throws StorageException
    {
        if (key instanceof MOFID) {
            return (MOFID) key;
        } else {
            throw new IllegalArgumentException(
                "Argument must be of org.netbeans.mdr.persistence.MOFID type");
        }
    }

    void objectStateChanged(Object key) throws StorageException
    {
        cache.setDirty(makeMOFID(key));
    }

    private boolean exists(MOFID key) throws StorageException
    {
        if (cache.get(key) != null) {   
            return true;
        } else if (cache.isDeleted(key)) {
            return false;
        } else {
            return super.getIfExists(key) != null;
        }
    }
    
    private void noSuchRecord(Object key) throws StorageException
    {
        throw new StorageBadRequestException(
            MessageFormat.format("No record exists with key {0}",
                new Object[] {key} ) );
    }
    
    private void addToCache(MOFID key, Object value) throws StorageException
    {
        cache.put(key, value);
        cache.setNew(key);
    }
    
    private void replaceInCache(MOFID key, Object value)
        throws StorageException
    {
        boolean isNew = cache.isNew(key);
        
        cache.replace(key, value);
        
        if (isNew) {
            cache.setNew(key);
        } else {
            cache.setDirty(key);
        }
    }
    
    // override JdbcSinglevaluedIndex
    public boolean put(
        Object key,Object value) throws StorageException
    {
        synchronized(storage) {
            MOFID mKey = makeMOFID(key);
            if (!exists(mKey)) {
                addToCache(mKey,value);
                return false;
            } else {
                replaceInCache(mKey,value);
                return true;
            }
        }
    }
    
    // override JdbcSinglevaluedIndex
    public void replace(Object key, Object value)
        throws StorageException, StorageBadRequestException
    {
        synchronized(storage) {
            MOFID mKey = makeMOFID(key);
        
            if (!exists(mKey)) {
                noSuchRecord(mKey);
            }

            replaceInCache(mKey, value);
        }
    }
    
    // override JdbcSinglevaluedIndex
    public Object get(Object key)
        throws StorageException, StorageBadRequestException
    {
        Object retval = getIfExists(key);
        if (retval == null) {
            noSuchRecord(key);
        }

        return retval;
    }

    // override JdbcSinglevaluedIndex
    public Object getObject(Object key, SinglevaluedIndex repos)
        throws StorageException
    {
        return get(key);
    }

    // override JdbcSinglevaluedIndex
    public Object getIfExists(Object key) throws StorageException
    {
        synchronized(storage) {
            MOFID mKey = makeMOFID(key);
            Object retval;
            retval = cache.get(mKey);
            if (retval == null) {
                if (cache.isDeleted(mKey)) {
                    return null;
                }
                retval = super.getIfExists(mKey);
                if (retval != null) {
                    cache.put(mKey, retval);
                }
            }

            return retval;
        }
    }
    
    // override JdbcSinglevaluedIndex
    public Object getObjectIfExists(Object key, SinglevaluedIndex repos)
        throws StorageException
    {
        return getIfExists(key);
    }

    // NOTE:  it's a pain to implement keySet() and values() correctly,
    // and as far as I can tell they're never used for primary indexes
    
    // override JdbcSinglevaluedIndex
    public Collection values()
        throws StorageException
    {
        throw new RuntimeException("oops, not yet implemented");
    }
    
    // override JdbcIndex
    public Set keySet() throws StorageException
    {
        throw new RuntimeException("oops, not yet implemented");
    }

    // override JdbcIndex
    public void add(
        Object key, Object value) throws StorageException
    {
        synchronized(storage) {
            MOFID mKey = makeMOFID(key);
            if (exists(mKey)) {
                throw new StorageBadRequestException(
                    MessageFormat.format("Record with key {0} already exists",
                        new Object[] {mKey} ) );
            }
            addToCache(mKey, value);
        }
    }
    
    // override JdbcIndex
    public boolean remove(Object key) throws StorageException
    {
        synchronized(storage) {
            MOFID mKey = makeMOFID(key);
            if (!exists(mKey)) {
                return false;
            } else {
                cache.remove(mKey);
                return true;
            }
        }
    }
}

// End JdbcPrimaryIndex.java
