/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.jdbcimpl;

import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.util.*;

import java.util.*;
import java.io.*;
import java.text.*;
import java.sql.*;

/**
 * JdbcMultivaluedIndex implements the MDR MultivaluedIndex interface using
 * JDBC.
 *
 * @author John V. Sichi
 * @version $Id: JdbcMultivaluedIndex.java,v 1.5.64.1 2006/07/01 05:19:21 jtulach Exp $
 */
class JdbcMultivaluedIndex
    extends JdbcIndex implements MultivaluedIndex
{
    protected boolean queryDuplicates;
    
    protected LazyPreparedStatement sqlDeleteWithValue;
    protected LazyPreparedStatement sqlDeleteWithSurrogate;
    protected LazyPreparedStatement sqlFindCount;
    protected LazyPreparedStatement sqlFindCountWithValue;
    protected LazyPreparedStatement sqlFindSurrogate;
    
    protected void defineSql()
    {
        super.defineSql();
        
        sqlDeleteWithValue = new LazyPreparedStatement(
            "delete from " + tableName
            + " where " + keyColName + " = ?"
            + " and " + valColName + " = ?");
        
        sqlDeleteWithSurrogate = new LazyPreparedStatement(
            "delete from " + tableName
            + " where " + keyColName + " = ?"
            + " and " + JdbcStorage.SURROGATE_COL_NAME + " = ?");
        
        sqlFindCount = new LazyPreparedStatement(
            "select count(*) from " + tableName
            + " where " + keyColName + " = ?");

        sqlFindCountWithValue = new LazyPreparedStatement(
            "select count(*) from " + tableName
            + " where " + keyColName + " = ?"
            + " and " + valColName + " = ?");

        sqlFindSurrogate = new LazyPreparedStatement(
            "select " + JdbcStorage.SURROGATE_COL_NAME
            + " from " + tableName
            + " where " + keyColName + " = ?"
            + " and " + valColName + " = ?");
    }
    
    // implement MultivaluedIndex
    public Collection getItems(Object key) throws StorageException
    {
        return new ItemCollection(key,null);
    }
    
    // implement MultivaluedIndex
    public Collection getObjects(
        Object key, SinglevaluedIndex repos) throws StorageException
    {
        if (keyType == Storage.EntryType.MOFID) {
            return new ItemCollection(key,repos);
        } else {
            return getItems(key);
        }
    }
    
    // implement MultivaluedIndex
    public boolean isUnique() throws StorageException
    {
        return !needSurrogate;
    }

    // override JdbcIndex
    public void add(Object key, Object value) throws StorageException
    {
        if (queryDuplicates) {
            int n = storage.getSingletonInt(
                sqlFindCountWithValue,
                new Object [] { key, value } );
            if (n > 0) {
                throw new StorageBadRequestException("duplicate detected");
            }
        }
        addImpl(key, value);
    }

    // implement MultivaluedIndex
    public boolean remove(Object key, Object value) throws StorageException
    {
        Object [] pair = new Object[]{key,value};

        int rowCount;
        if (isUnique()) {
            // since this index is unique, don't need to worry about
            // duplicate handling
            rowCount = storage.executeUpdate(sqlDeleteWithValue,pair);
        } else {
            // there may be duplicates, and we're only supposed to delete
            // one of them
            int surrogateKey = storage.getSingletonInt(sqlFindSurrogate,pair);
            if (surrogateKey == -1) {
                return false;
            }
            rowCount = storage.executeUpdate(
                sqlDeleteWithSurrogate,
                new Object[]{key,new Integer(surrogateKey)});
        }
        return rowCount > 0;
    }
    
    // implement MultivaluedIndex
    public Collection queryByKeyPrefix(
        Object prefix, SinglevaluedIndex repos) throws StorageException
    {
        // TODO:
        throw new RuntimeException("oops, not yet implemented");
    }

    private class ItemCollection extends AbstractCollection
    {
        private Object key;
        private SinglevaluedIndex repos;

        ItemCollection(Object key,SinglevaluedIndex repos)
        {
            this.key = key;
            this.repos = repos;
        }
        
        // implement AbstractCollection
        public Iterator iterator()
        {
            try {
                return new ItemCollectionIter(
                    storage.getResultSetIterator(
                        sqlFind,
                        new Object[]{key},
                        getValueType()),
                    repos,
                    key);
            } catch (StorageException ex) {
                throw new RuntimeStorageException(ex);
            }
        }

        public int size()
        {
            try {
                return storage.getSingletonInt(sqlFindCount,new Object[]{key});
            } catch (StorageException ex) {
                throw new RuntimeStorageException(ex);
            }
        }

        // implement AbstractCollection
        public boolean add(Object value)
        {
            try {
                JdbcMultivaluedIndex.this.add(key,value);
                return true;
            } catch (StorageException ex) {
                throw new RuntimeStorageException(ex);
            }
        }
    }

    private class ItemCollectionIter implements Iterator
    {
        private Iterator iter;
        private SinglevaluedIndex repos;
        private Object key;
        private Object value;

        ItemCollectionIter(Iterator iter,SinglevaluedIndex repos,Object key)
        {
            this.iter = iter;
            this.repos = repos;
            this.key = key;
        }

        // implement Iterator
        public boolean hasNext()
        {
            return iter.hasNext();
        }

        // implement Iterator
        public Object next()
        {
            value = iter.next();
            if (repos != null) {
                try {
                    return repos.get(value);
                } catch (StorageException ex) {
                    throw new RuntimeStorageException(ex);
                }
            }
            return value;
        }

        // implement Iterator
        public void remove()
        {
            try {
                // NOTE:  in the presence of duplicates, this won't necessarily
                // delete the one we're on, but it doesn't really matter
                // since the duplicates are externally indistinguishable
                // (they only differ in their surrogate keys, which are
                // totally internal)
                JdbcMultivaluedIndex.this.remove(key,value);
            } catch (StorageException ex) {
                throw new RuntimeStorageException(ex);
            }
        }
    }
}

// End JdbcMultivaluedIndex.java
