/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.junit;

import java.net.URL;

import java.util.ArrayList;
import javax.swing.Action;
import javax.swing.JEditorPane;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.queries.UnitTestForSourceQuery;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.spi.java.classpath.support.ClassPathSupport;
import org.openide.ErrorManager;
import org.openide.cookies.EditorCookie;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.nodes.Node;
import org.openide.util.HelpCtx;
import org.openide.util.NbBundle;

/** Action sensitive to some DataFolder or SourceCookie cookie
 * which tries to open JUnit test corresponding to the selected soruce file.
 *
 * @author  Nathan W. Phelps, David Konecny
 * @author  Marian Petras
 * @ version 1.0
 */
public class OpenTestAction extends TestAction {

    public OpenTestAction() {
        putValue("noIconInMenu", Boolean.TRUE);
    }
    

    protected void performAction (Node[] nodes) {
        FileObject selectedFO;
        
        for (int i = 0; i < nodes.length; i++) {
            // get test class or suite class file, if it was not such one pointed by the node
            selectedFO = TestUtil.getFileObjectFromNode(nodes[i]);
            if (selectedFO == null) {
                TestUtil.notifyUser(NbBundle.getMessage(OpenTestAction.class, "MSG_file_from_node_failed"));
                continue;
            }
            ClassPath cp = ClassPath.getClassPath(selectedFO, ClassPath.SOURCE);
            if (cp == null) {
                TestUtil.notifyUser(NbBundle.getMessage(OpenTestAction.class, 
                    "MSG_no_project", selectedFO));
                continue;
            }
 
            FileObject packageRoot = cp.findOwnerRoot(selectedFO);            
            URL[] testRoots = UnitTestForSourceQuery.findUnitTests(packageRoot);
            FileObject fileToOpen = null;
            for (int j = 0 ; j < testRoots.length; j++) {
                fileToOpen = findUnitTestInTestRoot(cp, selectedFO, testRoots[j]);
                if (fileToOpen != null) break;
            }
            
            if (fileToOpen != null) {
                openFile(fileToOpen);
            } else {                
                String testClsName = getTestName(cp, selectedFO).replace('/','.');                                                
                String pkgName = cp.getResourceName(selectedFO, '.', false);                
                boolean isPackage = selectedFO.isFolder();
                boolean isDefPkg = isPackage && (pkgName.length() == 0);
                String msgPattern = !isPackage
                       ? "MSG_test_class_not_found"                     //NOI18N
                       : isDefPkg
                         ? "MSG_testsuite_class_not_found_def_pkg"      //NOI18N
                         : "MSG_testsuite_class_not_found";             //NOI18N
                                
                String[] params = isDefPkg ? new String[] { testClsName }
                                           : new String[] { testClsName,
                                                            pkgName };                                                                             
                                 
                TestUtil.notifyUser(NbBundle.getMessage(OpenTestAction.class,
                                                        msgPattern, params),
                                    ErrorManager.INFORMATIONAL);
                continue;
            }
        }
    }

    private static FileObject findUnitTestInTestRoot(ClassPath cp, FileObject selectedFO, URL testRoot) {
        ClassPath testClassPath = null;
        if (testRoot == null) { //no tests, use sources instead
            testClassPath = cp;
        } else {
            try {
                ArrayList cpItems = new ArrayList();
                cpItems.add(ClassPathSupport.createResource(testRoot));
                testClassPath = ClassPathSupport.createClassPath(cpItems);
            } catch (IllegalArgumentException ex) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, ex);
                testClassPath = cp; 
            }
        }
        String testName = getTestName(cp, selectedFO);
        return testClassPath.findResource(testName+".java");
    }

    private static String getTestName(ClassPath cp, FileObject selectedFO) {
        String resource = cp.getResourceName(selectedFO, '/', false);        
        String testName = null;
        if (selectedFO.isFolder()) {
        //find Suite for package
            testName = TestUtil.convertPackage2SuiteName(resource);
        } else {
        // find Test for class
            testName = TestUtil.convertClass2TestName(resource);
        }
        
        return testName;
    }
    
    /**
     * Open given file in editor.
     * @return true if file was opened or false
     */
    public static boolean openFile(FileObject fo) {
        DataObject dobj;
        try {
            dobj = DataObject.find(fo);
        } catch (DataObjectNotFoundException e) {
            ErrorManager.getDefault().log(ErrorManager.ERROR, e.toString());
            return false;
        }
        if (dobj == null) {
             return false;
         }
         EditorCookie cookie = (EditorCookie)dobj.getCookie(EditorCookie.class);
         if (cookie == null) {
             return false;
         }
         cookie.open();
         return true;
    }
    
    /**
     */
    static boolean openFileAtElement(FileObject fileObject,
                                     Feature element) {
        final DataObject dataObject;
        try {
            dataObject = DataObject.find(fileObject);
        } catch (DataObjectNotFoundException e) {
            ErrorManager.getDefault().log(ErrorManager.ERROR, e.toString());
            return false;
        }
        assert dataObject != null;

        JEditorPane openedEditor = getEditor(dataObject);
        if (openedEditor == null) {
            return false;
        }
        
        int startOffset = element.getStartOffset();
        int endOffset = element.getEndOffset();
        int caretPosition = openedEditor.getCaretPosition();
        if ((caretPosition < startOffset) || (caretPosition >= endOffset)) {
            openedEditor.setCaretPosition(startOffset);
        }
        return true;
    }
    
    /**
     */
    private static JEditorPane getEditor(DataObject dataObj) {
        final EditorCookie editorCookie = (EditorCookie)
                                          dataObj.getCookie(EditorCookie.class);
        if (editorCookie == null) {
            return null;
        }
        
        editorCookie.open();
        
        JEditorPane[] editorPanes = editorCookie.getOpenedPanes();
        return (editorPanes != null) ? editorPanes[0] : null;
    }
    
    public String getName () {
        return NbBundle.getMessage (OpenTestAction.class, "LBL_Action_OpenTest");
    }

     protected String iconResource () {
         return "org/netbeans/modules/junit/resources/OpenTestActionIcon.gif";
     }

    public HelpCtx getHelpCtx () {
        return new HelpCtx(OpenTestAction.class);
    }

    /** Perform special enablement check in addition to the normal one.
    protected boolean enable (Node[] nodes) {
	if (! super.enable (nodes)) return false;
	if (...) ...;
    }
    */

    protected void initialize () {
	super.initialize ();
        putProperty(Action.SHORT_DESCRIPTION, NbBundle.getMessage(OpenTestAction.class, "HINT_Action_OpenTest"));
    }

}
