/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javadoc.comments;

import org.netbeans.jmi.javamodel.*;
import org.openide.util.Utilities;

import javax.jmi.reflect.JmiException;
import javax.swing.*;
import java.awt.*;
import java.lang.reflect.Modifier;
import java.text.Format;
import java.util.List;
import java.util.Iterator;

/** 
 * Describes jmi element. It is inteded to minimize access to MDR.
 */
final class ElementDescriptor {
    private static final int offsetPublic = 0;
    private static final int offsetPackage = 4;
    private static final int offsetProtected = 8;
    private static final int offsetPrivate = 12;
    private static final int iconNothing = 0;
    private static final int iconClass = 1;
    private static final int iconInterface = 2;
    private static final int iconField = 3;
    private static final int iconConstructor = iconField + offsetPrivate + 1;
    private static final int iconMethod = iconConstructor + offsetPrivate + 1;
    private static final int iconEnum = iconMethod + offsetPrivate + 1;
    private static final int iconAnnType = iconEnum + offsetPrivate + 1;
    private static final int iconConstant = iconAnnType + offsetPrivate + 1;
    
    private static String[] EMPTY_ARRAY = new String[0];
    
    private String name;
    private String formattedName;
    private int modifiers;
    private int effectiveAccess;
    private ImageIcon icon;
    private String declaringClassName;
    private String[] paramNames = EMPTY_ARRAY;
    private String[] throwFQNames = EMPTY_ARRAY;
    private String typeFQName;
    private final AutoCommenter.Element element;
    private final String identity;

    /** needs to be enclosed in mdr transaction */
    public ElementDescriptor(AutoCommenter.Element el, Format nameFormat) throws JmiException {
        this.element = el;
        ClassMember cm = el.getSrcElement();
        this.name = cm.getName();
        this.formattedName = nameFormat.format(cm);
        this.modifiers = cm.getModifiers();
        ClassDefinition cd = cm.getDeclaringClass();
        if (cd != null) this.declaringClassName = cd.getName();
        
        if (cm instanceof CallableFeature) {
            CallableFeature cf = (CallableFeature) cm;
            this.paramNames = getParameterNames(cf);
            this.throwFQNames = getThrowFQNames(cf);
        }
        
        if (cm instanceof TypedElement) {
            this.typeFQName = ((TypedElement) cm).getType().getName();
        }
        
        this.effectiveAccess = getEffectiveAccess(cm);
        this.identity = cm.refMofId();
    }
    
    public final String getName() {
        return this.name;
    }
    
    public final String getFormattedName() {
        return formattedName;
    }
    
    /**
     * name of the declaring class of the element
     * @return name or <code>null</code>
     */ 
    public final String getDeclaringClassName() {
        return declaringClassName;
    }
        
    public final int getModifiers() {
        return modifiers;   
    }
    
    public final int getEffectiveAccess() {
        return effectiveAccess;
    }
    
    public final String[] getParameterNames() {
        return paramNames;
    }
    
    public final String[] getThrowFQNames() {
        return throwFQNames;
    }
    
    public final String getTypeFQName() {
        return typeFQName;
    }
        
    public final Icon getIcon() {
        assert this.icon != null;
        return icon;
    }

    public String getIdentity() {
        return identity;
    }

    /**
     * recompute image; needs MDR transaction
     */ 
    public final void recomputeIcon() {
        if (this.icon == null) {
            this.icon = new ImageIcon();
        }
        this.icon.setImage(getMergedImage());
    }

    // private impl
    
    private static String[] getParameterNames(CallableFeature cf) {
        List params = cf.getParameters();
        String[] names = new String[params.size()];
        int i = 0;
        for (Iterator it = params.iterator(); it.hasNext(); i++) {
            Parameter param = (Parameter) it.next();
            names[i] = param.getName();
        }
        return names;
    }
    
    private static String[] getThrowFQNames(CallableFeature cf) {
        List exs = cf.getExceptions();
        String[] names = new String[exs.size()];
        int i = 0;
        for (Iterator it = exs.iterator(); it.hasNext(); i++) {
            JavaClass ex = (JavaClass) it.next();
            names[i] = ex.getName();
        }
        return names;
    }

    private Image getMergedImage() {
        int error = this.element.getErrorNumber();
        int type = resolveIconIndex();

        Image im1 = getImage(error);
        Image im2 = getMemberImage(type);
        return Utilities.mergeImages(im1, im2, 18, 0);
    }

    private int resolveIconIndex() throws JmiException {
        ClassMember me = this.element.getSrcElement();
        int offset;
        if ((Modifier.PUBLIC & modifiers) != 0)
            offset = offsetPublic;
        else if ((Modifier.PRIVATE & modifiers) != 0)
            offset = offsetPrivate;
        else if ((Modifier.PROTECTED & modifiers) != 0)
            offset = offsetProtected;
        else
            offset = offsetPackage;

        if (me instanceof JavaEnum)
            return offset + iconEnum;
        else if (me instanceof AnnotationType)
            return offset + iconAnnType;
        else if (me instanceof JavaClass)
            return offset + (Modifier.isInterface(modifiers) ? iconInterface : iconClass);
        else if (me instanceof Method || me instanceof Attribute)
            return offset + iconMethod;
        else if (me instanceof Constructor)
            return offset + iconConstructor;
        else if (me instanceof EnumConstant)
            return iconConstant;
        else if (me instanceof Field)
            return offset + iconField;
        else
            return iconNothing;
    }

    /**
     * @param index
     * @return  */
    private static Image getMemberImage(int index) {
        switch (index) {
            case iconClass + offsetPackage:
            case iconClass + offsetProtected:
            case iconClass + offsetPrivate:
            case iconClass + offsetPublic:
                return Utilities.loadImage("org/openide/src/resources/class.gif"); // NOI18N
            case iconInterface + offsetPackage:
            case iconInterface + offsetProtected:
            case iconInterface + offsetPrivate:
            case iconInterface + offsetPublic:
                return Utilities.loadImage("org/openide/src/resources/interface.gif"); // NOI18N
            case iconEnum + offsetPackage:
            case iconEnum + offsetProtected:
            case iconEnum + offsetPrivate:
            case iconEnum + offsetPublic:
                return Utilities.loadImage("org/netbeans/modules/java/resources/enum.gif"); // NOI18N
            case iconAnnType + offsetPackage:
            case iconAnnType + offsetProtected:
            case iconAnnType + offsetPrivate:
            case iconAnnType + offsetPublic:
                return Utilities.loadImage("org/netbeans/modules/java/resources/annotation_type.gif"); // NOI18N
            case iconConstant:
                return Utilities.loadImage("org/netbeans/modules/java/resources/constant.gif"); // NOI18N
            case iconField + offsetPublic:
                return Utilities.loadImage("org/openide/src/resources/variablePublic.gif"); // NOI18N
            case iconField + offsetPackage:
                return Utilities.loadImage("org/openide/src/resources/variablePackage.gif"); // NOI18N
            case iconField + offsetProtected:
                return Utilities.loadImage("org/openide/src/resources/variableProtected.gif"); // NOI18N
            case iconField + offsetPrivate:
                return Utilities.loadImage("org/openide/src/resources/variablePrivate.gif"); // NOI18N
            case iconConstructor + offsetPublic:
                return Utilities.loadImage("org/openide/src/resources/constructorPublic.gif"); // NOI18N
            case iconConstructor + offsetPackage:
                return Utilities.loadImage("org/openide/src/resources/constructorPackage.gif"); // NOI18N
            case iconConstructor + offsetProtected:
                return Utilities.loadImage("org/openide/src/resources/constructorProtected.gif"); // NOI18N
            case iconConstructor + offsetPrivate:
                return Utilities.loadImage("org/openide/src/resources/constructorPrivate.gif"); // NOI18N
            case iconMethod + offsetPublic:
                return Utilities.loadImage("org/openide/src/resources/methodPublic.gif"); // NOI18N
            case iconMethod + offsetPackage:
                return Utilities.loadImage("org/openide/src/resources/methodPackage.gif"); // NOI18N
            case iconMethod + offsetProtected:
                return Utilities.loadImage("org/openide/src/resources/methodProtected.gif"); // NOI18N
            case iconMethod + offsetPrivate:
                return Utilities.loadImage("org/openide/src/resources/methodPrivate.gif"); // NOI18N
            default:
                return null;
        }
    }

    /** Used to returnn image for given index, not cached here, by Utilities
     * @param index of Image
     * @return Image for index
     */
    private static Image getImage(int index) {
        switch (index) {
            case 1:
                return Utilities.loadImage("org/netbeans/modules/javadoc/comments/resources/ok.gif"); // NOI18N
            case 2:
                return Utilities.loadImage("org/netbeans/modules/javadoc/comments/resources/missing.gif"); // NOI18N
            case 4:
                return Utilities.loadImage("org/netbeans/modules/javadoc/comments/resources/error.gif"); // NOI18N        
            default:
                return null;
        }
    }

    private static int getEffectiveAccess(ClassMember el) {
        int access = el.getModifiers() & (Modifier.PUBLIC | Modifier.PROTECTED | Modifier.PRIVATE);
        if (access == Modifier.PRIVATE)
            return access;
        JavaClass decl = (JavaClass) el.getDeclaringClass();
        if (decl == null)
            // it is a top-level class
            return access;
        if (decl.isInterface())
            // interface members must be public
            access = Modifier.PUBLIC;
        
        int parentAccess = getEffectiveAccess(decl);
        switch (parentAccess) {
            case Modifier.PRIVATE:
            case 0:
                // private members were handled above, everything else is 
                // constrained by parent's privacy
                return parentAccess;
            case Modifier.PROTECTED:
                // member can be restricted to only the package
                return access == 0 ? 0 : parentAccess;
            case Modifier.PUBLIC:
            default:
                return access;
        }
    }


}
