/*****************************************************************************
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"

 * The Original Software is the CVS Client Library.
 * The Initial Developer of the Original Software is Robert Greig.
 * Portions created by Robert Greig are Copyright (C) 2000.
 * All Rights Reserved.

 * Contributor(s): Robert Greig.
 *****************************************************************************/
package org.netbeans.lib.cvsclient.event;

import org.netbeans.lib.cvsclient.ClientServices;

import java.io.File;

/**
 * This class is responsible for firing CVS events to registered listeners.
 * It can either fire events as they are generated or wait until a suitable
 * checkpoint and fire many events at once. This can prevent event storms
 * from degrading system performance.
 * @author  Robert Greig
 */
public class EventManager {
    /**
     * Registered listeners for events. This is an array for performance when
     * firing events. We take the hit when adding or removing listeners - that
     * should be a relatively rare occurrence.
     */
    private CVSListener[] listeners;

    /**
     * Holds value of property fireEnhancedEventSet.
     * If true, the library fires the EnhancedMessageEvents.
     * Default is true. Some builders might work badly, if set to false.
     */
    private boolean fireEnhancedEventSet = true;
    
    private final ClientServices services;

    /**
     * Construct a new EventManager
     */
    public EventManager(ClientServices services) {
        this.services = services;
    }

    /**
     * Returns Client services implementation tied to this event manager.
     * 
     * @return a ClientServices implementation
     */ 
    public ClientServices getClientServices() {
        return services;
    }

    /**
     * Add a listener to the list.
     * @param listener the listener to add
     */
    public synchronized void addCVSListener(CVSListener listener) {
        if (listeners == null || listeners.length == 0) {
            listeners = new CVSListener[1];
        }
        else {
            // allocate a new array and copy existing listeners
            CVSListener[] l = new CVSListener[listeners.length + 1];
            for (int i = 0; i < listeners.length; i++) {
                l[i] = listeners[i];
            }
            listeners = l;
        }
        listeners[listeners.length - 1] = listener;
    }

    /**
     * Remove a listeners from the list
     * @param listener the listener to remove
     */
    public synchronized void removeCVSListener(CVSListener listener) {
        // TODO: test this method!!
        if (listeners.length == 1) {
            listeners = null;
        }
        else {
            CVSListener[] l = new CVSListener[listeners.length - 1];
            int i = 0;
            while (i < l.length) {
                if (listeners[i] == listener) {
                    for (int j = i + 1; j < listeners.length; j++) {
                        l[j - 1] = listeners[j];
                    }
                    break;
                }
                else {
                    l[i] = listeners[i];
                }
                i++;
            }
            listeners = l;
        }
    }

    /**
     * Fire a CVSEvent to all the listeners
     * @param e the event to send
     */
    public void fireCVSEvent(CVSEvent e) {
        // if we have no listeners, then there is nothing to do
        if (listeners == null || listeners.length == 0)
            return;
        if (e instanceof FileInfoEvent) {
            File file = ((FileInfoEvent) e).getInfoContainer().getFile();
            if (services.getGlobalOptions().isExcluded(file)) return;
        }
        CVSListener[] l = null;
        synchronized (listeners) {
            l = new CVSListener[listeners.length];
            System.arraycopy(listeners, 0, l, 0, l.length);
        }

        for (int i = 0; i < l.length; i++) {
            e.fireEvent(l[i]);
        }
    }

    /** Getter for property fireEnhancedEventSet.
     * @return Value of property fireEnhancedEventSet.
     */
    public boolean isFireEnhancedEventSet() {
        return fireEnhancedEventSet;
    }

    /** Setter for property fireEnhancedEventSet.
     * @param fireEnhancedEventSet New value of property fireEnhancedEventSet.
     */
    public void setFireEnhancedEventSet(boolean fireEnhancedEventSet) {
        this.fireEnhancedEventSet = fireEnhancedEventSet;
    }

}
