/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.lib.cvsclient.connection;

import org.netbeans.lib.cvsclient.file.FileUtils;

import java.io.*;
import java.util.*;

/**
 * Represents .cvspass passwords file.
 *
 * @author Petr Kuzel
 */
public final class PasswordsFile {

    /**
     * Locates scrambled password for given CVS Root.
     *
     * @param cvsRootString identifies repository session [:method:][[user][:password]@][hostname[:[port]]]/path/to/repository
     * @return scrambled password or <code>null</code>
     */
    public static String findPassword(String cvsRootString) {
        File passFile = new File(System.getProperty("cvs.passfile", System.getProperty("user.home") + "/.cvspass"));
        BufferedReader reader = null;
        String password = null;

        try {
            reader = new BufferedReader(new FileReader(passFile));
            String line;
            while ((line = reader.readLine()) != null) {
                line = normalize(line);
                if (line.startsWith(cvsRootString+" ")) {
                    password = line.substring(cvsRootString.length() + 1);
                    break;
                }
            }
        } catch (IOException e) {
            return null;
        }
        finally {
            if (reader != null) {
                try { reader.close(); } catch (IOException e) {}
            }
        }
        return password;

    }

    /**
     * List roots matching given prefix e.g. <tt>":pserver:"</tt>.
     */
    public static Collection listRoots(String prefix) {

        List roots = new ArrayList();

        File passFile = new File(System.getProperty("cvs.passfile", System.getProperty("user.home") + "/.cvspass"));
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new FileReader(passFile));
            String line;
            while ((line = reader.readLine()) != null) {
                line = normalize(line);
                String elements[] = line.split(" ");  // NOI18N
                if (elements[0].startsWith(prefix)) {
                    roots.add(elements[0]);
                }
            }
        } catch (IOException e) {
            return Collections.EMPTY_SET;
        }
        finally {
            if (reader != null) {
                try { reader.close(); } catch (IOException e) {}
            }
        }
        return roots;
    }

    /**
     * Writes scrambled password for given CVS root.
     * Eliminates all previous values and possible duplicities.
     *
     * @param cvsRootString identifies repository session [:method:][[user][:password]@][hostname[:[port]]]/path/to/repository
     * @param encodedPassword
     * @throws IOException on write failure
     */
    public static void storePassword(String cvsRootString, String encodedPassword) throws IOException {
        File passFile = new File(System.getProperty("cvs.passfile",
                                                    System.getProperty("user.home") + File.separatorChar +
                                                    ".cvspass"));
        BufferedWriter writer = null;
        BufferedReader reader = null;
        try {
            final String LF = System.getProperty("line.separator"); // NOI18N
            if (passFile.createNewFile()) {
                writer = new BufferedWriter(new FileWriter(passFile));
                writer.write(cvsRootString + " " + encodedPassword + LF);
                writer.close();
            }
            else {
                File tempFile = File.createTempFile("cvs", "tmp");
                reader = new BufferedReader(new FileReader(passFile));
                writer = new BufferedWriter(new FileWriter(tempFile));
                String line;
                boolean stored = false;
                while ((line = reader.readLine()) != null) {
                    if (normalize(line).startsWith(cvsRootString + " ")) {
                        if (stored == false) {
                            writer.write(cvsRootString + " " + encodedPassword + LF);
                            stored = true;
                        }
                    }
                    else {
                        writer.write(line + LF);
                    }
                }
                if (stored == false) {
                    writer.write(cvsRootString + " " + encodedPassword + LF);
                }
                reader.close();
                writer.flush();
                writer.close();

                // copyFile needs less permissions than File.renameTo
                FileUtils.copyFile(tempFile, passFile);
                tempFile.delete();
            }
        }
        finally {
            try {
                if (writer != null) {
                    writer.close();
                }
                if (reader != null) {
                    reader.close();
                }
            }
            catch (Exception e) {
                // ignore
            }
        }
    }

    /**
     * Normalizes several possible line formats into
     * 'normal' one that allows to apply dumb string operations.
     */
    private static String normalize(String line) {
        if (line.startsWith("/1 ")) {  // NOI18N
            line = line.substring("/1 ".length()); // NOI18N
        }
        return line;
    }
}
