/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.lib.cvsclient.command;

import org.netbeans.lib.cvsclient.util.SimpleStringPattern;
import org.netbeans.lib.cvsclient.ClientServices;

import java.io.*;
import java.util.Map;
import java.util.StringTokenizer;

/**
 * Support for <tt>.cvswrappers</tt> parsing and merging.
 */
public class WrapperUtils {

    /**
     * Reads the wrappers from the specified source and populates the specified
     * map
     *
     * @param reader The source of wrappers which is being processed
     * @param theMap The map which is being updated
     */
    private static  void parseWrappers(BufferedReader reader, Map theMap)
                 throws IOException {

        String line;
        while ((line = reader.readLine()) != null){
            StringTokenizer tokenizer = new StringTokenizer(line);

            // the first token is the pattern
            SimpleStringPattern pattern = new SimpleStringPattern(tokenizer.nextToken());

            // it is followed by option value pairs
            String option, value;

            while (tokenizer.hasMoreTokens()) {
                option = tokenizer.nextToken();
                value = tokenizer.nextToken();

                // do not bother with the -m Options now
                if (option.equals("-k")) { //NOI18N

                    // This is a keyword substitution option
                    // strip the quotes
                    int first = value.indexOf('\'');
                    int last = value.lastIndexOf('\'');
                    if (first >=0 && last >= 0) {
                        value = value.substring(first+1, last);
                    }

                    KeywordSubstitutionOptions keywordOption = KeywordSubstitutionOptions.findKeywordSubstOption(value);
                    if (!theMap.containsKey(pattern)) {
                        theMap.put(pattern, keywordOption);
                    }
                }
            }
        }
    }

    /**
     * Reads the wrappers from the specified file and populates the specified
     * map
     *
     * @param file The File object corresponding to the file which is being processed
     * @param wrapperMap The map which is being updated
     */
    public static void readWrappersFromFile(File file, Map wrapperMap) throws IOException, FileNotFoundException{
        parseWrappers(new BufferedReader(new FileReader(file)), wrapperMap);
    }

    /**
     * Reads the wrappers from the specified System property and populates the specified
     * map. The map is unchanged if the property is not set.
     *
     * @param propertyName The system properties name
     * @param wrapperMap The map which is being updated
     */
    public static void readWrappersFromProperty(String propertyName, Map wrapperMap) throws IOException {
        String propertyValue = System.getProperty(propertyName);
        if (propertyValue != null)
        {
            parseWrappers(new BufferedReader(new StringReader(propertyValue)), wrapperMap);
        }
    }

    /**
     * This method consolidates the wrapper map so that it follows CVS prioritization
     * rules for the wrappers. Both AddCommand and ImportCommand will be calling
     * this.
     */
    public static Map mergeWrapperMap(ClientServices client) throws CommandException
    {
        String wrapperSource = null;
        Map wrappersMap = new java.util.HashMap(client.getWrappersMap());
        try
        {
            File home = new File(System.getProperty("user.home"));  // NOI18N
            File wrappers = new File(home, "./cvswrappers"); //NOI18N

            wrapperSource = CommandException.getLocalMessage("WrapperUtils.clientDotWrapper.text"); //NOI18N

            if (wrappers.exists()) {
                readWrappersFromFile(wrappers, wrappersMap);
            }

            wrapperSource = CommandException.getLocalMessage("WrapperUtils.environmentWrapper.text"); //NOI18N

            //process the Environment variable CVSWRAPPERS
            readWrappersFromProperty("Env-CVSWRAPPERS", wrappersMap);   //NOI18N
        }
        catch (FileNotFoundException fnex) {
            // should not happen as we check for file existence. Even if it does
            // it just means the .cvswrappers are not there and can be ignored
        }
        catch (Exception ioex) {
            Object [] parms = new Object[1];
            parms[0] = wrapperSource;
            String localizedMessage = CommandException.getLocalMessage("WrapperUtils.wrapperError.text", parms); //NOI18N
            throw new CommandException(ioex, localizedMessage);
        }

        return wrappersMap;
    }


}
