/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.versioning.system.cvss.ui.selectors;

import java.awt.Dialog;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.io.*;
import javax.swing.BorderFactory;
import javax.swing.text.Document;
import javax.swing.event.DocumentListener;
import javax.swing.event.DocumentEvent;
import org.netbeans.modules.versioning.system.cvss.settings.HistorySettings;

import org.openide.DialogDescriptor;
import org.openide.DialogDisplayer;
import org.openide.ErrorManager;
import org.openide.util.NbBundle;
import org.openide.util.HelpCtx;

/**
 * Prototype impl of defined modules listing.
 *
 * @author Petr Kuzel
 */
public final class ProxySelector implements ActionListener, DocumentListener {

    private ProxyDescriptor proxyDescriptor;

    private ProxyConfigurationPanel proxyPanel;

    private DialogDescriptor dialogDescriptor;

    /** Creates a new instance of ProxySelector */
    public ProxySelector() {
    }
    
    /**
     * Ask user for proxy settings. Shows modal UI.
     *
     * @return proxy descriptor <code>null</code> on cancel
     */
    public ProxyDescriptor selectProxy() {
        proxyPanel = new ProxyConfigurationPanel();

        // set initial values
        if (proxyDescriptor != null) {
            if (proxyDescriptor.isSystemProxyDescriptor()) {
                proxyPanel.systemProxyRadioButton.setSelected(true);
            } else if (proxyDescriptor.getType() == ProxyDescriptor.TYPE_HTTP) {
                proxyPanel.httpProxyRadioButton.setSelected(true);
            } else if (proxyDescriptor.getType() == ProxyDescriptor.TYPE_SOCKS) {
                proxyPanel.socksProxyRadioButton.setSelected(true);
            } else if (proxyDescriptor.getType() == ProxyDescriptor.TYPE_DIRECT) {
                proxyPanel.directRadioButton.setSelected(true);
            }
        } else {
            // use system settings
            proxyDescriptor = new ProxyDescriptor();
        }

        updateFields(proxyDescriptor);

        proxyPanel.systemProxyRadioButton.addActionListener(this);
        proxyPanel.directRadioButton.addActionListener(this);
        proxyPanel.httpProxyRadioButton.addActionListener(this);
        proxyPanel.socksProxyRadioButton.addActionListener(this);
        proxyPanel.portTextField.getDocument().addDocumentListener(this);
        proxyPanel.serverTextField.getDocument().addDocumentListener(this);

        // show dialog
        // XXX mask error in wizards framework, it shoudl be provided by the container
        proxyPanel.setBorder(BorderFactory.createEmptyBorder(6,6,6,6));
        dialogDescriptor = new DialogDescriptor(proxyPanel, NbBundle.getMessage(ProxySelector.class, "BK2004"));
        dialogDescriptor.setModal(true);
        dialogDescriptor.setHelpCtx(new HelpCtx(ProxySelector.class));
        revalidate();
        Dialog dialog = DialogDisplayer.getDefault().createDialog(dialogDescriptor);
        dialog.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(ProxySelector.class, "ACSD_ProxyDialog"));
        dialog.setVisible(true);

        // handle results
        if (DialogDescriptor.OK_OPTION.equals(dialogDescriptor.getValue())) {
            if (proxyPanel.systemProxyRadioButton.isSelected()) {
                return new ProxyDescriptor();
            } else {
                int type = ProxyDescriptor.TYPE_HTTP;
                if (proxyPanel.socksProxyRadioButton.isSelected()) {
                    type = ProxyDescriptor.TYPE_SOCKS;
                } else if (proxyPanel.directRadioButton.isSelected()) {
                    return ProxyDescriptor.DIRECT;
                }

                String host = proxyPanel.serverTextField.getText(); // XXX put input mask allowing DNS addresses
                int port = -1;
                String portNum = proxyPanel.portTextField.getText();  // XXX put input mask allowing only numbers
                try {
                    port = Integer.parseInt(portNum);
                } catch (NumberFormatException ignore) {
                }
                String user = null;
                String password = null;
                if (proxyPanel.loginCheckBox.isSelected()) {
                    user = proxyPanel.nameTextField.getText();
                    password = proxyPanel.passwordTextField.getText();
                }
                
                HistorySettings history = HistorySettings.getDefault();
                if (type == ProxyDescriptor.TYPE_HTTP) {
                    history.setHttpProxyHost(host);
                    history.setHttpProxyPort(portNum);
                } else if (type == ProxyDescriptor.TYPE_SOCKS) {
                    history.setSocksProxyHost(host);
                    history.setSocksProxyPort(portNum);                
                }
                
                return new ProxyDescriptor(type, host, port, user, password);
            }
        }
        return null;
    }

    private void updateFields(ProxyDescriptor proxyDescriptor) {

        boolean editable = !proxyDescriptor.isSystemProxyDescriptor() && proxyDescriptor.getType() != ProxyDescriptor.TYPE_DIRECT;

        String tip = null;
        if (editable == false) {
            tip = proxyDescriptor.getDescription();            
        }
        if (tip == null) {
            tip = java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/selectors/Bundle").getString("TT_UseSystemProxy");
        }
        proxyPanel.systemProxyRadioButton.setToolTipText(tip);

        String host = null;
        String port = null;
        String user = null;
        String password = null;
        if (proxyDescriptor.getType() != ProxyDescriptor.TYPE_DIRECT) {
            host = proxyDescriptor.getHost();
            int i = proxyDescriptor.getPort();
            if (i != -1) port = "" + i; // NOI18N
            user = proxyDescriptor.getUserName();
            password = proxyDescriptor.getPassword();
        }

        proxyPanel.serverTextField.setText(host);
        proxyPanel.portTextField.setText(port);
        proxyPanel.loginCheckBox.setSelected(user != null);
        proxyPanel.nameTextField.setText(user);
        proxyPanel.passwordTextField.setText(password);

        proxyPanel.serverTextField.setEditable(editable);
        proxyPanel.portTextField.setEditable(editable);
        proxyPanel.loginCheckBox.setEnabled(editable);
        proxyPanel.passwordTextField.setEnabled(editable && user != null);
        proxyPanel.nameTextField.setEnabled(editable && user != null);

        proxyPanel.enableServerPart(editable);
    }

    private void revalidate() {
        if (proxyPanel.directRadioButton.isSelected() || proxyPanel.systemProxyRadioButton.isSelected()) {
            dialogDescriptor.setValid(true);
        } else {
            boolean valid = true;
            Document doc = proxyPanel.portTextField.getDocument();
            try {
                int port = Integer.parseInt(doc.getText(0, doc.getLength()));
                valid &= port > 0;
            } catch (Exception e) {
                valid = false;
            }
            doc = proxyPanel.serverTextField.getDocument();
            valid &= doc.getLength() > 0;
            dialogDescriptor.setValid(valid);
        }
    }

    /**
     * Detects proxy in platform dependent way.
     *
     * @return Bootstrap, <b>static</b> proxy descriptor or null
     */
    static ProxyDescriptor detectSystemProxy() {

        // first try Java platform SOCKS
        String hostName = System.getProperty("socksProxyHost");  // NOI18N
        if (hostName != null) {
            int port = Integer.parseInt(System.getProperty("socksProxyPort", "1080"));  // NOI18N
            String user = System.getProperty("java.net.socks.username", System.getProperty("user.name"));
            String password = System.getProperty("java.net.socks.password");
            return new ProxyDescriptor(ProxyDescriptor.TYPE_SOCKS, hostName, port, user, password);
        }

        // then try Netbeans HTTP proxy
        // it detects proxy for Windows, Solaris, Linux Gnome, etc

        ProxyDescriptor pd = detectNetbeansHttpProxy();
        if (pd != null) {
            return pd;
        }

        // then try Java platform HTTP
        String hostHttp = System.getProperty("http.ProxyHost"); // NOI18N
        String portHttp = System.getProperty("http.ProxyPort"); // NOI18N
        if (hostHttp != null) {
            int port;
            try {
                port = Integer.parseInt(portHttp);
            } catch(NumberFormatException e) {
                port = 8080;
            }
            return new ProxyDescriptor(ProxyDescriptor.TYPE_HTTP, hostHttp, port, null, null);
        }

        if ("Linux".equalsIgnoreCase(System.getProperty("os.name"))) { // NOI18N
            // try KDE registry
            String home = System.getProperty("user.home"); // NOI18N
            File kioslaverc = new File(home, ".kde/share/config/kioslaverc"); // NOI18N
            if (kioslaverc.exists()) {
                BufferedReader r = null;
                try {
                    FileInputStream in = new FileInputStream(kioslaverc);
                    InputStreamReader inr = new InputStreamReader(in);
                    r = new BufferedReader(inr);
                    final String MAGIC = "httpProxy="; // NOI18N
                    String line = r.readLine();
                    while (line != null) {
                        if (line.startsWith(MAGIC)) {
                            String setting = line.substring(MAGIC.length());  // contains http://hostOffset:port
                            String host;
                            int port = -1;
                            int hostOffset = setting.indexOf("//");  // NOI18N
                            if (hostOffset != -1) {
                                setting = setting.substring(hostOffset +2);
                            }
                            int portOffset = setting.indexOf(':');
                            if (portOffset != -1) {
                                host = setting.substring(0, portOffset);
                                String s = setting.substring(portOffset+1);
                                port = Integer.parseInt(s);
                            } else {
                                host = setting;
                            }
                            String msg = NbBundle.getMessage(ProxySelector.class, "BK2001");
                            ProxyDescriptor desc = new ProxyDescriptor(ProxyDescriptor.TYPE_HTTP, host, port, null, null, msg);
                            return desc;
                        }
                        line = r.readLine();
                    }
                } catch (IOException e) {
                    ErrorManager err = ErrorManager.getDefault();
                    err.annotate(e, org.openide.util.NbBundle.getMessage(ProxySelector.class, "BK2023") + kioslaverc);
                    err.notify(e);
                } finally {
                     if (r != null) {
                         try {
                             r.close();
                         } catch (IOException alreadyClosed) {
                         }
                     }
                }
            }
        }

        return null;
    }

    /**
     * Reads property detected by native launcher (core/launcher).
     * Implemented for Windows and GNOME.
     */
    private static ProxyDescriptor detectNetbeansHttpProxy() {
        String host = getNetbeansProxyHost();
        if (host == null) {
            return null;
        }

        String  portHttp = getNetbeansProxyPort();
        int port;
        try {
            port = Integer.parseInt(portHttp);
        } catch(NumberFormatException e) {
            port = 8080;
        }
        String desc = NbBundle.getMessage(ProxySelector.class, "BK2000");
        return new ProxyDescriptor(ProxyDescriptor.TYPE_HTTP, host, port, null, null, desc);

    }

    private static String getNetbeansProxyHost () {
        String systemProxy = System.getProperty ("netbeans.system_http_proxy"); // NOI18N
        if (systemProxy == null) {
            return null;
        }

        int i = systemProxy.indexOf (":"); // NOI18N
        if (i <= 0 || i >= systemProxy.length () - 1) {
            return null; // NOI18N
        }

        return systemProxy.substring (0, i);
    }

    private static String getNetbeansProxyPort () {
        String systemProxy = System.getProperty ("netbeans.system_http_proxy"); // NOI18N
        if (systemProxy == null) {
            return ""; // NOI18N
         }

        int i = systemProxy.indexOf (":"); // NOI18N
        if (i <= 0 || i >= systemProxy.length () - 1) {
            return ""; // NOI18N
        }

        return systemProxy.substring (i+1);
    }

    /** Sets initial value, that is shows on selectProxy*/
    public void setProxyDescriptor(ProxyDescriptor proxyDescriptor) {
        this.proxyDescriptor = proxyDescriptor;
    }

    /** Loads system values into fields so user sees system proxy. */
    public void actionPerformed(ActionEvent e) {
        if (proxyPanel.systemProxyRadioButton.isSelected()) {
            ProxyDescriptor bootstrap = detectSystemProxy();
            if (bootstrap == null) {
                bootstrap = ProxyDescriptor.DIRECT;
            }
            updateFields(bootstrap);
        } else if (proxyPanel.directRadioButton.isSelected()) {
            updateFields(ProxyDescriptor.DIRECT);
        } else if (proxyPanel.httpProxyRadioButton.isSelected()) {
            if ("".equals(proxyPanel.portTextField.getText()) && "".equals(proxyPanel.serverTextField.getText())) {  // NOI18N
                HistorySettings history = HistorySettings.getDefault();
                proxyPanel.serverTextField.setText(history.getHttpProxyHost());
                proxyPanel.portTextField.setText(history.getHttpProxyPort());
            }
        } else if (proxyPanel.socksProxyRadioButton.isSelected()) {
            if ("".equals(proxyPanel.portTextField.getText()) && "".equals(proxyPanel.serverTextField.getText())) {  // NOI18N
                HistorySettings history = HistorySettings.getDefault();
                proxyPanel.serverTextField.setText(history.getSocksProxyHost());
                proxyPanel.portTextField.setText(history.getSocksProxyPort());                
            }            
        }
        revalidate();
    }

    public void changedUpdate(DocumentEvent e) {
    }

    public void insertUpdate(DocumentEvent e) {
        revalidate();
    }

    public void removeUpdate(DocumentEvent e) {
        revalidate();
    }
}
