/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.versioning.system.cvss.settings;

import org.netbeans.lib.cvsclient.CVSRoot;
import org.netbeans.modules.versioning.system.cvss.ui.selectors.ProxyDescriptor;
import org.openide.options.SystemOption;

import java.util.Set;
import java.util.Map;
import java.util.HashMap;
import java.io.Serializable;
import java.io.IOException;

/**
 * Persistent singleton storage of CVSRoots. Can list all
 * configured roots and query their settings.
 * <p>
 * Persistence is driven by associated BeanInfo.
 * <p>
 * TODO store passwords here too, to backup .cvspass PasswordFiles
 *
 * @author Petr Kuzel
 */
public final class CvsRootSettings extends SystemOption {

    private static final long serialVersionUID = 1L;

    public static final String PROP_ROOTS_MAP = "rootsMap";  // NOI18N

    /** For exteralization purposes. */
    public CvsRootSettings() {
    }

    static int id;

    // clients code ~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Loads value stored by {@link #setProxyFor}. By default
     * it returns {@link ProxyDescriptor#SYSTEM}.
     *
     * @return a proxy descriptor, never <code>null</code>
     */
    public static ProxyDescriptor getProxyFor(CVSRoot root) {

        ProxyDescriptor proxyDescriptor = ProxyDescriptor.SYSTEM;

        if (root != null) {        
            Map rootsMap = getDefault().getRootsMap();
            String rootString = root.toString();
            RootSettings rootSettings = (RootSettings) rootsMap.get(rootString);
            if (rootSettings != null && rootSettings.proxyDescriptor != null) {
                proxyDescriptor = rootSettings.proxyDescriptor;
            }
        }
        return proxyDescriptor;
    }

    public static void setProxyFor(CVSRoot root, ProxyDescriptor proxyDescriptor) {
        Map map = getDefault().getRootsMap();
        String key = root.toString();
        RootSettings settings = (RootSettings) map.get(key);
        if (settings == null) {
            settings = new RootSettings();
        }
        settings.proxyDescriptor = proxyDescriptor;
        map.put(key, settings);

        // need a new instance to trick setter optimalization
        Map rootsMap = new PersistentMap(map);
        getDefault().putProperty(PROP_ROOTS_MAP, rootsMap, true);
    }

    public static boolean hasExtSettingsFor(CVSRoot root) {
        assert "ext".equals(root.getMethod());  // NOI18N
        Map rootsMap = getDefault().getRootsMap();
        String rootString = root.toString();
        RootSettings rootSettings = (RootSettings) rootsMap.get(rootString);
        if (rootSettings != null) {
            ExtSettings extSettings = rootSettings.extSettings;
            return extSettings != null;
        }
        return false;
    }

    /**
     * Loads a value set by {@link #setExtSettingsFor}.
     *
     * @param root cvs root with <code>:ext:</code> connection method
     * @return  additional ext settings or their default value
     */
    public static ExtSettings getExtSettingsFor(CVSRoot root) {
        assert "ext".equals(root.getMethod());  // NOI18N
        Map rootsMap = getDefault().getRootsMap();
        String rootString = root.toString();
        RootSettings rootSettings = (RootSettings) rootsMap.get(rootString);
        if (rootSettings != null) {
            ExtSettings extSettings = rootSettings.extSettings;
            if (extSettings != null) {
                if (extSettings.extUseInternalSsh == false && extSettings.extCommand == null) {
                    extSettings.extCommand = System.getProperty("Env-CVS_RSH"); // NOI18N
                }
                return extSettings;
            }
        }

        // hardcoded default value
        ExtSettings defaults = new ExtSettings();
        defaults.extRememberPassword = false;
        defaults.extCommand = System.getProperty("Env-CVS_RSH"); // NOI18N
        defaults.extUseInternalSsh = true;
        defaults.extPassword = null;
        return defaults;
    }

    public static void setExtSettingsFor(CVSRoot root, ExtSettings extSettings) {
        assert "ext".equals(root.getMethod());  // NOI18N
        Map map = getDefault().getRootsMap();
        String key = root.toString();
        RootSettings settings = (RootSettings) map.get(key);
        if (settings == null) {
            settings = new RootSettings();
        }
        settings.extSettings = extSettings;
        map.put(key, settings);

        // need a new instance to trick setter optimalization
        Map rootsMap = new PersistentMap(map);
        getDefault().putProperty(PROP_ROOTS_MAP, rootsMap, true);
    }

    /**
     * @return Set of strings
     */
    public static Set listCvsRoots() {
        return getDefault().getRootsMap().keySet();
    }

    // System option implementation ~~~~~~~~~~~~~~~~~~

    private static CvsRootSettings getDefault() {
        return (CvsRootSettings) CvsRootSettings.findObject(CvsRootSettings.class, true);
    }


    public String displayName() {
        return getClass().getName();
    }

    public Map getRootsMap() {
        Map map = (Map) getProperty(PROP_ROOTS_MAP);
        if (map == null) {
            map = new PersistentMap();
        }
        return map;
    }

    public void setRootsMap(Map rootsMap) {
        putProperty(PROP_ROOTS_MAP, new PersistentMap(rootsMap), true);
    }

    /**
     * Holds associated settings.
     */
    private final static class RootSettings implements Serializable {
        private static final long serialVersionUID = 1L;

        private ProxyDescriptor proxyDescriptor;

        private ExtSettings extSettings;
    }

    /** External method additional settings */
    public final static class ExtSettings implements Serializable {
        private static final long serialVersionUID = 1L;

        public boolean extUseInternalSsh;

        /** Makes sense if extUseInternalSsh == true */
        public boolean extRememberPassword;

        /** Makes sense if extUseInternalSsh == true */
        public String extPassword;

        /** Makes sense if extUseInternalSsh == false */
        public String extCommand;

        private void writeObject(java.io.ObjectOutputStream out) throws IOException {
            String mem = extPassword;
            if (extRememberPassword == false) {
                extPassword = null;
            }
            out.defaultWriteObject();
            extPassword = mem;
        }
    }

    /**
     * Map that tricks system option setter optimalization
     */
    private final static class PersistentMap extends HashMap {
        private static final long serialVersionUID = 1L;

        public PersistentMap() {
        }

        public PersistentMap(Map map) {
            super(map);
        }

        public boolean equals(Object o) {
            return o == this;
        }

        public int hashCode() {
            return System.identityHashCode(this);
        }
    }
}
