/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.versioning.system.cvss.settings;

import org.openide.options.SystemOption;
import org.openide.util.Lookup;
import org.netbeans.modules.versioning.system.cvss.FileStatusProvider;

import java.util.regex.Pattern;
import java.util.*;
import java.io.File;

/**
 * Stores CVS module configuration.
 * 
 * @author Maros Sandor
 */
public class CvsModuleConfig extends SystemOption {
    
    private static final long serialVersionUID = 1L;

    public static final String PROP_IGNORED_FILEPATTERNS    = "ignoredFilePatterns";  // NOI18N
    public static final String PROP_COMMIT_EXCLUSIONS       = "commitExclusions";  // NOI18N
    public static final String PROP_DEFAULT_VALUES          = "defaultValues";  // NOI18N
    public static final String PROP_TEXT_ANNOTATIONS_FORMAT = "textAnnotations";  // NOI18N

    public static CvsModuleConfig getDefault() {
        return (CvsModuleConfig) CvsModuleConfig.findObject(CvsModuleConfig.class, true);
    }
    
    /** For exteralization purposes. */
    public CvsModuleConfig() {
    }

    static int id;

    // properties ~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Returns current text annotations format.
     * 
     * @return String current text annotations format or null if text annotations are off
     */ 
    public String getTextAnnotationsFormat() {
        return (String) getProperty(PROP_TEXT_ANNOTATIONS_FORMAT);
    }

    /**
     * Turns on/off text CVS annotations.
     * 
     * @param format new format for text annotations; recognized macros are {sticky} and {status}. To turn text
     * annotations off, set it to null.
     */ 
    public void setTextAnnotationsFormat(String format) {
        putProperty(PROP_TEXT_ANNOTATIONS_FORMAT, format, true);
    }
    
    /**
     * For serialization purposes only!
     *
     * <p>Use add/remove instead.
     */
    public Map getDefaultValues() {
        Map values = (Map) getProperty(PROP_DEFAULT_VALUES);
        if (values == null) {
            values = new PersistentHashMap(0);
            putProperty(PROP_DEFAULT_VALUES, values, false);
        }
        return values;
    }

    public Object getDefaultValue(String name, Object defaultValue) {
        Object value = getDefaultValues().get(name);
        return value != null ? value : defaultValue;
    }

    public String getDefaultValue(String name, String defaultValue) {
        Object value = getDefaultValues().get(name);
        return value != null ? (String) value : defaultValue;
    }

    public boolean getDefaultValue(String name, boolean defaultValue) {
        Object value = getDefaultValues().get(name);
        return value != null ? ((Boolean) value).booleanValue() : defaultValue;
    }

    public void setDefaultValue(String name, Object value) {
        Map values = getDefaultValues();
        Object oldValue = values.get(name);
        if (oldValue == value || value != null && value.equals(oldValue)) return;
        if (value == null) {
            values.remove(name);
        } else {
            values.put(name, value);
        }
        setDefaultValues(new PersistentHashMap(values));
    }

    public void setDefaultValue(String name, boolean value) {
        setDefaultValue(name, Boolean.valueOf(value));
    }

    /**
     * For serialization purposes only!
     *
     * <p>Use add/remove instead.
     */
    public void setDefaultValues(Map values) {
        putProperty(PROP_DEFAULT_VALUES, values, true);
    }

    public Pattern [] getIgnoredFilePatterns() {
        Pattern [] patterns = (Pattern []) getProperty(PROP_IGNORED_FILEPATTERNS);
        if (patterns == null) {
            patterns = getDefaultFilePatterns();
            putProperty(PROP_IGNORED_FILEPATTERNS, patterns, false);
        }
        return patterns;
    }

    public void setIgnoredFilePatterns(Pattern [] filePatterns) {
        putProperty(PROP_IGNORED_FILEPATTERNS, filePatterns, true);
    }
    
    public Set getCommitExclusions() {
        Set exclusions = (Set) getProperty(PROP_COMMIT_EXCLUSIONS);
        if (exclusions == null) {
            exclusions = new PersistentHashSet(0);
            putProperty(PROP_COMMIT_EXCLUSIONS, exclusions, false);
        }
        return exclusions;
    }

    /**
     * For deserialization purposes only!
     *
     * <p>Use add/remove instead.
     */
    public void setCommitExclusions(Set exclusions) {
        putProperty(PROP_COMMIT_EXCLUSIONS, exclusions, true);
    }

    // business methods ~~~~~~~~~~~~~~~~~~

    /**
     * @param path File.getAbsolutePath()
     */
    public boolean isExcludedFromCommit(String path) {
        return getCommitExclusions().contains(path);
    }

    /**
     * @param path File.getAbsolutePath()
     */
    public void addExclusionPath(String path) {
        Set exclusions = getCommitExclusions();
        if (exclusions.add(path)) {
            setCommitExclusions(new PersistentHashSet(exclusions));
            // it can influence annotations, fire status change event
            // TODO: handle this in FileStatusProvider directly
            FileStatusProvider statusProvider = (FileStatusProvider) Lookup.getDefault().lookup(FileStatusProvider.class);
            File f = new File(path);
            statusProvider.fireFileStatusEvent(f);
        }
    }

    /**
     * @param path File.getAbsolutePath()
     */
    public void removeExclusionPath(String path) {
        Set exclusions = getCommitExclusions();
        if (exclusions.remove(path)) {
            setCommitExclusions(new PersistentHashSet(exclusions));
            // it can influence annotations, fire status change event
            // TODO: handle this in FileStatusProvider directly
            FileStatusProvider statusProvider = (FileStatusProvider) Lookup.getDefault().lookup(FileStatusProvider.class);
            File f = new File(path);
            statusProvider.fireFileStatusEvent(f);
        }
    }
    
    // private methods ~~~~~~~~~~~~~~~~~~
    
    private static Pattern[] getDefaultFilePatterns() {
        return new Pattern [] {
                        Pattern.compile("cvslog\\..*"),  // NOI18N
                        Pattern.compile("\\.make\\.state"), // NOI18N
                        Pattern.compile("\\.nse_depinfo"), // NOI18N
                        Pattern.compile(".*~"), // NOI18N
                        Pattern.compile("#.*"), // NOI18N
                        Pattern.compile("\\.#.*"), // NOI18N
                        Pattern.compile(",.*"), // NOI18N
                        Pattern.compile("_\\$.*"), // NOI18N
                        Pattern.compile(".*\\$"), // NOI18N
                        Pattern.compile(".*\\.old"), // NOI18N
                        Pattern.compile(".*\\.bak"), // NOI18N
                        Pattern.compile(".*\\.BAK"), // NOI18N
                        Pattern.compile(".*\\.orig"), // NOI18N
                        Pattern.compile(".*\\.rej"), // NOI18N
                        Pattern.compile(".*\\.del-.*"), // NOI18N
                        Pattern.compile(".*\\.a"), // NOI18N
                        Pattern.compile(".*\\.olb"), // NOI18N
                        Pattern.compile(".*\\.o"), // NOI18N
                        Pattern.compile(".*\\.obj"), // NOI18N
                        Pattern.compile(".*\\.so"), // NOI18N
                        Pattern.compile(".*\\.exe"), // NOI18N
                        Pattern.compile(".*\\.Z"), // NOI18N
                        Pattern.compile(".*\\.elc"), // NOI18N
                        Pattern.compile(".*\\.ln"), // NOI18N
                    };
    }

    // System option implementation ~~~~~~~~~~~~~~~~~~

    public String displayName() {
        return getClass().getName();
    }
    
    private static final class PersistentHashSet extends HashSet {

        private static final long serialVersionUID = 1L;
        
        public PersistentHashSet() {
        }

        public PersistentHashSet(int initialCapacity) {
            super(initialCapacity);
        }

        public PersistentHashSet(Collection c) {
            super(c);
        }

        public int hashCode() {
            return System.identityHashCode(this);
        }

        public boolean equals(Object o) {
            return this == o;
        }
    }

    private static final class PersistentHashMap extends HashMap {

        private static final long serialVersionUID = 1L;

        public PersistentHashMap() {
        }

        public PersistentHashMap(Map m) {
            super(m);
        }

        public PersistentHashMap(int initialCapacity) {
            super(initialCapacity);
        }

        public int hashCode() {
            return System.identityHashCode(this);
        }

        public boolean equals(Object o) {
            return this == o;
        }
    }
}
