/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.versioning.system.cvss;

import org.netbeans.lib.cvsclient.admin.Entry;

import java.io.*;
import java.util.*;

/**
 * Encapsulates content of CVS/* files that is used to save to-be-deleted metadata. 
 * 
 * @author Maros Sandor
 */
public class CvsMetadata implements Serializable {
    
    private static final long serialVersionUID = 1L;
    
    private String      repository;
    private String      root;
    private String []   entries;

    /**
     * Reads and stores CVS metadata and also marks all files in entries as removed. 
     * 
     * @param file
     * @return
     * @throws IOException
     */ 
    public static CvsMetadata readAndRemove(File file) throws IOException {
        if (!file.isDirectory()) throw new IllegalArgumentException(file + " is not a directory"); // NOI18N
        BufferedReader r = null;

        CvsMetadata data = new CvsMetadata();
        
        try {
            r = new BufferedReader(new FileReader(new File(file, "CVS/Repository"))); // NOI18N
            data.repository = r.readLine();
        } finally {
            if (r != null) r.close();
        }

        try {
            r = new BufferedReader(new FileReader(new File(file, "CVS/Root"))); // NOI18N
            data.root = r.readLine();
        } finally {
            if (r != null) r.close();
        }
        
        try {
            r = new BufferedReader(new FileReader(new File(file, "CVS/Entries"))); // NOI18N
            String entry;
            List entries = new ArrayList();
            while ((entry = r.readLine()) != null) {
                Entry e = new Entry(entry);
                if (!"D".equals(entry) && !e.isDirectory()) { // NOI18N
                    if (e.getRevision().charAt(0) != '-') {
                        e.setRevision("-" + e.getRevision()); // NOI18N
                    }
                    e.setConflict(Entry.DUMMY_TIMESTAMP);
                    entry = e.toString();
                }
                entries.add(entry);
            }
            data.entries = (String[]) entries.toArray(new String[entries.size()]);
        } finally {
            if (r != null) r.close();
        }
        return data;
    }

    /** Saves metadata in the given folder, typically named CVS.
     *
     * @param dir folder to save to
     * @throws IOException 
     */
    public void save(File dir) throws IOException {
        dir.mkdirs();
        PrintWriter w = null;
        try {
            w = new PrintWriter(new FileWriter(new File(dir, "Repository"))); // NOI18N
            w.println(repository);
        } finally {
            if (w != null) w.close();
        }
        try {
            w = new PrintWriter(new FileWriter(new File(dir, "Root"))); // NOI18N
            w.println(root);
        } finally {
            if (w != null) w.close();
        }
        try {
            w = new PrintWriter(new FileWriter(new File(dir, "Entries"))); // NOI18N
            for (int i = 0; i < entries.length; i++) {
                w.println(entries[i]);
            }
        } finally {
            if (w != null) w.close();
        }
    }

    public CvsMetadata(String repository, String root, String [] entries) {
        this.repository = repository;
        this.root = root;
        this.entries = (String[]) entries.clone();
    }

    public CvsMetadata() {
    }
    
    public String getRepository() {
        return repository;
    }

    public String getRoot() {
        return root;
    }

    public String[] getEntries() {
        return entries;
    }

    public Entry[] getEntryObjects() {
        Entry [] eo = new Entry[entries.length];
        for (int i = 0; i < entries.length; i++) {
            eo[i] = new Entry(entries[i]);
        }
        return eo;
    }
}
