/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src.nodes;

import java.util.ResourceBundle;

import org.openide.src.*;
import org.openide.options.SystemOption;
import org.openide.util.HelpCtx;
import org.openide.util.NbBundle;

/*
* TODO:
* <UL>
*  <LI> weak listeners for listening on format changes - all element nodes should react on it.
* </UL>
*/
/** Display options for the hierarchy of source elements.
* These options determine the display name format
* of each kind of element.
* <p>Also included are read-only properties for the "long formats",
* which are in practice used for {@link ElementNode#getHintElementFormat}.
* <p>Changes to settings will fire property change events.
*
* @author Petr Hamernik
*/
public final class SourceOptions extends SystemOption {
    private static final int lastCompatibleVersionTag = 1;
    private static final int currentVersionTag = 1;
    
    /** Resource bundle. */
    private static ResourceBundle bundle;

    /** Kinds of the format. */
    private static final byte T_INITIALIZER = 0;
    private static final byte T_FIELD = 1;
    private static final byte T_CONSTRUCTOR = 2;
    private static final byte T_METHOD = 3;
    private static final byte T_CLASS = 4;
    private static final byte T_INTERFACE = 5;

    /** Names of all properties. */
    static final String[] PROP_NAMES = {
        "initializerElementFormat", "fieldElementFormat", // NOI18N
        "constructorElementFormat", "methodElementFormat", // NOI18N
        "classElementFormat", "interfaceElementFormat" // NOI18N
    };
    
    static Element[] TEST_ELEMENTS;

    /** default values for the formats - short form. */
    private static final ElementFormat[] DEFAULT_FORMATS_SHORT = new ElementFormat[6];

    /** default values for the formats - long form. */
    private static final ElementFormat[] DEFAULT_FORMATS_LONG = new ElementFormat[6];

    /**
     * Current format for individual element types, or null if the format is
     * not yet specified by the user.
     */
    private ElementFormat[] formats = new ElementFormat[6];
    
    /**
     * Version tag to use;
     */
    private int version;
    
    private static synchronized void loadBundle() {
        if (bundle != null)
            return;
        bundle = NbBundle.getBundle(SourceOptions.class);
    }
    
    private static void loadDefaultFormats() {
        if (DEFAULT_FORMATS_SHORT[0] != null)
            return;
        synchronized (SourceOptions.class) {
            if (DEFAULT_FORMATS_SHORT[0] != null)
                return;
            loadBundle();
            for (int i = 0; i < 6; i++) {
                DEFAULT_FORMATS_SHORT[i] = new ElementFormat(bundle.getString("SHORT_"+PROP_NAMES[i]));
                DEFAULT_FORMATS_LONG[i] = new ElementFormat(bundle.getString("LONG_"+PROP_NAMES[i]));
            }
        }
    }
    
    /**
     * Resets all element formats to their default values.
     */
    private void clearElementFormats() {
        formats = new ElementFormat[6];
    }

    /** Property name of the initializer display format. */
    public static final String PROP_INITIALIZER_FORMAT = PROP_NAMES[T_INITIALIZER];

    /** Property name of the field display format. */
    public static final String PROP_FIELD_FORMAT = PROP_NAMES[T_FIELD];

    /** Property name of the constructor display format. */
    public static final String PROP_CONSTRUCTOR_FORMAT = PROP_NAMES[T_CONSTRUCTOR];

    /** Property name of the method display format. */
    public static final String PROP_METHOD_FORMAT = PROP_NAMES[T_METHOD];

    /** Property name of the class display format. */
    public static final String PROP_CLASS_FORMAT = PROP_NAMES[T_CLASS];

    /** Property name of the interface display format. */
    public static final String PROP_INTERFACE_FORMAT = PROP_NAMES[T_INTERFACE];

    /** Property name of the 'categories usage' property. */
    public static final String PROP_CATEGORIES_USAGE = "categoriesUsage"; // NOI18N

    /** CategoriesUsage property current value */
    private static boolean categories = true;

    static final long serialVersionUID =-2120623049071035434L;

    /** @return display name
    */
    public String displayName () {
        loadBundle();
        return bundle.getString("MSG_sourceOptions");
    }

    public HelpCtx getHelpCtx () {
        return new HelpCtx (SourceOptions.class);
    }

    // ============= public methods ===================

    /** Set the initializer format.
    * @param format the new format
    */
    public void setInitializerElementFormat(ElementFormat format) {
        setElementFormat(T_INITIALIZER, format);
    }

    /** Get the initializer format.
    * @return the current format
    */
    public ElementFormat getInitializerElementFormat() {
        return getElementFormat(T_INITIALIZER);
    }

    /** Set the field format.
    * @param format the new format
    */
    public void setFieldElementFormat(ElementFormat format) {
        setElementFormat(T_FIELD, format);
    }
    
    private ElementFormat getElementFormat(int type) {
        synchronized (this) {
            if (formats[type] != null)
                return formats[type];
            // if writing the option to the disk, return a default == null value.
            if (isWriteExternal())
                return null;
        }
        loadDefaultFormats();
        return DEFAULT_FORMATS_SHORT[type];
    }

    /** Get the field format.
    * @return the current format
    */
    public ElementFormat getFieldElementFormat() {
        return getElementFormat(T_FIELD);
    }

    /** Set the constructor format.
    * @param format the new format
    */
    public void setConstructorElementFormat(ElementFormat format) {
        setElementFormat(T_CONSTRUCTOR, format);
    }

    /** Get the constructor format.
    * @return the current format
    */
    public ElementFormat getConstructorElementFormat() {
        return getElementFormat(T_CONSTRUCTOR);
    }

    /** Set the method format.
    * @param format the new format
    */
    public void setMethodElementFormat(ElementFormat format) {
        setElementFormat(T_METHOD, format);
    }

    /** Get the method format.
    * @return the current format
    */
    public ElementFormat getMethodElementFormat() {
        return getElementFormat(T_METHOD);
    }

    /** Set the class format.
    * @param format the new format
    */
    public void setClassElementFormat(ElementFormat format) {
        setElementFormat(T_CLASS, format);
    }

    /** Get the class format.
    * @return the current format
    */
    public ElementFormat getClassElementFormat() {
        return getElementFormat(T_CLASS);
    }

    /** Set the interface format.
    * @param format the new format
    */
    public void setInterfaceElementFormat(ElementFormat format) {
        setElementFormat(T_INTERFACE, format);
    }

    /** Get the interface format.
    * @return the current format
    */
    public ElementFormat getInterfaceElementFormat() {
        return getElementFormat(T_INTERFACE);
    }

    // ============= getters for long form of formats =================

    /** Get the initializer format for longer hints.
    * @return the current format
    */
    public ElementFormat getInitializerElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_INITIALIZER];
    }

    /** Get the field format for longer hints.
    * @return the current format
    */
    public ElementFormat getFieldElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_FIELD];
    }

    /** Get the constructor format for longer hints.
    * @return the current format
    */
    public ElementFormat getConstructorElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_CONSTRUCTOR];
    }

    /** Get the method format for longer hints.
    * @return the current format
    */
    public ElementFormat getMethodElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_METHOD];
    }

    /** Get the class format for longer hints.
    * @return the current format
    */
    public ElementFormat getClassElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_CLASS];
    }

    /** Get the interface format for longer hints.
    * @return the current format
    */
    public ElementFormat getInterfaceElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_INTERFACE];
    }

    // ============= categories of elements usage ===================

    /** Set the property whether categories under class elements should be used or not.
    * @param cat if <CODE>true</CODE> the elements under class elements are divided into
    *     categories: fields, constructors, methods. Otherwise (<CODE>false</CODE>) all elements
    *     are placed directly under class element.
    */
    public void setCategoriesUsage(boolean cat) {
        categories = cat;
    }

    /** Test whether categiries under class elements are used or not.
    * @return <CODE>true</CODE> if the elements under class elements are divided into
    *     categories: fields, constructors, methods. Otherwise <CODE>false</CODE> (all elements
    *     are placed directly under class element).
    */
    public boolean getCategoriesUsage() {
        return categories;
    }

    // ============= private methods ===================
    
    private synchronized static Element getTestElement(int index) {
        if (TEST_ELEMENTS == null) {
            Element[] els = new Element[6];
            
            try {
                els[T_INITIALIZER] = new InitializerElement();
                FieldElement f = new FieldElement();
                Identifier id = Identifier.create("foo"); // NOI18N
                f.setName(id); // NOI18N
                f.setType(Type.INT);
                els[T_FIELD] = f;

                MethodElement m = new MethodElement();
                m.setName(id);
                m.setReturn(Type.VOID);
                els[T_METHOD] = m;
                els[T_CONSTRUCTOR] = new ConstructorElement();

                ClassElement c = new ClassElement();
                c.setName(id);
                els[T_CLASS] = els[T_INTERFACE] = c;
                TEST_ELEMENTS = els;
            } catch (SourceException ex) {
                // cannot happen.
            }
        }
        return TEST_ELEMENTS[index];
    }
        
    /** Sets the format for the given index.
    * @param index One of the constants T_XXX
    * @param format the new format for the specific type.
    */
    private void setElementFormat(byte index, ElementFormat format) {
        ElementFormat old = formats[index];
        if (format != null) {
            // check whether the format is valid for the element type:
            Element el = getTestElement(index);
            try {
                format.format(el);
            } catch (IllegalArgumentException iae) {
                throw (IllegalArgumentException)
                    org.openide.ErrorManager.getDefault().annotate(
                        iae, org.openide.ErrorManager.USER, null,
                        bundle.getString("MSG_IllegalElementFormat"), // NOI18N
                        null, null);
            }
        }
        formats[index] = format;
        firePropertyChange (PROP_NAMES[index], old, formats[index]);
    }
    
    public void writeExternal(java.io.ObjectOutput out) throws java.io.IOException {
        super.writeExternal(out);
        out.writeInt(version);
    }
    
    public void readExternal (java.io.ObjectInput in)
    throws java.io.IOException, ClassNotFoundException {
        super.readExternal(in);
        if (in.available() > 0) {
	    version = in.readInt();
        }
        if (version < lastCompatibleVersionTag) {
            clearElementFormats();
            version = currentVersionTag;
        }
    }
}
