/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src.nodes;

import java.beans.*;
import java.io.IOException;
import java.lang.reflect.Modifier;
import java.lang.reflect.InvocationTargetException;

import org.openide.src.*;
import org.openide.nodes.*;

/** Node for a method element.
* @see MethodElement
* @author Petr Hamernik
*/
public class MethodElementNode extends ConstructorElementNode {
    /** Create a new method node.
    * @param element method element to represent
    * @param writeable <code>true</code> to be writable
    */
    public MethodElementNode(MethodElement element, boolean writeable) {
        super(element, writeable);
        setElementFormat0(sourceOptions.getMethodElementFormat());
    }


    public org.openide.util.HelpCtx getHelpCtx () {
        return new org.openide.util.HelpCtx ("org.openide.src.nodes.MethodNode"); // NOI18N
    }

    /* Resolve the current icon base.
    * @return icon base string.
    */
    protected String resolveIconBase() {
        int modif = ((MethodElement)element).getModifiers();
        if (Modifier.isStatic(modif)) {
            // static method...
            if (Modifier.isPrivate(modif))
                return METHOD_ST_PRIVATE;
            else if (Modifier.isProtected(modif))
                return METHOD_ST_PROTECTED;
            else if (Modifier.isPublic(modif))
                return METHOD_ST_PUBLIC;
            else
                return METHOD_ST_PACKAGE;
        }
        else {
            // non-static method...
            if (Modifier.isPrivate(modif))
                return METHOD_PRIVATE;
            else if (Modifier.isProtected(modif))
                return METHOD_PROTECTED;
            else if (Modifier.isPublic(modif))
                return METHOD_PUBLIC;
            else
                return METHOD_PACKAGE;
        }
    }

    /** Indicate that this node cannot be renamed.
    * An constructor must have the same name like class
    * @return <code>true</code>
    */
    public boolean canRename() {
        return true;
    }


    /* This method resolve the appropriate hint format for the type
    * of the element. It defines the short description.
    */
    protected ElementFormat getHintElementFormat() {
        return sourceOptions.getMethodElementLongFormat();
    }

    /* Creates property set for this node */
    protected Sheet createSheet () {
        Sheet sheet = Sheet.createDefault();
        Sheet.Set ps = sheet.get(Sheet.PROPERTIES);
        ps.put(createModifiersProperty(writeable));
        ps.put(createNameProperty(writeable));
        ps.put(createParametersProperty(writeable));
        ps.put(createReturnProperty(writeable));
        ps.put(createExceptionsProperty(writeable));
        return sheet;
    }

    /* Removes the element from the class and calls superclass.
    *
    * @exception IOException if SourceException is thrown
    *            from the underlayed Element.
    */
    public void destroy() throws IOException {
        SourceEditSupport.invokeAtomicAsUser(element, new SourceEditSupport.ExceptionalRunnable() {
                                                 public void run() throws SourceException {
                                                     MethodElement el = (MethodElement) element;
                                                     el.getDeclaringClass().removeMethod(el);
                                                 }
                                             });
        super.destroy();
    }

    /** Create a property for the method return value.
    * @param canW <code>false</code> to force property to be read-only
    * @return the property
    */
    protected Node.Property createReturnProperty(boolean canW) {
        return new ElementProp(PROP_RETURN, Type.class, canW) {
                   /** Gets the value */
                   public Object getValue () {
                       return ((MethodElement)element).getReturn();
                   }

                   /** Sets the value */
                   public void setValue(final Object val) throws IllegalArgumentException,
                       IllegalAccessException, InvocationTargetException {
                       super.setValue(val);
                       if (!(val instanceof Type))
                           throw new IllegalArgumentException();

                       runAtomic(element, new SourceEditSupport.ExceptionalRunnable() {
                                     public void run() throws SourceException {
                                         ((MethodElement)element).setReturn((Type)val);
                                     }
                                 });
                   }
               };
    }
}
