/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src.nodes;

import java.io.IOException;
import java.beans.*;
import java.lang.reflect.InvocationTargetException;

import org.openide.src.*;
import org.openide.nodes.*;

/** Node representing an initializer (static or nonstatic).
* @see InitializerElement
*
* @author Petr Hamernik
*/
public class InitializerElementNode extends ElementNode {
    /** Return value of getIconAffectingProperties method. */
    private static final String[] ICON_AFFECTING_PROPERTIES = new String[] {
                PROP_STATIC
            };

    /** Create a new initializer node.
    * @param element initializer element to represent
    * @param writeable <code>true</code> to be writable
    */
    public InitializerElementNode(InitializerElement element, boolean writeable) {
        super(element, Children.LEAF, writeable);
        setElementFormat0(sourceOptions.getInitializerElementFormat());
        superSetName("<initializer>"); // NOI18N
    }


    public org.openide.util.HelpCtx getHelpCtx () {
        return new org.openide.util.HelpCtx ("org.openide.src.nodes.InitializerNode"); // NOI18N
    }

    /* Resolve the current icon base.
    * @return icon base string.
    */
    protected String resolveIconBase() {
        return ((InitializerElement)element).isStatic() ? INITIALIZER_ST : INITIALIZER;
    }

    /* This method is used for resolving the names of the properties,
    * which could affect the icon (such as "modifiers").
    * @return the appropriate array.
    */
    protected String[] getIconAffectingProperties() {
        return ICON_AFFECTING_PROPERTIES;
    }

    /* This method resolve the appropriate hint format for the type
    * of the element. It defines the short description.
    */
    protected ElementFormat getHintElementFormat() {
        return sourceOptions.getInitializerElementLongFormat();
    }

    /* Creates property set for this node */
    protected Sheet createSheet () {
        Sheet sheet = Sheet.createDefault();
        Sheet.Set ps = sheet.get(Sheet.PROPERTIES);
        ps.put(createStaticProperty(writeable));
        return sheet;
    }

    /** Indicate that this node cannot be renamed.
    * An initializer has no name.
    * @return <code>false</code>
    */
    public boolean canRename() {
        return false;
    }

    /* Removes the element from the class and calls superclass.
    *
    * @exception IOException if SourceException is thrown
    *            from the underlayed Element.
    */
    public void destroy() throws IOException {
        SourceEditSupport.invokeAtomicAsUser(element, new SourceEditSupport.ExceptionalRunnable() {
                                                 public void run() throws SourceException {
                                                     InitializerElement el = (InitializerElement) element;
                                                     el.getDeclaringClass().removeInitializer(el);
                                                 }
                                             });
        super.destroy();
    }

    /** Create a property for whether or not the initializer is static.
    * @param canW <code>false</code> to force property to be read-only
    * @return the property
    */
    protected Node.Property createStaticProperty(boolean canW) {
        return new ElementProp(ElementProperties.PROP_STATIC, Boolean.TYPE, canW) {
                   /** Gets the value */
                   public Object getValue () {
                       return ((InitializerElement)element).isStatic() ? Boolean.TRUE : Boolean.FALSE;
                   }

                   /** Sets the value */
                   public void setValue(final Object val) throws IllegalArgumentException,
                       IllegalAccessException, InvocationTargetException {
                       super.setValue(val);

                       if (!(val instanceof Boolean))
                           throw new IllegalArgumentException();

                       runAtomic(element, new SourceEditSupport.ExceptionalRunnable() {
                                     public void run() throws SourceException {
                                         ((InitializerElement)element).setStatic(((Boolean)val).booleanValue());
                                     }
                                 });
                   }
               };
    }
}
