/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src.nodes;

import java.awt.Component;
import java.beans.*;
import java.io.IOException;
import java.lang.reflect.Modifier;
import java.lang.reflect.InvocationTargetException;

import org.openide.src.*;
import org.openide.nodes.*;

/** Node representing a field (variable).
* @see FieldElement
* @author Petr Hamernik
*/
public class FieldElementNode extends MemberElementNode {
    /** Create a new field node.
    * @param element field element to represent
    * @param writeable <code>true</code> to be writable
    */
    public FieldElementNode(FieldElement element, boolean writeable) {
        super(element, Children.LEAF, writeable);
        setElementFormat0(sourceOptions.getFieldElementFormat());
    }

    public org.openide.util.HelpCtx getHelpCtx () {
        return new org.openide.util.HelpCtx ("org.openide.src.nodes.FieldNode"); // NOI18N
    }

    /* Resolve the current icon base.
    * @return icon base string.
    */
    protected String resolveIconBase() {
        int modif = ((FieldElement)element).getModifiers();
        if (!Modifier.isStatic(modif)) {
            // non-static field...
            if (Modifier.isPrivate(modif))
                return FIELD_PRIVATE;
            else if (Modifier.isProtected(modif))
                return FIELD_PROTECTED;
            else if (Modifier.isPublic(modif))
                return FIELD_PUBLIC;
            else
                return FIELD_PACKAGE;
        }
        else {
            // static field...
            if (Modifier.isPrivate(modif))
                return FIELD_ST_PRIVATE;
            else if (Modifier.isProtected(modif))
                return FIELD_ST_PROTECTED;
            else if (Modifier.isPublic(modif))
                return FIELD_ST_PUBLIC;
            else
                return FIELD_ST_PACKAGE;
        }
    }

    /* This method resolve the appropriate hint format for the type
    * of the element. It defines the short description.
    */
    protected ElementFormat getHintElementFormat() {
        return sourceOptions.getFieldElementLongFormat();
    }

    /* Creates property set for this node */
    protected Sheet createSheet () {
        Sheet sheet = Sheet.createDefault();
        Sheet.Set ps = sheet.get(Sheet.PROPERTIES);
        ps.put(createModifiersProperty(writeable));
        ps.put(createNameProperty(writeable));
        ps.put(createTypeProperty(writeable));
        ps.put(createInitValueProperty(writeable));
        return sheet;
    }

    /* Removes the element from the class and calls superclass.
    *
    * @exception IOException if SourceException is thrown
    *            from the underlayed Element.
    */
    public void destroy() throws IOException {
        SourceEditSupport.invokeAtomicAsUser(element, new SourceEditSupport.ExceptionalRunnable() {
                                                 public void run() throws SourceException {
                                                     FieldElement el = (FieldElement) element;
                                                     el.getDeclaringClass().removeField(el);
                                                 }
                                             });
        super.destroy();
    }

    public Component getCustomizer() {
        return new FieldCustomizer((FieldElement)element);
    }

    public boolean hasCustomizer() {
        return isWriteable();
    }

    /** Create a property for the field type.
    * @param canW <code>false</code> to force property to be read-only
    * @return the property
    */
    protected Node.Property createTypeProperty(boolean canW) {
        Node.Property prop = new ElementProp(PROP_TYPE, Type.class, canW) {
                   /** Gets the value */
                   public Object getValue () {
                       return ((FieldElement)element).getType();
                   }

                   /** Sets the value */
                   public void setValue(final Object val) throws IllegalArgumentException,
                       IllegalAccessException, InvocationTargetException {
                       super.setValue(val);
                       if (!(val instanceof Type))
                           throw new IllegalArgumentException();

                       runAtomic(element, new SourceEditSupport.ExceptionalRunnable() {
                                     public void run() throws SourceException {
                                         ((FieldElement)element).setType((Type)val);
                                     }
                                 });
                   }
               };
       prop.setValue("acceptVoidType", Boolean.FALSE); // NOI18N
       return prop;
    }

    /** Create a property for the field init value.
    * @param canW <code>false</code> to force property to be read-only
    * @return the property
    */
    protected Node.Property createInitValueProperty(boolean canW) {
        return new ElementProp(PROP_INIT_VALUE, String.class, canW) {
                   /** Gets the value */
                   public Object getValue () {
                       return ((FieldElement)element).getInitValue();
                   }

                   /** Sets the value */
                   public void setValue(final Object val) throws IllegalArgumentException,
                       IllegalAccessException, InvocationTargetException {
                       super.setValue(val);
                       if (!(val instanceof String))
                           throw new IllegalArgumentException();

                       runAtomic(element, new SourceEditSupport.ExceptionalRunnable() {
                                     public void run() throws SourceException {
                                         ((FieldElement)element).setInitValue((String)val);
                                     }
                                 });
                   }
               };
    }
}
