/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src.nodes;

import java.lang.reflect.InvocationTargetException;
import java.beans.*;

import org.openide.src.*;
import org.openide.nodes.PropertySupport;
import org.openide.ErrorManager;
import org.openide.explorer.propertysheet.PropertyPanel;
import org.openide.explorer.propertysheet.editors.ModifierEditor;

/** The class calls Node.Property.setValue in runAtomicAsUser
 * to provide guarded block cheking.
 * 
 * @author  sdedic
 * @version 
 */
class ElementBeanModel extends PropertySupport.Reflection {
    private Element bean;
    private PropertyEditor editor;

    public ElementBeanModel(Element bean, String propertyName) throws NoSuchMethodException {
        this(bean, findInfo(bean, propertyName));
    }
    
    public ElementBeanModel(Element bean, PropertyDescriptor desc) throws NoSuchMethodException {
        super(bean, desc.getPropertyType(), desc.getName());
        setPropertyEditorClass(desc.getPropertyEditorClass());
        this.bean = bean;
    }

    public PropertyEditor getPropertyEditor() {
        if (editor == null) {
            editor = super.getPropertyEditor();
        }
        return editor;
    }

    /** indicates if it is running inside the atomic section */
    private boolean isRunningAtomic = false;
    
    public void setValue(final Object val) throws
            IllegalAccessException, IllegalArgumentException, InvocationTargetException {
        
        if (isRunningAtomic) {
            super.setValue(val);
            return;
        }
        
        SourceElement srcel = SourceEditSupport.findSource(bean);
        
        if (srcel == null) {
            super.setValue(val);
            return;
        }
        
        final InvocationTargetException[] ex = new InvocationTargetException[1];
        final IllegalAccessException[] ex2 = new IllegalAccessException[1];
        try {
            isRunningAtomic = true;
            srcel.runAtomicAsUser(new Runnable() {
                public void run() {
                    try {
                        setValue(val);
                    } catch (InvocationTargetException e) {
                        ex[0] = e;
                        ErrorManager.getDefault().annotate(
                            e, e.getTargetException()
                        );
                    } catch (IllegalAccessException e) {
                        ex2[0] = e; 
                    }
                }
            });
        } catch (SourceException e) {
            ex[0] = new InvocationTargetException(e);
	    ErrorManager.getDefault().annotate(ex[0], e);
        } finally {
            isRunningAtomic = false;
        }
        if (ex[0] != null) throw ex[0];
        if (ex2[0] != null) throw ex2[0];
        
    }
    
    /** Finds property descriptor for a bean.
     * @param bean the bean
     * @param name name of the property to find
     * @return the descriptor
     * @exception IllegalArgumentException if the method is not found
     */
    private static PropertyDescriptor findInfo (Object bean, String name) 
    throws IllegalArgumentException {
        try {
            BeanInfo beanInfo = Introspector.getBeanInfo(bean.getClass());
            PropertyDescriptor[] descr = beanInfo.getPropertyDescriptors();
            for (int i = 0; i < descr.length; i++) {
                if (descr[i].getName().equals(name)) {
                    return descr[i];
                }
            }
            throw new IllegalArgumentException (
                "No property named " + name + " in class " + bean.getClass () // NOI18N
            ); 
        } catch (IntrospectionException e) {
            IllegalArgumentException newEx = new IllegalArgumentException();
            ErrorManager.getDefault().annotate (newEx, e);
            throw newEx;
        }
    }

    public static PropertyPanel createPropertyPanel(MemberElement element, String propertyName) {
        try {
            PropertyPanel panel = new PropertyPanel (
                                            new ElementBeanModel(element, propertyName),
                                            PropertyPanel.PREF_CUSTOM_EDITOR
                                        );
            
            return panel;
        } catch (NoSuchMethodException e) {
            IllegalStateException ise = new IllegalStateException("Corrupted code"); // NOI18N
            ise.initCause(e);
            throw ise;
        }
    }
    
    public static PropertyPanel createModifiersPanel(MemberElement element) {
        PropertyPanel modifPanel = createPropertyPanel(element, ElementProperties.PROP_MODIFIERS);
        PropertyEditor propEdit = modifPanel.getProperty().getPropertyEditor();
        if (propEdit instanceof ModifierEditor) {
            ((ModifierEditor) propEdit).setMask(element.getModifiersMask());
        }
        return modifPanel;
    }

}
