/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src;

import java.beans.PropertyChangeEvent;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;

/**
 * This class is much like the ordinary PropertyChange event, but refines the message
 * so it carry diff between the old and the new contents. There are several operations
 * that can be made on a multivalued property:<UL>
 * <LI>add - new items were introduced into the property.
 * <LI>remove - items were removed from the property.
 * <LI>change - some items were atomically replaced by new ones.
 * <LI>reorder - no items were removed or added, but only reordered within the multivalued
 * property.
 * </UL>
 * <STRONG>ADD</STRONG> operation only introduces new items. For ordered multivalued properties,
 * there's a index array with positions for each newly inserted element.<BR>
 * <STRONG>REMOVE</STRONG> only mentiones elements that are being removed.<BR>
 * <STRONG>CHANGE</STRONG> event specifies what items are being removed and what are replacing them.<BR>
 * <STRONG>REORDER</STRONG> can specify:<UL>
 * <LI>a permutation describing the reorder operation
 * <LI>old and new state of the whole property
 * <LI>old state and the permutation
 * <LI>new state and the permutation
 * </UL>
 * <BR>
 * Note: this event is not particularly useful for indexed properties with primitive
 * values since the values does not have an identity and would need to be converted to
 * wrapper Objects which is not cheap anyway.
 * @author  sdedic
 * @version 0.1
 * @since 24/11/2000
 */
public class MultiPropertyChangeEvent extends PropertyChangeEvent {
    /**
     * Items are being added to the container.
     */
    public static final int TYPE_ADD      = 1;
    
    /**
     * Items are being removed from the container.
     */
    public static final int TYPE_REMOVE   = 2;
    
    /** Items are being replaced.
     */
    public static final int TYPE_MODIFY   = 3;
    
    /**
     * Items are being reordered.
     */
    public static final int TYPE_REORDER  = 4;
    
    /** Compound event, resulting from a generic change and comparisons
     */
    public static final int TYPE_COMPOUND = 5;

    /** Items that are affected by the change. These are items being added or removed,
     * items being replaced or partial changes for a compound event/
     */
    private Collection  affected;
    /** Collection of original items
     */
    
    private Collection newItems;
    /**
     * Permutation for reorder changes.
     */
    private int[] indices;
    /**
     * Type of the event.
     */
    private int eventType;

    public MultiPropertyChangeEvent(Object source, String propName, 
        Object oldVal, Object newVal) {
        super(source, propName, oldVal, newVal);
    }
    
    public void makeInsertion(Collection items, int[] indices) {
        checkUninitialized();
        this.affected = items;
        this.indices = indices;
        this.eventType = TYPE_ADD;
    }
    
    public void makeRemoval(Collection items, int[] indices) {
        checkUninitialized();
        this.affected = items;
        this.indices = indices;
        this.eventType = TYPE_REMOVE;
    }
    
    public void makeRemoval(Collection items) {
        checkUninitialized();
        this.affected = items;
        this.eventType = TYPE_REMOVE;
    }
    
    public void makeReorder(int[] permutation) {
        checkUninitialized();
        this.indices = permutation;
        this.eventType = TYPE_REORDER;
    }
    
    public void makeReplacement(Collection old, Collection n, int[] indices) {
        checkUninitialized();
        this.indices = indices;
        this.affected = old;
        this.newItems = n;
        this.eventType = TYPE_MODIFY;
    }
    
    public void makeCompound(Collection partialChanges, int[] offsets) {
        checkUninitialized();
        this.affected = partialChanges;
        this.eventType = TYPE_COMPOUND;
        this.indices = offsets;
    }
    
    protected void checkUninitialized() {
        if (this.eventType != 0) {
            throw new IllegalStateException("Event object is already initialized."); // NOI18N
        }
    }
    
    /** Returns items affected by the operation. For add and remove operations, these
     * are items added to or removed from the container, respectively. For change operation,
     * the collection contains items that are being replaced in the container.
     * @return collection of affected items. The collection is read-only.
     */
    public Collection getAffectedItems() {
        return this.affected;
    }
    
    public Collection getComponents() {
        if (eventType != TYPE_COMPOUND)
            throw new IllegalStateException("Not a compound event"); // NOI18N
        return affected;
    }
    
    /**
     * Returns items that will replace {@link #getAffectedItems} in the container.
     * @return r/o collection of new items that are being inserted in the container.
     */
    public Collection getReplacement() throws IllegalStateException {
        if (eventType != TYPE_MODIFY)
            throw new IllegalStateException("No replacement"); // NOI18N
        return this.newItems;
    }
    
    /**
     * Returns the permutation that describes the REORDER operation.
     * @return array of item's indexes in the new property's state. The array is indexed by positions in the old state.
     * @throws IllegalStateException if the event is not REORDER type.
     */
    public int[] getPermutation() throws IllegalStateException {
        if (eventType != TYPE_REORDER)
            throw new IllegalStateException("Not reordered"); // NOI18N
        return this.indices;
    }

    /** Returns array of relevant indices. For ADD operation, these are indexes which
     * the elements are inserted at (0-based, one position beyond item count permitted).
     * for REMOVE operation, indices are indexes to elements which are to be removed
     * (indexes are computed prior to the removing any of the affected items).
     * For MODIFY operation, this represents positions of elements which were replaced
     * by another ones. Last, REORDER operation uses indices as a permutation that
     * describes the reorder operation.
     * @return array of (integer) indices
     */
    public int[] getIndices() {
        return this.indices;
    }
    
    public final int getEventType() {
        return this.eventType;
    }
    
    /**
     * Returns iterator over partial operations collection in a compound change event.
     * If the event is simple, the returned iterator will return the event itself.
     */
    public Iterator getIterator() {
        if (getEventType() == TYPE_COMPOUND)
            return getComponents().iterator();
        else
            return Collections.singleton(this).iterator();
    }
}
