/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src;

import java.io.*;
import java.lang.reflect.Modifier;

import org.openide.util.NbBundle;

/** Representation of a method.
* It extends the constructor representation since
* all that is added is the return type.
*
* @author Petr Hamernik
*/
public final class MethodElement extends ConstructorElement {
    /** Format for the header - used in code generator */
    private static final ElementFormat HEADER_FORMAT =
        new ElementFormat("{m,,\" \"}{r} {n}({a}){e,\" throws \",}"); // NOI18N

    static final long serialVersionUID =2366156788906032138L;

    /** Create a method element held in memory. */
    public MethodElement() {
        this(new Memory(), null);
    }

    /** Create a method element.
    * @param impl implementation of functionality
    * @param clazz declaring class, or <code>null</code>
    */
    public MethodElement(MethodElement.Impl impl, ClassElement clazz) {
        super(impl, clazz);
    }

    /** Clone the method.
    * @return new method with the same values as the original,
    *   but represented in memory
    */
    public Object clone () {
        return new MethodElement (new Memory (this), null);
    }

    final MethodElement.Impl getMethodImpl() {
        return (MethodElement.Impl) impl;
    }

    /** Get the method's return type.
    * @return the return type
    */
    public Type getReturn() {
        return getMethodImpl().getReturn();
    }

    /** Set the method's return type.
    * @param ret the new type
    * @throws SourceException if impossible
    */
    public void setReturn (Type ret) throws SourceException {
        getMethodImpl().setReturn(ret);
    }

    /* @return the mask of possible modifiers for this element. */
    public int getModifiersMask() {
        if (isDeclaredInInterface()) {
            return Modifier.PUBLIC | Modifier.ABSTRACT;
        }
        else {
            return Modifier.PUBLIC | Modifier.PROTECTED | Modifier.PRIVATE |
                   Modifier.ABSTRACT | Modifier.STATIC | Modifier.FINAL |
                   Modifier.NATIVE | Modifier.SYNCHRONIZED;
        }
    }

    /** Set the name of this member.
    * @param name the name
    * @throws SourceException if impossible
    */
    public final void setName(Identifier name) throws SourceException {
        ClassElement c = getDeclaringClass();
        if (c != null) {
            MethodParameter[] params = getParameters();
            Type[] types = new Type[params.length];
            for (int i = 0; i < types.length; i++)
                types[i] = params[i].getType();
            MethodElement m = c.getMethod(name, types);
            if ((m != null) && (m != this)) {
		String msg = NbBundle.getMessage(ElementFormat.class,
			"FMT_EXC_RenameMethod", c.getName().getName(), name ); // NOI18N
                throwSourceException(msg);
            }
        }
        super.setName(name);
    }

    /** Get the printing format.
    * May be overridden by subclasses.
    * @return the format
    */
    ElementFormat getFormat() {
        return HEADER_FORMAT;
    }

    /** Marks the notable point in the writer.
    * This method calls markMethod.
    */
    void printerMark(ElementPrinter printer, int what) throws ElementPrinterInterruptException {
        printer.markMethod(this, what);
    }

    /** Pluggable behavior of the method element.
    * @see MethodElement
    */
    public interface Impl extends ConstructorElement.Impl {
        /** @deprecated Only public by accident. */
        /* public static final */ long serialVersionUID = 7273573865765501815L;
        /** Get the method's return type.
         * @return the return type
         */
        public Type getReturn();

        /** Set the method's return type.
         * @param ret the new type
         */
        public void setReturn (Type ret) throws SourceException;
    }

    /** A key for method elements, for use in hash tables etc.
    */
    public static final class Key extends ConstructorElement.Key {
        /** Name of the method */
        private Identifier name;
        private Type returnType;
        
        /** Constructs a key by name and parameters.
        * @param name the method name
        * @param params the method's parameters
        */
        public Key (final Identifier name, final Type[] params) {
            super(params);
            this.name = name;
        }
	
	/** Constructs a key that checks also a return type of the method.
	 * Methods with different return types are considered to be different although
	 * they have the same name and parameter types.
	 * @param name name of the method
	 * @param params types of method's arguments
	 * @param returnType return type for the method; if null, the key behaves like
	 * it was constructed using {@link MethodElement.Key#MethodElement.Key(Identifier, Type[])}
	 */
	public Key (final Identifier name, final Type[] params, final Type returnType) {	    
	    this(name, params);
	    this.returnType = returnType;
	}

        /** Constructs a key for a method.
        * Does not hold any reference to the element.
        * @param me the method element
        */
        public Key (final MethodElement me) {
            super(me);
            name = me.getName();
        }

	/**
	 * Constructs a key that checks for method's name, its argument types and
	 * optionally for a return type, if <I>checkReturn</I> is true.
	 * @param me method to construct the key for
	 * @param checkReturn if true, the key contains and also checks the retunr type.
	 */	
	public Key(final MethodElement me, boolean checkReturn) {
	    this(me);
	    if (checkReturn)
		returnType = me.getReturn();
	}

        /* Returns true if parameters are the same */
        public boolean equals (Object obj) {
            if (!(obj instanceof Key)) return false;
	    Key other = (Key)obj;
            if (!name.equals(other.name) || !super.equals(obj)) return false;
            if (returnType == null || other.returnType == null)
                return true;
            return returnType.equals(other.returnType);
        }

        /* Computes hashcode as exclusive or of
        * superclass hashcode and return type string hashcode.
        */
        public int hashCode () {
	    // Note: hashcode of the returntype was intentionally left out
	    // from the computation to allow matching against keys with no
	    // return type specified.
            return super.hashCode() ^ name.getFullName().hashCode();
        }

    } // end of Key inner class

    static class Memory extends ConstructorElement.Memory implements Impl {
        /** Type of exception */
        private Type type;

        static final long serialVersionUID =2015834437815195149L;
        Memory() {
            type = Type.VOID;
        }

        /** Copy constructor */
        Memory (MethodElement el) {
            super (el);
            type = el.getReturn ();
        }

        /** Setter for the return type */
        public Type getReturn() {
            return type;
        }

        /** @return the return type */
        public void setReturn (Type ret) {
            Type t = type;
            type = ret;
            firePropertyChange (PROP_RETURN, t, ret);
        }

        public Object readResolve() {
            return new MethodElement(this, null);
        }

    }

}
