/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src;

import java.io.*;

/** Superclass for containable Java source members
* (fields, methods and classes). Provides support
* for associating this element with a declaring class.
*
* @author Petr Hamernik, Jaroslav Tulach
*/
public abstract class MemberElement extends Element implements Cloneable {
    /** the class this element belongs to */
    private ClassElement declaringClass;

    static final long serialVersionUID =7896378970641663987L;

    /** Create a member element.
    * @param impl the pluggable implementation
    * @param declaringClass the class this element belongs to, or <code>null</code> if unattached
    */
    protected MemberElement(MemberElement.Impl impl, ClassElement declaringClass) {
        super(impl);
        this.declaringClass = declaringClass;
    }

    /** @return the current implementation. */
    final MemberElement.Impl getMemberImpl() {
        return (MemberElement.Impl) impl;
    }

    // [PENDING] Modifier explicitly disallows assuming its constants
    // are bitwise composable--this is technically illegal
    // (although in fact they are and this will probably never change)

    /** Get the modifier flags for this element.
    * Constrained by {@link #getModifiersMask}.
    * @return disjunction of constants from <CODE>java.lang.reflect.Modifier</CODE>
    */
    public final int getModifiers() {
        return getMemberImpl().getModifiers();
    }

    /** Set the modifier flags for this element.
    * @param mod disjunction of constants from <CODE>java.lang.reflect.Modifier</CODE>
    * @throws SourceException if impossible (e.g. if <code>mod & ~ getModifiersMask() != 0</code>)
    */
    public final void setModifiers(int mod) throws SourceException {
        getMemberImpl().setModifiers(mod);
    }

    /** Get the permitted modifiers for this type of element.
    * @return disjunction of constants from <CODE>java.lang.reflect.Modifier</CODE>
    */
    public abstract int getModifiersMask();

    /** Test whether declaring class is interface or class.
    * @return <CODE>true</CODE> for interfaces otherwise <CODE>false</CODE>
    */
    boolean isDeclaredInInterface() {
        return (declaringClass != null) && (declaringClass.isInterface());
    }

    /** Get the name of this member.
    * @return the name
    */
    public final Identifier getName() {
        return getMemberImpl().getName();
    }

    /** Set the name of this member.
    * @param name the name
    * @throws SourceException if impossible
    */
    public void setName(Identifier name) throws SourceException {
        getMemberImpl().setName(name);
        updateConstructorsNames(name);
    }

    /** Implemented in ClassElement - update names of the constructors.
    */
    void updateConstructorsNames(Identifier name) throws SourceException {
    }

    public Object clone() throws CloneNotSupportedException {
        return super.clone();
    }

    // no --jglick
    /* This field is automaticly sychnronized
    * when a MemberElement is added to the class. */
    /** Get the declaring class.
    *
    * @return the class that owns this member element, or <code>null</code> if the element is not
    *    attached to any class
    */
    public final ClassElement getDeclaringClass () {
        return declaringClass;
    }

    /** Pluggable implementation of member elements.
    * @see MemberElement
    */
    public interface Impl extends Element.Impl {
        /** @deprecated Only public by accident. */
        /* public static final */ long serialVersionUID = 2037286733482347462L;
        /** Get the modifier flags for this element.
         * Constrained by {@link MemberElement#getModifiersMask}.
         * @return disjunction of constants from <CODE>java.lang.reflect.Modifier</CODE>
         */
        public int getModifiers();

        /** Set the modifier flags for this element.
         * @param mod disjunction of constants from <CODE>java.lang.reflect.Modifier</CODE>
         * @throws SourceException if impossible (e.g. if <code>mod & ~ memberElt.getModifiersMask() != 0</code>)
         */
        public void setModifiers(int mod) throws SourceException;

        /** Get the name of this member.
         * @return the name
         */
        public Identifier getName();

        /** Set the name of this member.
         * @param name the name
         * @throws SourceException if impossible
         */
        public void setName(Identifier name) throws SourceException;
    }

    /** Default implementation of the Impl interface.
    * It just holds the property values.
    */
    static abstract class Memory extends Element.Memory implements MemberElement.Impl {
        /** Modifiers for this element */
        private int mod;

        /** Name of this element */
        private Identifier name;

        static final long serialVersionUID =1876531129266668488L;
        /** Default */
        public Memory () {
        }

        /** Copy */
        public Memory (MemberElement el) {
            mod = el.getModifiers ();
            name = el.getName ();
        }

        /** Getter for modifiers for this element.
        * @see java.lang.reflect.Modifier
        * @return constants from <CODE>java.lang.reflect.Modifier</CODE>
        */
        public int getModifiers() {
            return mod;
        }

        /** Setter for modifiers for this element.
        * @see java.lang.reflect.Modifier
        * @param mod constants from <CODE>java.lang.reflect.Modifier</CODE>
        */
        public void setModifiers(int mod) {
            int old = this.mod;
            this.mod = mod;
            firePropertyChange (PROP_MODIFIERS, new Integer (old), new Integer (mod));
        }

        /** Getter for name of the field.
        * @return the name
        */
        public synchronized Identifier getName() {
            if (name == null) {
                // lazy initialization !?
                name = Identifier.create(""); // NOI18N
            }
            return name;
        }

        /** Setter for name of the field.
        * @param name the name of the field
        */
        public synchronized void setName(Identifier name) {
            Identifier old = this.name;
            this.name = name;
            firePropertyChange (PROP_NAME, old, name);
        }
    
        public void markCurrent(boolean after) {
            ClassElement decl = ((MemberElement)element).getDeclaringClass();
            if (decl == null) {
                throw new IllegalStateException();
            }
            ((ClassElement.Memory)decl.getCookie(ClassElement.Memory.class)).markCurrent(element, after);
        }
    }
}
