/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src;

/** Represents a JavaDoc comment block. The representation object is mutable,
 * if it represents a portion of a source text, any changes made to the object
 * are reflected in the source code and vice versa.<BR>
 * There are a few inerclasses, which subclass JavaDoc class, and add accessors for tags specific
 * to individual source text elements.
 *
* @author Jaroslav Tulach, Petr Hrebejk, Petr Hamernik
*/
public interface JavaDoc {

    /** Serves as second parameter in method {@link #changeTags}.
     */
    public static final int ADD = 1;

    /** Serves as second parameter in method {@link #changeTags}.
     */
    public static final int REMOVE = 2;

    /** Serves as second parameter in method {@link #changeTags}.
     */
    public static final int SET = 3;

    /** Get the entire text of the comment. If the javadoc comment is empty (except the boxing
     * stars), it returns an empty string. If the comment is missing at all, the method returns <code>null</code>
     * ({@link #isEmpty} returns <code>true</code> at the same time).
    * @return the whole text, or <code>null</code>, if the comment is missing at all on the {@link Element}.
    */
    public String getRawText ();

    /** Set the raw text of the comment.
    * @param s the whole text to set
    * @exception SourceException if the modification cannot be performed
    */
    public void setRawText (String s) throws SourceException;

    /** Get the actual text, cleared of all (non-inline) tags.
    * @return the plain text
    */
    public String getText ();

    /** Set the actual text.
    * @param s the actual text, without any (non-inline) tags
    * @exception SourceException if the modification cannot be performed
    */
    public void setText (String s) throws SourceException;

    /** Clears the javadoc from the source.
    */
    public void clearJavaDoc() throws SourceException;

    /** Test if this javadoc is empty.
    * @return true if it is not generated to the source.
    */
    public boolean isEmpty();

    /** Gets all tags from comment.
     */
    public JavaDocTag[] getTags();

    /** Gets all tags of given name
     */
    public JavaDocTag[] getTags( String name );

    /** Adds removes or sets tags used in this comment
     * @param tags the new initializers
     * @param action {@link #ADD}, {@link #REMOVE}, or {@link #SET}
     * @exception SourceException if impossible
     */
    public void changeTags( JavaDocTag[] tags, int action ) throws SourceException;

    /** Gets all @see tags
     */
    public JavaDocTag.See[] getSeeTags();

    /** The JavaDoc of a class.
    * Class javadoc adds no special tags.
    */
    public static interface Class extends JavaDoc {

        /** @deprecated Only public by accident. */
        /* public static final */ long serialVersionUID = 3206093459760846163L;
    }

    /** The JavaDoc of a field.
    * <p>Currently adds special @SerialField tag
    */
    public static interface Field extends JavaDoc {
        /** Gets SerialField tags.
        */
        public JavaDocTag.SerialField[] getSerialFieldTags();

    }

    /** The JavaDoc of a method. Adds two special tags: @para tag and @throws tag.
    */
    public static interface Method extends JavaDoc {
        /** Gets param tags.
        */
        public JavaDocTag.Param[] getParamTags();

        /** Gets throws tags.
        */
        public JavaDocTag.Throws[] getThrowsTags();
    }

}
