/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src;

/** Represents one identifier.
*
* @author Petr Hamernik, Jaroslav Tulach
*/
public final class Identifier extends Object implements java.io.Serializable {
    /**
     * The identifier was not yet resolved, it was constructed without context information
     * or with unsuitable context.
     */
    public static final int NOT_YET_RESOLVED    = 0;
    /**
     * The identifier was resolved, the full name is filled in.
     */
    public static final int RESOLVED            = 1;
    /**
     * The identifier cannot be resolved. Although the full name is == source name,
     * the identifier does not correspond to anything.
     */
    public static final int UNRESOLVED          = 2;
    
    /** Full name of identifier. */
    private String fullName;

    /** The name which is either the same like fullName either
    * not fully qualified name - which was taken from the source
    * by java parser.
    */
    private String sourceName;

    /** Position after last dot in the identifier.
    * It is used in getName method.
    */
    private int namePos;

    /** Resolver for sourceName */
    private transient Resolver resolver;
    
    /**
     * Status of the identifier - one of the symbolic resolution constants
     */
    private int status;

    static final long serialVersionUID =-2614114568575211024L;

    /** New identifier.
    * @param resolver Resolver for fullName 
    * @param sourceName name for code generation
    */
    private Identifier(Resolver resolver, String sourceName) {
        this.fullName = null;
        this.sourceName = sourceName;
        this.resolver = resolver;
        this.status = NOT_YET_RESOLVED;
    }

    Identifier(String fullName, String sourceName, int status) {
        if (sourceName == null)
            throw new IllegalArgumentException();
        this.sourceName = sourceName;
        this.fullName = fullName;
        this.status = status;
    }

    /** Initiates namePos variable */
    private void initNamePos() {
        setFullName();
        namePos = fullName.lastIndexOf("."); // NOI18N
        if (namePos != -1)
            namePos++;
        if (fullName.startsWith(".") || fullName.endsWith(".")) // NOI18N
            throw new IllegalArgumentException(fullName);
    }

    /** Sets fullName variable. */
    private void setFullName() {
        if (fullName == null) {
            fullName = (resolver == null ? sourceName : resolver.resolve().intern());
            resolver = null;  // clear it
            if (fullName == null) { // not resolved?
                fullName = sourceName;
            }
        }
    }

    /** Finds the existing instance */
    private Object readResolve() {
	if (fullName == null) {
	    // in case the identifier was not resolved before it was saved, it's too late, sorry.
	    return create(sourceName);
	} else {
            return create(fullName, sourceName);
	}
    }

    /** Create an identifier with the same source name and fully qualified name.
    *
    * @param name the fully qualified name to create the identifier from
    * @return the identifier
    */
    public static Identifier create (String name) {
        String n = name.intern();
        return new Identifier(null, n);
    }

    /** Create an identifier.
    *
    * @param fullName fully qualified name
    * @param sourceName name for code generation
    * @return the identifier
    */
    public static Identifier create(String fullName, String sourceName) {
        Identifier ret = new Identifier(null, sourceName.intern());
        ret.fullName = fullName.intern();
        ret.initNamePos();
        return ret;
    }
    
    public static Identifier create(String full, String sourceName,
    int status) {
	Identifier id = Identifier.create(full, sourceName);
	id.status = status;
	return id;
    }

    /** Create an Identifier
    *
    * @param resolver a Resolver
    * @param name the name of the identifier to create
    */
    public static Identifier create(Resolver resolver, String name) {
        return new Identifier(resolver, name.intern());
    }

    /** Get the simple name within a package.
    * @return the simple name
    */
    public String getName () {
        int pos = sourceName.lastIndexOf("."); // NOI18N
        return (pos == -1) ? sourceName : sourceName.substring(pos + 1);
    }

    /** Get the identifier for the code generation.
    * @return the name from the source code
    */
    public String getSourceName () {
        return sourceName;
    }

    /** Get the package prefix.
    * @return the prefix, with no trailing dot, or an empty string if in the default package
    */
    public String getQualifier () {
        if (fullName == null) {
            initNamePos();
        }
        return (namePos == -1) ? "" : fullName.substring(0, namePos - 1); // NOI18N
    }

    /** Test whether this identifier is qualified by package.
    * @return <CODE>true</CODE> if so
    */
    public boolean isQualified () {
        if (fullName == null) {
            initNamePos();
        }
        return (namePos != -1);
    }

    /** Get the qualified name with the package prefix (if any).
    * @return the fully qualified name
    */
    public String getFullName () {
        if (fullName == null) {
            initNamePos();
        }
        return fullName;
    }
    
    /**
     * Returns the resolution status of the identifier. An identifier can be
     * one of the following:<UL>
     * <LI> NOT_YET_RESOLVED - no attempt to resolve the identifier was done.
     * <LI> UNRESOLVED - the identifier does not point to a symbol
     * <LI> RESOLVED - the identifier identifies the entity specified by the identifier's
     * fullName property.
     * </UL>
     */
    public int  getResolutionStatus() {
        return this.status;
    }

    /** This function was changed to match the behaviour of {@link Type#toString Type.toString} that
        returns text representation suitable for the source file.
    * @return source name of the identifier.
    */
    public String toString() {
        return getSourceName();
    }

    /** Compare the specified Identifier with this Identifier for equality.
    * @param id Identifier to be compared with this
    * @param source Determine if the source name should be also compared.
    *       If <CODE>false</CODE> only fully qualified name is compared.
    * @return <CODE>true</CODE> if the specified object equals to
    *         specified Identifier otherwise <CODE>false</CODE>.
    */
    public boolean compareTo(Identifier id, boolean source) {
        if (fullName == null) {
            initNamePos();
        }
        if (id.fullName == null) {
            id.initNamePos();
        }
        if (id.fullName == fullName) {
            return source ? (id.sourceName == sourceName) : true;
        }
        return false;
    }

    /** Compare the specified object with this Identifier for equality.
    * There are tested only full qualified name.
    * @param o Object to be compared with this
    * @return <CODE>true</CODE> if the specified object is Identifier
    *         with the same fully qualified name,
    *         otherwise <CODE>false</CODE>.
    */
    public boolean equals(Object o) {
        return (o instanceof Identifier) ? compareTo((Identifier) o, false) : false;
    }

    /** @return the hash code of full name String object.
    */
    public int hashCode() {
        if (fullName == null) {
            initNamePos();
        }
        return fullName.hashCode();
    }

    /** The interface allows lazy resolving of an Identifier to a fully qualified name. */
    public static interface Resolver {

        /**
         * @return fully qualified name
         */
        String resolve();
    }
}
