/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.explorer.propertysheet.editors;

import java.beans.*;
import java.util.*;

import javax.swing.*;

import org.openide.*;
import org.openide.src.MethodParameter;
import org.openide.src.Type;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;
import org.openide.explorer.propertysheet.ExPropertyEditor;
import org.openide.explorer.propertysheet.PropertyEnv;

/** Property editor for array of org.openide.src.MethodParameter classes
*
* @author Petr Hamernik
*/
public class MethodParameterArrayEditor extends PropertyEditorSupport implements ExPropertyEditor {

    /** Custom property editor Component. */
    MethodParameterArrayPanel panel;

    /** Flag for prevention of cycle in firing
    * of the properties changes.
    */
    boolean ignoreEditor = false;

    /** Flag for prevention of cycle in firing
    * of the properties changes.
    */
    boolean ignorePanel = false;
    
    private PropertyEnv env;

    /**
    * @return The property value as a human editable string.
    * <p>   Returns null if the value can't be expressed as an editable string.
    * <p>   If a non-null value is returned, then the PropertyEditor should
    *       be prepared to parse that string back in setAsText().
    */
    public String getAsText() {
        MethodParameter[] params = (MethodParameter[]) getValue();
        StringBuffer buf = new StringBuffer();
        if (params != null) {
            for (int i = 0; i < params.length; i++) {
                if (i > 0)
                    buf.append(", "); // NOI18N
                buf.append(params[i].getSourceString());
            }
        }
        return buf.toString();
    }

    /** Set the property value by parsing a given String.  May raise
    * java.lang.IllegalArgumentException if either the String is
    * badly formatted or if this kind of property can't be expressed
    * as text.
    * @param text  The string to be parsed.
    */
    public void setAsText(String text) throws IllegalArgumentException {
        StringTokenizer tok = new StringTokenizer(text, ",", false); // NOI18N
        ArrayList list = new ArrayList();
        while (tok.hasMoreTokens()) {
            String token = tok.nextToken();
            MethodParameter par;
            
            try {
                par = MethodParameter.parse(token);
            } catch (IllegalArgumentException ex) {
                String msg = java.text.MessageFormat.format(
                        getString("MSG_IllegalMethodParameter"), // NOI18N
                        new Object[] { token });
                ErrorManager.getDefault().annotate(ex,
			    ErrorManager.USER, null, msg, null, null);
                throw ex;
            }
            for (Iterator it = list.iterator(); it.hasNext(); ) {
                if (par.getName().equals(((MethodParameter)it.next()).getName())) {
                    String msg = java.text.MessageFormat.format(
                            getString("MSG_DuplicateName2"), // NOI18N
                            new Object[] { par.getName() });
                    IllegalArgumentException ex= new IllegalArgumentException("Ambiguous name"); // NOI18N
                    ErrorManager.getDefault().annotate(ex,
			    ErrorManager.USER, null,  msg, null, null);
                    throw ex;
                }
            }
            list.add(par);
        }
        MethodParameter[] params = new MethodParameter[list.size()];
        list.toArray(params);
        setValue(params);
    }

    /** Sets the value */
    public void setValue(Object o) {
        ignoreEditor = true;
        boolean saveIgnorePanel = ignorePanel;
        
        ignorePanel = false;
        super.setValue(o);
        if ((panel != null) & !saveIgnorePanel) {
            panel.setMethodParameters((MethodParameter[])o);
        }
        ignoreEditor = false;
    }

    /** @return <CODE>true</CODE> */
    public boolean supportsCustomEditor () {
        return true;
    }

    /** Create new panel for this property editor.
    * @return the visual component for editing the property
    */
    public java.awt.Component getCustomEditor () {
        if (panel == null) {
            panel = new MethodParameterArrayPanel();
            panel.setMnemonics(env);
            panel.setMethodParameters((MethodParameter[]) getValue());

            panel.addPropertyChangeListener(new PropertyChangeListener() {
                                                public void propertyChange(PropertyChangeEvent evt) {
                                                    if (!ignoreEditor && MethodParameterArrayPanel.PROP_METHOD_PARAMETERS.equals(evt.getPropertyName())) {
                                                        ignorePanel = true;
                                                        setValue(evt.getNewValue());
                                                        ignorePanel = false;
                                                    }
                                                }
                                            });

        }
        return panel;
    }

    /**
     * This method is called by the IDE to pass
     * the environment to the property editor.
     */
    public void attachEnv(PropertyEnv env) {
        this.env = env;
    }
    
    /** Implementation of the abstract ObjectArrayPanel2 class.
    * It is used for editing of arrays of Identifier objects.
    */
    static class MethodParameterArrayPanel extends ObjectArrayPanel2
    implements EnhancedCustomPropertyEditor {
        /** Name of the 'methodParameters' property */
        public static final String PROP_METHOD_PARAMETERS = "methodParameters"; // NOI18N

        /** Previous value */
        MethodParameter[] prevValue;

        /** Constructor */
        public MethodParameterArrayPanel() {
            prevValue = new MethodParameter[0];

            this.getListComponent().setCellRenderer(new DefaultListCellRenderer() {
                                                        public java.awt.Component getListCellRendererComponent(JList list,
                                                                Object value, int index, boolean isSelected, boolean cellHasFocus) {
                                                            java.awt.Component comp = super.getListCellRendererComponent(list,
                                                                                      value, index, isSelected, cellHasFocus);
                                                            if (comp == this) {
                                                                setText(((MethodParameter)value).toString());
                                                            }
                                                            return comp;
                                                        }
                                                    });
        }

        /** @return the current value
        */
        public MethodParameter[] getMethodParameters() {
            MethodParameter[] ret = new MethodParameter[model.size()];
            model.copyInto(ret);
            return ret;
        }

        /** Set new value
        */
        public void setMethodParameters(MethodParameter[] data) {
            model = new DefaultListModel();
            if (data != null) {
                for (int i = 0; i < data.length; i++)
                    model.addElement(data[i]);
            }
            this.getListComponent().setModel(model);
            modelChanged();
        }

        /** Fire the 'methodParameters' property change. */
        protected void modelChanged() {
            super.modelChanged();
            MethodParameter[] newValue = getMethodParameters();
            firePropertyChange(PROP_METHOD_PARAMETERS, prevValue, newValue);
            prevValue = newValue;
        }

        /** Ask user for new value.
        * @return new value or <CODE>null</CODE> when 
        *    operation was canceled.
        */
        protected Object insertNewValue() {
            return openInputDialog(null);
        }

        /** Ask user for edit value.
        * @param oldValue The previous value to be edited
        * @return new value or <CODE>null</CODE> when 
        *    operation was canceled.
        */
        protected Object editValue(Object oldValue) {
            return openInputDialog((MethodParameter) oldValue);
        }

        /** Show dialog and allow user to enter new method parameter.
        * @param defName Default value which is predefined.
        * @param titleKey the key to resource bundle for the title of input dialog
        * @return New valid name or <CODE>null</CODE> if user cancel the operation.
        */
        protected MethodParameter openInputDialog(MethodParameter origValue) {
            MethodParameterPanel panel = new MethodParameterPanel();

            NotifyDescriptor desriptor = new NotifyDescriptor(
                                             panel,
                                             getString("LAB_EnterParameter"),
                                             NotifyDescriptor.OK_CANCEL_OPTION,
                                             NotifyDescriptor.PLAIN_MESSAGE, null, null);

            if (origValue != null) {
                panel.nameTextField.setText(origValue.getName().toString());
                panel.typeCombo.setSelectedItem(origValue.getType().toString());
                panel.finalCheckBox.setSelected(origValue.isFinal());
            }

            for (;;) {
                Object ret = DialogDisplayer.getDefault().notify(desriptor);
                if (ret == NotifyDescriptor.OK_OPTION) {
                    String errMsg = null;
                    String name = panel.nameTextField.getText();
                    if (!Utilities.isJavaIdentifier(name))
                        errMsg = "MSG_NotValidID"; // NOI18N
                    else {
                        try {
                            Type type = Type.parse(panel.typeCombo.getSelectedItem().toString());
                            boolean isFinal = panel.finalCheckBox.isSelected();
                          Enumeration methodEnum = model.elements();
                          
                          while(methodEnum.hasMoreElements())
                          {   MethodParameter oldPar = (MethodParameter)methodEnum.nextElement();
                          
                              if (origValue!=null && oldPar.equals(origValue))
                                  continue;
                              if (name.equals(oldPar.getName()))
                              {    errMsg = "MSG_DuplicateName"; // NOI18N
                                  break;
                              }
                          }
                          if (errMsg == null)
                              return new MethodParameter(name, type, isFinal);
                        }
                        catch (IllegalArgumentException e) {
                            errMsg = "MSG_NotValidType"; // NOI18N
                        }
                    }
                    DialogDisplayer.getDefault().notify(new NotifyDescriptor.Message(getString(errMsg)));
                }
                else {
                    return null;
                }
            }
        }
        
        /** Get the customized property value.
         * @return the property value
         * @exception InvalidStateException when the custom property editor does not contain a valid property value
         *           (and thus it should not be set)
 */
        public Object getPropertyValue() throws IllegalStateException {
            return getMethodParameters();
        }
        
    }
    
    private static String getString(String key) {
        return NbBundle.getBundle("org.openide.explorer.propertysheet.editors.Bundle2", Locale.getDefault(), MethodParameterArrayEditor.class.getClassLoader()).getString(key);
    }

}
