/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.explorer.propertysheet.editors;

import java.beans.*;
import java.util.*;

import javax.swing.*;

import org.openide.*;
import org.openide.src.Identifier;
import org.openide.src.Type;
import org.openide.util.Utilities;
import org.openide.util.NbBundle;
import org.openide.explorer.propertysheet.ExPropertyEditor;
import org.openide.explorer.propertysheet.PropertyEnv;

/** Property editors for array of org.openide.src.Identifier
*
* @author Petr Hamernik
*/
public class IdentifierArrayEditor extends PropertyEditorSupport implements ExPropertyEditor {

    /** Custom property editor Component. */
    IdentifierArrayPanel panel;

    /** Flag for prevention of cycle in firing
    * of the properties changes.
    */
    boolean ignoreEditor = false;

    /** Flag for prevention of cycle in firing
    * of the properties changes.
    */
    boolean ignorePanel = false;

    PropertyEnv env;

    /** @return text representation of the value */
    public String getAsText() {
        Identifier[] id = (Identifier []) getValue();
        if ( id == null )
            return ""; // NOI18N
            
        StringBuffer buf = new StringBuffer();

        for (int i = 0; i < id.length; i++) {
            if (i > 0)
                buf.append(", "); // NOI18N
            buf.append(id[i].getSourceName());
        }

        return buf.toString();
    }

    /** Sets the value as the text */
    public void setAsText(String text) throws IllegalArgumentException {
        StringTokenizer tukac = new StringTokenizer(text, ", ", false); // NOI18N
        ArrayList list = new ArrayList();

        while (tukac.hasMoreTokens()) {
            String id = tukac.nextToken();
            try {
                // Assume any Java identifier can serve as a reference type name:
                Type t = Type.parse(id);
                if (!t.isClass())
                    throw new IllegalArgumentException();
            } catch (IllegalArgumentException ex) {
                String msg = java.text.MessageFormat.format(
                        getString("MSG_InvalidIdentifier"),
                        new Object[] { id });
                ErrorManager.getDefault().annotate(ex,
			    ErrorManager.USER, null, msg, null, null);
                throw ex;
            }
            list.add(Identifier.create(id));
        }

        Identifier[] ret = new Identifier[list.size()];
        list.toArray(ret);
        setValue(ret);
    }

    /** Set new value */
    public void setValue(Object o) {
        ignoreEditor = true;
        boolean saveIgnorePanel = ignorePanel;
        
        ignorePanel = false;
        super.setValue(o);
        if ((panel != null) & !saveIgnorePanel) {
            panel.setIdentifiers((Identifier[])o);
        }
        ignoreEditor = false;
    }

    /** @return <CODE>true</CODE> */
    public boolean supportsCustomEditor () {
        return true;
    }

    /** Create new panel for this property editor.
    * @return the visual component for editing the property
    */
    public java.awt.Component getCustomEditor () {
        if (panel == null) {
            panel = new IdentifierArrayPanel();
            panel.setIdentifiers((Identifier[])getValue());
            panel.setMnemonics(env);
            panel.addPropertyChangeListener(new PropertyChangeListener() {
                                                public void propertyChange(PropertyChangeEvent evt) {
                                                    if (!ignoreEditor && IdentifierArrayPanel.PROP_IDENTIFIERS.equals(evt.getPropertyName())) {
                                                        ignorePanel = true;
                                                        setValue(evt.getNewValue());
                                                        ignorePanel = false;
                                                    }
                                                }
                                            });
        }
        return panel;
    }

    /**
     * This method is called by the IDE to pass
     * the environment to the property editor.
     */
    public void attachEnv(PropertyEnv env) {
        this.env = env;
    }
    
    /** Implementation of the abstract ObjectArrayPanel2 class.
    * It is used for editing of arrays of Identifier objects.
    */
    static class IdentifierArrayPanel extends ObjectArrayPanel2 {

        /** Name of the 'identifiers' property. */
        public static final String PROP_IDENTIFIERS = "identifiers"; // NOI18N

        /** Previous value */
        Identifier[] prevValue;

        static final long serialVersionUID =-8655189809250688928L;
        /** Constructor */
        public IdentifierArrayPanel() {
            prevValue = new Identifier[0];

            this.getListComponent().setCellRenderer(new DefaultListCellRenderer() {
                                                        public java.awt.Component getListCellRendererComponent(JList list,
                                                                Object value, int index, boolean isSelected, boolean cellHasFocus) {
                                                            java.awt.Component comp = super.getListCellRendererComponent(list,
                                                                                      value, index, isSelected, cellHasFocus);
                                                            if (comp == this) {
                                                                setText(((Identifier)value).getFullName());
                                                            }
                                                            return comp;
                                                        }
                                                    });
        }

        /** @return the current value */
        public Identifier[] getIdentifiers() {
            Identifier[] ret = new Identifier[model.size()];
            model.copyInto(ret);
            return ret;
        }

        /** Set new value.
        */
        public void setIdentifiers(Identifier[] data) {
            model = new DefaultListModel();
            if (data != null) {
                for (int i = 0; i < data.length; i++)
                    model.addElement(data[i]);
            }
            this.getListComponent().setModel(model);
            modelChanged();
        }

        /** Fire the 'identifiers' property change. */
        protected void modelChanged() {
            Identifier[] newValue = getIdentifiers();
            firePropertyChange(PROP_IDENTIFIERS, prevValue, newValue);
            prevValue = newValue;
        }

        /** Ask user for new value.
        * @return new value or <CODE>null</CODE> when 
        *    operation was canceled.
        */
        protected Object insertNewValue() {
            return openInputDialog(null);
        }

        /** Ask user for edit value.
        * @param oldValue The previous value to be edited
        * @return new value or <CODE>null</CODE> when 
        *    operation was canceled.
        */
        protected Object editValue(Object oldValue) {
            return openInputDialog((Identifier) oldValue);
        }

        /** Show dialog and allow user to enter new name.
        * @param defName Default value which is predefined.
        * @param titleKey the key to resource bundle for the title of input dialog
        * @return New valid name or <CODE>null</CODE> if user cancel the operation.
        */
        protected Identifier openInputDialog(Identifier origValue) {
            NotifyDescriptor.InputLine input = new NotifyDescriptor.InputLine(
                                                   getString("LAB_NewName"),
                                                   getString("LAB_NewIdentifier")
                                               );
            if (origValue != null)
                input.setInputText(origValue.getSourceName());

            for (;;) {
                Object ret = DialogDisplayer.getDefault().notify(input);
                if (ret == NotifyDescriptor.OK_OPTION) {
                    String retValue = input.getInputText();
                    if (retValue != null && !"".equals(retValue)) { // NOI18N
                        if (!retValue.startsWith(".") && !retValue.endsWith(".") && // NOI18N
                                (retValue.indexOf("..") == -1)) { // NOI18N
                            boolean ok = true;
                            StringTokenizer tokenizer = new StringTokenizer(retValue, ".", false); // NOI18N
                            while (tokenizer.hasMoreTokens()) {
                                String token = tokenizer.nextToken();
                                if (!Utilities.isJavaIdentifier(token)) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (ok)
                                return Identifier.create(retValue);
                        }
                    }
                    DialogDisplayer.getDefault().notify(new NotifyDescriptor.Message(getString("MSG_NotValidID")));
                }
                else {
                    return null;
                }
            }
        }
    }
         
    private static String getString(String key) {
        return NbBundle.getBundle("org.openide.explorer.propertysheet.editors.Bundle2", Locale.getDefault(), IdentifierArrayEditor.class.getClassLoader()).getString(key);
    }
         
}
